/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.debug.
 *
 * de.bsvrz.sys.funclib.debug is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.debug is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.debug; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.sys.funclib.debug;

import java.io.PrintWriter;
import java.text.DecimalFormat;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.logging.Formatter;
import java.util.logging.Level;
import java.util.logging.LogRecord;

/**
 * Gibt die DebugMeldungen als Text aus.
 *
 * @author Hans Christian Kniß (HCK)
 * @version $Revision$ / $Date$
 */
public class DebugFormatterStdErrText extends Formatter {

    /**
     * Formatstring für das Ausgabeformat des Zeitstempels. Ausgabe erfolgt mit Datum, Uhrzeit, Millisekunden und Zeitoffset zur Zeitangabe in UTC.
     */
    static final DateTimeFormatter _dateFormat = DateTimeFormatter.ofPattern("dd.MM.yyyy HH:mm:ss,SSS:Z");
    /**
     * Formatstring mit Angabe für die Formatierung der Zahlenausgaben für die lfd. Nr. der Meldung und die ThreadId.
     */
    static final ThreadLocal<DecimalFormat> _numberFormat = ThreadLocal.withInitial(() -> new DecimalFormat("000000"));
    /**
     * Systemabhängig zur Laufzeit ermittelter String für den Zeilenumbruch. Dadurch wird die Meldungsausgabe plattformunabhängig.
     */
    static final String NEWLINE = System.lineSeparator();
    /**
     * Linientyp wird bei der Ausgabe vom Level FEHLER verwendet
     */
    static final String HASH_LINE = " ###################### ";
    /**
     * Linientyp wird bei der Ausgabe vom Level WARNUNG verwendet
     */
    static final String DOUBLE_LINE = " ====================== ";
    /**
     * Linientyp wird bei der Ausgabe vom Level INFO verwendet
     */
    static final String SINGLE_LINE = " ---------------------- ";
    /**
     * Linientyp wird bei der Ausgabe vom Level KONFIGURATION verwendet
     */
    static final String DOT_LINE = " ...................... ";
    /**
     * Linientyp wird bei der Ausgabe vom Level FEIN, FEINER und DETAIL verwendet
     */
    static final String SIMPLE_LINE = " .  .  .  .  .  .  .  . ";

    private final boolean _debugStdErrWithStackTraces;

    public DebugFormatterStdErrText() {
        _debugStdErrWithStackTraces = true;
    }

    public DebugFormatterStdErrText(final boolean debugStdErrWithStackTraces) {
        _debugStdErrWithStackTraces = debugStdErrWithStackTraces;
    }

    /**
     * Gibt die Debug-Meldung aus
     *
     * @param lr LogRecord mit den Informationen einer Meldung.
     *
     * @return Gibt den als Text formatierten Meldungstext mit den im LogRecord übergebenen Informationen aus.
     */
    public String format(LogRecord lr) {

        StringBuilder sb = new StringBuilder().append(NEWLINE);
        sb.append("#").append(_numberFormat.get().format(lr.getSequenceNumber()));            // Kopfzeile: LfdNr...
        sb.append(" ");
        // TODO: lr.getInstant() ab Java 9
        ZonedDateTime zonedDateTime = ZonedDateTime.ofInstant(Instant.ofEpochMilli(lr.getMillis()), ZoneId.systemDefault());
        sb.append(_dateFormat.format(zonedDateTime));                                    // ... Datum ...
        // TODO: lr.getLongThreadID() ab Java 16
        sb.append(" (TID:").append(_numberFormat.get().format(lr.getThreadID())).append(")");  // ...Thread ID ...

        if (lr.getLevel() == Debug.ERROR) {
            sb.append(HASH_LINE);                                                        // ... Linie abhängig ...
        }                                                                                // ... vom Level ...
        else if (lr.getLevel() == Debug.WARNING) {
            sb.append(DOUBLE_LINE);
        } else if (lr.getLevel() == Debug.INFO) {
            sb.append(SINGLE_LINE);
        } else if (lr.getLevel() == Debug.CONFIG) {
            sb.append(DOT_LINE);
        } else {
            sb.append(SIMPLE_LINE);
        }
        sb.append(NEWLINE);

        Level l = lr.getLevel();                                                        // ... und Level im Klartext.
        if (l == Debug.ERROR) {
            sb.append("FEHLER : ");
        } else if (l == Debug.WARNING) {
            sb.append("WARNUNG: ");
        } else if (l == Debug.INFO) {
            sb.append("INFO   : ");
        } else if (l == Debug.CONFIG) {
            sb.append("KONFIG : ");
        } else if (l == Debug.FINE) {
            sb.append("FEIN   : ");
        } else if (l == Debug.FINER) {
            sb.append("FEINER : ");
        } else if (l == Debug.FINEST) {
            sb.append("DETAIL : ");
        }
        sb.append(lr.getLoggerName()).append(NEWLINE);                                    // 2. Zeile: Wurzel+Klassenname

        sb.append(lr.getMessage());

        Object[] objects = lr.getParameters();

        if ((objects != null) && (objects.length == 1)) {
            final Object object = objects[0];
            sb.append(": ").append(object == null ? "null" : object.toString());
        }

        if ((objects != null) && (objects.length > 1)) {
            sb.append(':').append(NEWLINE);
            for (int i = 0; i < objects.length; i++) {
                final Object object = objects[i];
                sb.append("[").append(i).append("] ").append(object == null ? "null" : object.toString()).append(NEWLINE);
            }
        }

        if (lr.getThrown() != null) {
            sb.append(":").append(NEWLINE);
            if (_debugStdErrWithStackTraces) {
                try {
                    sb.append("      ");
                    lr.getThrown().printStackTrace(new PrintWriter(new StringBuilderWriter(sb)));
                } catch (Exception ignored) {
                }
            } else {
                sb.append(lr.getThrown());
            }
        }

        sb.append(NEWLINE);
        return sb.toString();
    }
}
