/*
 * Copyright 2004 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview;

import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.pat.sysbed.dataview.filtering.FilterAttributeGroup;
import de.bsvrz.pat.sysbed.dataview.selectionManagement.RowKey;
import de.bsvrz.pat.sysbed.dataview.selectionManagement.SelectionManager;
import de.bsvrz.pat.sysbed.main.TooltipAndContextUtil;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridLayout;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.border.EtchedBorder;

/**
 * Diese Klasse liefert zu einem Datensatz ({@link DataTableObject}) alle für seine Darstellung notwendigen Komponenenten, d.h. Spalten- und
 * Zeilen-Header und auch die Felder in Form der hierarchischen Struktur eines {@link RowData}.
 * <p>
 * Alle abrufbaren Informationen werden erst beim ersten Abruf gebildet.
 *
 * @author Kappich Systemberatung
 */
public class DataTableObjectRenderer {

    /** speichert einen Datensatz vom Datenverteiler */
    private final DataTableObject _dataTableObject;

    /** speichert die Struktur des Spaltenheaders, damit die Nutzdaten damit verknüpft werden können */
    private final HeaderGrid _header;
    /** speichert den SelectionManager */
    private final SelectionManager _selectionManager;
    /** speichert den RowKey */
    private final RowKey _rowKey;
    /** speichert den AtgFilter */
    private final FilterAttributeGroup _filterAttributeGroup;
    /** speichert die hierarchischen Struktur des Datensatzes */
    private RowData _rowData;
    /** speichert die zum Datensatz gehörende Zeilenkopf-Komponente */
    private JComponent _rowHeaderRow;
    /** speichert die zum Datensatz gehörende Datenzeilen-Komponente */
    private JComponent _viewportRow;
    /** speichert die Höhe der Komponente einer Zeile */
    private int _height = -1;
    /** speichert die Breite des Headers */
    private int _headerWidth;

    /**
     * Der Konstruktor.
     *
     * @param header           Element, welches die hierarchische Struktur darstellt
     * @param dataTableObject  neuer Datensatz
     * @param selectionManager Selektions-Manager
     */
    DataTableObjectRenderer(final HeaderGrid header, final DataTableObject dataTableObject, final SelectionManager selectionManager,
                            final FilterAttributeGroup filterAttributeGroup) {
        _header = header;
        _dataTableObject = dataTableObject;
        _selectionManager = selectionManager;
        _rowKey = new RowKey(_dataTableObject.getObject().getPidOrId() + RowKey.getSeparator() + _dataTableObject.getDataIndex());
        _filterAttributeGroup = filterAttributeGroup;
    }

    /**
     * Legt einen Text für jeden DataState fest; wird für die "Keine-Daten-Fälle" benötigt.
     *
     * @param dataState der Zustand des Datensatzes
     *
     * @return der Text
     */
    public static String getTextForState(final DataState dataState) {
        String text;
        if (dataState == DataState.NO_DATA) {
            text = "keine Daten";
        } else if (dataState == DataState.NO_SOURCE) {
            text = "keine Daten (keine Quelle)";
        } else if (dataState == DataState.NO_RIGHTS) {
            text = "keine Daten (keine Rechte)";
        } else if (dataState == DataState.POSSIBLE_GAP) {
            text = "keine Daten (potentielle Datenlücke)";
        } else if (dataState == DataState.END_OF_ARCHIVE) {
            text = "Ende des Archivanfragezeitraums";
        } else if (dataState == DataState.DELETED_BLOCK) {
            text = "Gelöschter Bereich";
        } else if (dataState == DataState.UNAVAILABLE_BLOCK) {
            text = "Ausgelagerter Bereich";
        } else if (dataState == DataState.INVALID_SUBSCRIPTION) {
            text = "keine Daten (fehlerhafte Anmeldung)";
        } else if (dataState == DataState.DATA) {
            text = "Nutzdaten";
        } else {
            text = "keine Daten (Undefinierte Objektkodierung)";
        }
        return text;
    }

    /**
     * Legt die Hintergrundfarbe für die "Keine-Daten-Fälle" in Abhängigkeit von dem Status fest.
     *
     * @param dataState ein DataState
     *
     * @return die Hintergrundfarbe
     */
    @Nullable
    static Color getColorForState(final DataState dataState) {
        if (dataState == DataState.NO_DATA) {
            return Color.green;
        } else if (dataState == DataState.NO_SOURCE) {
            return Color.orange;
        } else if (dataState == DataState.NO_RIGHTS) {
            return Color.red;
        } else if (dataState == DataState.POSSIBLE_GAP) {
            return Color.magenta;
        } else if (dataState == DataState.END_OF_ARCHIVE) {
            return Color.cyan;
        } else if (dataState == DataState.DELETED_BLOCK) {
            return Color.red;
        } else if (dataState == DataState.UNAVAILABLE_BLOCK) {
            return Color.yellow;
        } else if (dataState == DataState.INVALID_SUBSCRIPTION) {
            return Color.red;
        } else if (dataState == DataState.DATA) {
            return null;
        } else {
            return Color.red;
        }
    }

    /**
     * Ermittelt den 2-Zeichentext für die allererste Spalte der Onlinetabelle, die mit 'Art' überschrieben ist.
     *
     * @param dataKind die ArchiveDataKind
     *
     * @return der 2-Zeichentext der Spalte 'Art'
     */
    public static String getDatakindText(final ArchiveDataKind dataKind) {
        if (dataKind == ArchiveDataKind.ONLINE) {
            return "OA";
        } else if (dataKind == ArchiveDataKind.ONLINE_DELAYED) {
            return "ON";
        } else if (dataKind == ArchiveDataKind.REQUESTED) {
            return "NA";
        } else if (dataKind == ArchiveDataKind.REQUESTED_DELAYED) {
            return "NN";
        } else {
            return "??";
        }
    }

    private static String getDatakindTooltipText(final ArchiveDataKind dataKind) {
        if (dataKind == ArchiveDataKind.ONLINE) {
            return "online aktueller Datensatz";
        } else if (dataKind == ArchiveDataKind.ONLINE_DELAYED) {
            return "online nachgelieferter Datensatz";
        } else if (dataKind == ArchiveDataKind.REQUESTED) {
            return "nachgefordert aktueller Datensatz";
        } else if (dataKind == ArchiveDataKind.REQUESTED_DELAYED) {
            return "nachgefordert nachgelieferter Datensatz";
        } else {
            return "Art des Datensatzes nicht bekannt";
        }
    }

    /**
     * Gibt den Datensatz zurück.
     *
     * @return Datensatz
     */
    public DataTableObject getDataTableObject() {
        return _dataTableObject;
    }

    /**
     * Gibt die hierarchische Struktur des Datensatzes zurück.
     *
     * @return hierarchische Struktur des Datensatzes
     */
    RowData getRowData() {
        if (_rowData == null) {
            createRowData();
        }
        return _rowData;
    }

    /**
     * Gibt den RowKey zurück.
     *
     * @return den RowKey
     */
    RowKey getRowKey() {
        return _rowKey;
    }

    /**
     * Gibt die Komponente des Zeilenkopfes zurück.
     *
     * @param timeFormat das gewünschte Format
     *
     * @return Komponente des Zeilenkopfes
     */
    JComponent getRowHeaderRow(String timeFormat) {
        if (_rowHeaderRow == null) {
            _rowHeaderRow = createRowHeaderRow(_dataTableObject, timeFormat);
        }
        return _rowHeaderRow;
    }

    /**
     * Gibt die Komponente der Datenzeile zurück.
     *
     * @return Komponente der Datenzeile
     */
    JComponent getViewportRow() {
        if (_viewportRow == null) {
            createViewportRow();
        }
        return _viewportRow;
    }

    /**
     * Gibt die Höhe dieser Zeile in Pixel zurück.
     *
     * @return Höhe dieser Zeile
     */
    public int getHeight() {
        if (_height == -1) {
            setHeight();
        }
        return _height;
    }

    /* ################# Private Methoden ############# */

    /**
     * Erstellt die Verbindungen zwischen den Daten und dem Header. Jedes Blattelement meldet sich beim entsprechenden Blatt im Header an, damit
     * etwaige Größenänderungen vom Header an die Datenstruktur übergeben werden kann.
     */
    void setLinks() {
        _rowHeaderRow = null; // wenn die Elemente verbunden werden sollen, müssen die Komponenten erst gelöscht
        _viewportRow = null; // werden. Nach dem nächsten Erstellen, benutzen sie auch die Informationen aus dem
        // Header
        if (_rowData == null) {
            createRowData();
        }
        if (_filterAttributeGroup.getAtgFilter() == null) {
            linkData(_rowData, _header);
        } else {
            AtomicInteger nextHeaderGridColumn = new AtomicInteger(0);
            linkDataFlat(_rowData, _header, nextHeaderGridColumn);
        }
        createViewportRow();
    }

    /** Entfernt alle Einträge, außer den Datensatz und die Höhe einer Zeile. */
    void unsetLinks() {
        // löscht alles außer ResultData und Height (Objekte einschließlich Verweise)
        if (_rowData != null) { // nur dann macht es Sinn, sonst gibt es nichts zum Löschen
            unlinkData(_rowData, _header);
        }
        removeComponents();
    }

    /** Erzeugt aus einem Datensatz eine hierarchische Struktur. */
    private void createRowData() {
        _rowData = new RowData(_dataTableObject, _selectionManager);
    }

    /**
     * Rekursive Hilfsmethode. Sie wird von {@link #setLinks()} aufgerufen. Die Verbindungen zum Spaltenheader werden hergestellt und die Breite der
     * Komponenten, welche die Daten anzeigen wird initial festgelegt. Diese Methode wird nur ausgeführt, wenn im Datensatz auch Daten vorhanden
     * sind.
     *
     * @param rowData    darzustellende Daten
     * @param headerGrid Spaltenheader
     */
    @SuppressWarnings({"unchecked", "OverlyNestedMethod"})
    private void linkData(RowData rowData, HeaderGrid headerGrid) {
        if (_dataTableObject != null) { // _resultData.getData() != null
            if (_dataTableObject.getData() == null) {
                // Idee: bis in die Blätter gehen, dort anmelden und die Summe ergibt dann die Breite
                // des leeren Datensatzes
                _headerWidth = 0;
                getHeaderWidth(headerGrid, rowData);
                rowData.setInitialWidth(_headerWidth);
            } else {
                if (rowData.getSuccessors().isEmpty()) { // RowData ist Blatt
                    int width1 = headerGrid.getHeaderElement().getSize().width;
                    int width2 = headerGrid.getSplitter().getSize().width;
                    rowData.setInitialWidth(width1 + width2); // Breite mitteilen
                    headerGrid.addColumnWidthChangeListener(rowData);
                } else { // RowData ist kein Blatt
                    // entweder sind alle vom Typ RowData oder vom Typ RowSuccessor je nachdem, ob Array oder nicht
                    List<Object> array = rowData.getSuccessors();
                    if (!rowData.isArray()) { // kein Array -> alle Nachfolger vom Typ RowData
                        Iterator<HeaderGrid> gridIt = headerGrid.getHeaderSuccessors().iterator();
                        Iterator<Object> rowIt = array.iterator();
                        while (rowIt.hasNext() && gridIt.hasNext()) {
                            RowData nextRowData = (RowData) rowIt.next();
                            HeaderGrid nextHeaderGrid = gridIt.next();
                            linkData(nextRowData, nextHeaderGrid);
                        }
                    } else {
                        for (final Object anArray : array) {
                            RowSuccessor rowSuccessor = (RowSuccessor) anArray;
                            if (headerGrid.getHeaderSuccessors().isEmpty()) {
                                for (final RowData rowData1 : rowSuccessor.getSuccessors()) {
                                    linkData(rowData1, headerGrid);
                                }
                            } else {
                                Iterator<HeaderGrid> gridIt = headerGrid.getHeaderSuccessors().iterator();
                                Iterator<RowData> succIt = rowSuccessor.getSuccessors().iterator();
                                while (succIt.hasNext() && gridIt.hasNext()) {
                                    RowData nextRowData = succIt.next();
                                    HeaderGrid nextHeaderGrid = gridIt.next();
                                    linkData(nextRowData, nextHeaderGrid);
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    /*
     * Teil-rekursive Hilfsmethode. Sie wird von {@link #setLinks()} im Filterfall aufgerufen. Die Verbindungen zum
     * Spaltenheader werden hergestellt und die Breite der Komponenten, welche die Daten anzeigen
     * wird initial festgelegt. Diese Methode wird nur ausgeführt, wenn im Datensatz auch Daten vorhanden sind.
     *
     * @param rowData darzustellende Daten (rekursiv)
     * @param headerGrid Spaltenheader (flach, da Aufruf nur im Filterfall)
     * @param nextHeaderGridColumn der Index der nächsten Spalte des flachen HeaderGrids
     *
     * Die Verwendung eines AtomicIntegers hat einen Grund: auf diese Art gibt es ein einziges Objekt, das auch
     * verändert werden kann. Integer, int und Iteratoren taten es stattdessen nicht.
     * Trotzdem ist das nicht unbedingt schön zu nennen.
     */
    @SuppressWarnings("OverlyNestedMethod")
    private void linkDataFlat(RowData rowData, HeaderGrid headerGrid, AtomicInteger nextHeaderGridColumn) {
        // Im Unterschied zu linkData wird hier nur RowData rekursiv bearbeitet, während das HeaderGrid
        // sequentiell bearbeitet wird.
        if (_dataTableObject != null) { // _resultData.getData() != null
            if (_dataTableObject.getData() == null) {
                // Analog zu linkData; getHeaderWidth scheint auch hier das richtige zu machen.
                _headerWidth = 0;
                getHeaderWidth(headerGrid, rowData);
                rowData.setInitialWidth(_headerWidth);
            } else {
                if (rowData.getSuccessors().isEmpty()) { // RowData ist ein Blatt
                    HeaderGrid successor = headerGrid.getHeaderSuccessors().get(nextHeaderGridColumn.getAndIncrement());
                    int width1 = successor.getHeaderElement().getSize().width;
                    int width2 = successor.getSplitter().getSize().width;
                    rowData.setInitialWidth(width1 + width2); // Breite mitteilen
                    successor.addColumnWidthChangeListener(rowData);
                } else {
                    // entweder sind alle vom Typ RowData oder vom Typ RowSuccessor: je nachdem, ob Array oder nicht
                    List<Object> successorArray = rowData.getSuccessors();
                    if (!rowData.isArray()) { // kein Array -> alle Nachfolger sind vom Typ RowData
                        for (final Object aSuccessor : successorArray) {
                            RowData nextRowData = (RowData) aSuccessor;
                            linkDataFlat(nextRowData, headerGrid, nextHeaderGridColumn);
                        }
                    } else {  // Array -> alle Nachfolger sind vom Typ RowSuccessor
                        int nextColumn = nextHeaderGridColumn.get();
                        for (final Object aSuccessor : successorArray) {
                            RowSuccessor rowSuccessor = (RowSuccessor) aSuccessor;
                            nextHeaderGridColumn.set(nextColumn);
                            for (final RowData nextRowData : rowSuccessor.getSuccessors()) {
                                linkDataFlat(nextRowData, headerGrid, nextHeaderGridColumn);
                            }
                        }
                    }
                }
            }
        }
    }

    private void getHeaderWidth(HeaderGrid headerGrid, RowData rowData) {
        List<HeaderGrid> succs = headerGrid.getHeaderSuccessors();
        if (succs.isEmpty()) { // Blattknoten erreicht
            _headerWidth += headerGrid.getHeaderElement().getSize().width + headerGrid.getSplitter().getSize().width;
            headerGrid.addColumnWidthChangeListener(rowData);
        } else {
            for (HeaderGrid grid : succs) {
                getHeaderWidth(grid, rowData);
            }
        }
    }

    /**
     * Rekursive Hilfsmethode. Sie entfernt die Verbindungen zwischen den Daten und dem Spaltenheader. Diese Methode wird nur ausgeführt, wenn auch
     * Daten vorhanden sind.
     *
     * @param rowData    Daten, die mit dem Spaltenheader verbunden sind
     * @param headerGrid Spaltenheader
     */
    @SuppressWarnings({"unchecked", "OverlyNestedMethod"})
    private void unlinkData(RowData rowData, HeaderGrid headerGrid) {
        //noinspection VariableNotUsedInsideIf
        if (_dataTableObject != null) {
            if (rowData.getSuccessors().isEmpty()) {
                headerGrid.removeColumnWidthChangeListener(rowData);
            } else {
                // entweder sind alle vom Typ RowData oder vom Typ RowSuccessor
                List<Object> array = rowData.getSuccessors();
                Object object = array.get(0);
                if (object instanceof RowData) {
                    Iterator<HeaderGrid> gridIt = headerGrid.getHeaderSuccessors().iterator();
                    Iterator<Object> rowIt = array.iterator();
                    while (rowIt.hasNext() && gridIt.hasNext()) {
                        RowData nextRowData = (RowData) rowIt.next();
                        HeaderGrid nextHeaderGrid = gridIt.next();
                        unlinkData(nextRowData, nextHeaderGrid);
                    }
                }
                if (object instanceof RowSuccessor) {
                    for (final Object anArray : array) {
                        RowSuccessor rowSuccessor = (RowSuccessor) anArray;
                        if (headerGrid.getHeaderSuccessors().isEmpty()) {
                            for (final RowData rowData1 : rowSuccessor.getSuccessors()) {
                                unlinkData(rowData1, headerGrid);
                            }
                        } else {
                            Iterator<HeaderGrid> gridIt = headerGrid.getHeaderSuccessors().iterator();
                            Iterator<RowData> succIt = rowSuccessor.getSuccessors().iterator();
                            while (succIt.hasNext() && gridIt.hasNext()) {
                                RowData nextRowData = succIt.next();
                                unlinkData(nextRowData, gridIt.next());
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * Erzeugt anhand der Daten eine neue Zeile im Zeilenheader, bestehend aus einem Zeitstempel und dem dazugehörenden Objekt.
     *
     * @param dataTableObject das Objekt der Online-Tabelle
     * @param format          das Zeitformat, um Datum und Zeit darzustellen
     *
     * @return der erstellte Zeilen-Header
     */
    private JComponent createRowHeaderRow(final DataTableObject dataTableObject, String format) {
        final ArchiveDataKind dataKind = dataTableObject.getDataKind();
        final String dataKindText = getDatakindText(dataKind);
        final String dataKindTooltipText = getDatakindTooltipText(dataKind);
        JLabel dataKindLabel = new JLabel(dataKindText);
        dataKindLabel.setToolTipText(dataKindTooltipText);
        dataKindLabel.setBorder(new EtchedBorder());
        dataKindLabel.setHorizontalAlignment(SwingConstants.CENTER);
        dataKindLabel.setVerticalAlignment(SwingConstants.TOP);

        JPanel dataKindPanel = new JPanel(new BorderLayout());
        dataKindPanel.add(dataKindLabel, BorderLayout.CENTER);

        JLabel timeLabel = new JLabel(_dataTableObject.getTimeText(format));
        timeLabel.setBorder(new EtchedBorder());
        timeLabel.setHorizontalAlignment(SwingConstants.CENTER);
        timeLabel.setVerticalAlignment(SwingConstants.TOP);

        String toolTipText = "<html>";
        toolTipText += "Datenzeit: " + _dataTableObject.getDataTime(format) + "<br>";
        if (dataTableObject.getArchiveTime() > 0) {
            toolTipText += "Archivzeit: " + _dataTableObject.getArchiveTime(format) + "<br>";
        }
        toolTipText += "Datenindex: " + _dataTableObject.getDataIndexString();
        toolTipText += "</html>";
        timeLabel.setToolTipText(toolTipText);

        final SystemObject object = dataTableObject.getObject();
        final JLabel objectLabel = new JLabel(object.getNameOrPidOrId());
        objectLabel.setBorder(new EtchedBorder());
        objectLabel.setHorizontalAlignment(SwingConstants.CENTER);
        objectLabel.setVerticalAlignment(SwingConstants.TOP);

        objectLabel.setToolTipText(TooltipAndContextUtil.getTooltip(object));

        JPanel gridPanel = new JPanel(new GridLayout(1, 2));
        gridPanel.add(timeLabel);
        gridPanel.add(objectLabel);

        JPanel rowHeaderPanel = new JPanel(new BorderLayout());
        rowHeaderPanel.add(dataKindPanel, BorderLayout.WEST);
        rowHeaderPanel.add(gridPanel, BorderLayout.CENTER);

        if (_selectionManager.isRowSelected(getRowKey())) {
            dataKindPanel.setBackground(dataKindPanel.getBackground().darker());
            gridPanel.setBackground(gridPanel.getBackground().darker());
        }

        final MouseListener mouseListener = new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                int modifiersEx = e.getModifiersEx();
                _selectionManager.mousePressed(_rowKey, modifiersEx);
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                int modifiersEx = e.getModifiersEx();

                if ((modifiersEx & InputEvent.CTRL_DOWN_MASK) > 0) {
                    _selectionManager.mouseReleased(_rowKey, InputEvent.CTRL_DOWN_MASK);
                } else if ((modifiersEx & InputEvent.SHIFT_DOWN_MASK) > 0) {
                    _selectionManager.mouseReleased(_rowKey, InputEvent.SHIFT_DOWN_MASK);
                } else {
                    _selectionManager.mouseReleased(_rowKey, 0);
                }
            }

        };
        /*
         * Da die beiden Label einen Tooltipp haben, muss man den MouseListener extra hinzufügen,
         * da er sonst auf deren Flächen nicht arbeitet. Dieser Workaround funktioniert nicht für
         * mouseEntered und mouseExited.
         */
        rowHeaderPanel.addMouseListener(mouseListener);
        timeLabel.addMouseListener(mouseListener);
        dataKindLabel.addMouseListener(mouseListener);
        objectLabel.addMouseListener(mouseListener);

        return rowHeaderPanel;
    }

    /** Erzeugt aus einem Datensatz eine Swing-Komponente, damit die Daten angezeigt werden können. */
    private void createViewportRow() {
        if (_rowData == null) {
            createRowData();
        }
        _viewportRow = _rowData.createComponent();
        _height = _viewportRow.getPreferredSize().height;
    }

    /**
     * Ermittelt die Höhe der diesen Datensatz repräsentierenden Swing-Komponente. Falls die Komponenten extra für die Ermittlung der Höhe erzeugt
     * werden, dann werden sie anschließend auch wieder gelöscht.
     */
    private void setHeight() {
        if (_viewportRow == null) {
            createViewportRow();
            removeComponents();
        } else {
            _height = _viewportRow.getPreferredSize().height;
        }
    }

    /** Löscht alle nicht mehr benötigten Komponenten, außer des Datensatzes, des Spaltenheaders und der Höhe. */
    private void removeComponents() {
        _rowData = null;
        _rowHeaderRow = null;
        _viewportRow = null;
    }

    @Override
    public String toString() {
        return "DataTableObjectRenderer{}";
    }
}
