/*
 * Copyright 2004 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.pat.sysbed.dataview;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.pat.sysbed.dataview.filtering.AtgFilter;
import de.bsvrz.pat.sysbed.dataview.filtering.AtgFilterNode;
import de.bsvrz.pat.sysbed.dataview.selectionManagement.CellKey;
import de.bsvrz.pat.sysbed.dataview.selectionManagement.SelectionManager;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import javax.swing.JComponent;
import javax.swing.JPanel;

/**
 * Diese Klasse bietet eine hierarchische Struktur für einen Datensatz aus dem Datenverteiler. Es werden Attribute, Listen, Arrays von Attributen und
 * Arrays von Listen berücksichtigt.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("OverlyLongMethod")
public class RowData implements ColumnWidthChangeListener {

    /** Der Debug-Logger */
    private static final Debug _debug = Debug.getLogger();
    /**
     * Speichert alle Nachfolger in einer Liste. Die Nachfolger sind alle entweder vom Typ {@link RowSuccessor} oder {@link RowData}, je nachdem, ob
     * _isArray {@code true} oder {@code false} ist.
     */
    private final List<Object> _successors = new ArrayList<>();
    /** der empfangene Datensatz */
    private final DataTableObject _dataTableObject;
    /** der Selektion-Manager */
    private final SelectionManager _selectionManager;
    /** speichert den anzuzeigenden Text dieses Feldes */
    private String _value = "";
    /** Gibt an, ob sich dieses Objekt um ein Array von Listen/Attributen handelt, oder nicht. */
    private boolean _isArray;
    /** speichert die Komponente, die aus sich selbst und allen Nachfolgern besteht */
    private JComponent _component;
    /** speichert die initiale Breite dieser Komponente */
    private int _initialWidth;
    /** speichert die optimale Spaltenbreite */
    private int _optimalColumnWidth;
    /** Key zur eindeutigen Identifizierung */
    private CellKey _cellKey;

    /**
     * Stellt die Daten eines Ergebnisdatensatzes in der Onlinetabelle dar.
     *
     * @param dataTableObject ein Datensatz der Online-Tabelle
     */
    public RowData(@Nullable final DataTableObject dataTableObject, final SelectionManager selectionManager) {
        _dataTableObject = dataTableObject;
        _selectionManager = selectionManager;
        if (_dataTableObject != null) {
            initHierarchy();
            // Die im Moment implementierte Version für den Fall eines Filters erzeugt erst das ganze
            // RowData, und filtert danach die nicht gebrauchten Teile heraus. Besser wäre es im Filterfall
            // eine eigene Initialisierung zu haben. Eine solche habe ich mit initFlat() versucht, doch
            // läuft sie (noch) nicht.
            // Tests mit einem Filter, der die Hälfte ausblendete, zeigten, dass doFilter etwa 3,5% der
            // Rechenzeit von initHierarchy verbraucht. Eine Optimierung scheint also nicht notendig.
            if (_dataTableObject.getFilterAttributeGroup().getAtgFilter() != null) {
                doFilter(_dataTableObject.getFilterAttributeGroup().getAtgFilter(), this,
                         _dataTableObject.getFilterAttributeGroup().getAtgFilter().getRoot());
            }
        }
    }

    private static void doFilter(AtgFilter atgFilter, RowData rowData, Object node) {
        if (atgFilter.isLeaf(node)) {
            return;
        }
        List<Object> successors = rowData._successors;
        if (successors.isEmpty()) {
            return;
        }
        Object firstElement = successors.get(0);
        if (firstElement instanceof RowSuccessor) {
            for (Object successor : successors) {
                RowSuccessor rowSuccessor = (RowSuccessor) successor;
                List<RowData> rowSuccessorSuccessors = rowSuccessor.getSuccessors();
                Iterator<RowData> it = rowSuccessorSuccessors.iterator();
                int index = 0;
                while (it.hasNext()) {
                    RowData rowSuccessorSuccessor = it.next();
                    Object child = atgFilter.getChild(node, index);
	                if (child instanceof AtgFilterNode childNode) {
                        if (childNode.hasNonSuppressedAncestor(true)) {
                            doFilter(atgFilter, rowSuccessorSuccessor, child);
                        } else {
                            it.remove();
                        }
                    }
                    ++index;
                }
            }
        } else {
            Iterator<Object> it = successors.iterator();
            int index = 0;
            while (it.hasNext()) {
                Object successor = it.next();
                Object child = atgFilter.getChild(node, index);
	            if (child instanceof AtgFilterNode childNode) {
                    if (childNode.hasNonSuppressedAncestor(true)) {
                        if (successor instanceof RowData) {
                            doFilter(atgFilter, (RowData) successor, child);
                        }
                    } else {
                        it.remove();
                    }
                }
                ++index;
            }
        }
    }

    /**
     * Hilfsfunktion zur Konstruktion des Panels. Hierüber werden die Bedingungen für die Anordnung der Elemente gesetzt.
     *
     * @param gridx      Spaltennummer
     * @param gridy      Zeilennummer
     * @param gridwidth  Anzahl der Spalten über die das Element reicht
     * @param gridheight Anzahl der Zeilen über die das Element reicht
     * @param weightx    Verteilung von zur Verfügung stehendem Platz (horizontal)
     * @param weighty    Verteilung von zur Verfügung stehendem Platz (vertikal)
     *
     * @return die Bedingungen für die Anordnung des Elements
     */
    @SuppressWarnings("SameParameterValue")
    private static GridBagConstraints makeGBC(int gridx, int gridy, int gridwidth, int gridheight, double weightx, double weighty) {
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = gridx;
        gbc.gridy = gridy;
        gbc.gridwidth = gridwidth;
        gbc.gridheight = gridheight;
        gbc.weightx = weightx;
        gbc.weighty = weighty;
        return gbc;
    }

    /*
     * Zum Verständnis (könnte bei initFlat() von Nutzen sein):
     * Ein RowData enthält Nachfolger in _successors; dies sind entweder wieder RowData-Objekte,
     * außer es handelt sich um ein Array: dann sind es RowSuccessors, und die haben ebenfalls
     * Nachfolger, aber ausschließlich RowData-Objekte. Turnt man die Hierarchie herunter, so
     * muss man im Fall von RowSuccessors bis zu deren RoaDatas gehen, denn die RowSuccessors
     * sind nur Hilfsobjekte für die Array-Struktur.
     */

    private static RowData getNextRowData(final Data data, final String path, final SelectionManager selectionManager) {
        final RowData nextRowData = new RowData(null, selectionManager);

        String localPath = getNextPath(data, path);
        final CellKey key = new CellKey(localPath, false);
        nextRowData.setCellKey(key);

        createNextLevel(nextRowData, localPath, data, selectionManager);

        return nextRowData;
    }

    private static String getNextPath(final Data data, final String path) {
        String nextPath = path;
        try {
            int parseInt = Integer.parseInt(data.getName());
            nextPath += "[" + parseInt + "]";
        } catch (NumberFormatException ignore) {
            nextPath += CellKey.getSECOND_SEPARATOR() + data.getName();
        }
        return nextPath;
    }

    @SuppressWarnings("unchecked")
    private static void createNextLevel(final RowData rowData, final String path, final Data data, final SelectionManager selectionManager) {
        if (data.isPlain()) {
            String value = data.valueToString();
            rowData.setValue(value);
        } else {
            if (data.isList()) { // kein Array
                for (final Data nextData : data) {
                    final RowData nextRowData = getNextRowData(nextData, path, selectionManager);
                    rowData.addSuccessor(nextRowData);
                }
            }
            if (data.isArray()) { // ein Array
                rowData.setIsArray(true);
                Data.Array dataArray = data.asArray();
                for (int i = 0, n = dataArray.getLength(); i < n; i++) {
                    Data nextData = dataArray.getItem(i);
                    RowSuccessor succ = new RowSuccessor();
                    String elementPath = path + "[" + i + "]";
                    succ.setKey(new CellKey(elementPath, false));
                    if (nextData.isList()) {
                        for (final Data nextNextData : nextData) {
                            RowData nextRowData = getNextRowData(nextNextData, elementPath, selectionManager);
                            succ.addSuccessor(nextRowData);
                        }
                    } else { // keine Liste
                        final RowData nextRowData = getNextRowData(nextData, path, selectionManager);
                        succ.addSuccessor(nextRowData);
                    }
                    rowData.addSuccessor(succ);
                }
            }
        }
    }

    @SuppressWarnings("OverlyNestedMethod")
    private static void createTheLevel(final RowData rowData, String path, final Data data, final SelectionManager selectionManager,
                                       final AtgFilter atgFilter, final AtgFilterNode atgFilterNode) {

        if (atgFilter.isLeaf(atgFilterNode)) {
            if (!atgFilterNode.isSuppressed()) {
                if (data.isArray()) {
                    Data.Array dataArray = data.asArray();
                    for (int i = 0, n = dataArray.getLength(); i < n; i++) {
                        Data nextData = dataArray.getItem(i);
                        RowSuccessor successor = new RowSuccessor();
                        String elementPath = path + "[" + i + "]";
                        successor.setKey(new CellKey(elementPath, false));
                        rowData.addSuccessor(successor);
                        RowData nextRowData = new RowData(null, selectionManager);
                        String value = nextData.valueToString();
                        nextRowData.setValue(value);
                        successor.addSuccessor(nextRowData);
                        final String nextPath = getNextPath(nextData, path);
                        nextRowData.setCellKey(new CellKey(nextPath, false));
                    }
                } else {
                    RowData newRowData = new RowData(null, selectionManager);
                    String value = data.valueToString();
                    newRowData.setValue(value);
                    rowData.addSuccessor(newRowData);
                    newRowData.setCellKey(new CellKey(path, false));
                }
            }
        } else {  // kein Leaf, also List oder Array
            if (data.isList()) {
                int i = 0;
                for (Data nextData : data) {
                    Object child = atgFilter.getChild(atgFilterNode, i);
	                if (child instanceof AtgFilterNode childNode) {
                        final String nextPath = getNextPath(nextData, path);
                        createTheLevel(rowData, nextPath, nextData, selectionManager, atgFilter, childNode);
                    }
                    ++i;
                }
            }
            if (data.isArray()) {
                if (atgFilterNode.hasNonSuppressedAncestor(false)) {
                    RowData newRowData = new RowData(null, selectionManager);
                    newRowData.setCellKey(new CellKey(path, false));
                    rowData.addSuccessor(newRowData);
                    newRowData.setIsArray(true);
                    Data.Array dataArray = data.asArray();
                    for (int i = 0, n = dataArray.getLength(); i < n; i++) {
                        Data nextData = dataArray.getItem(i);
                        RowSuccessor succ = new RowSuccessor();
                        String elementPath = path + "[" + i + "]";
                        succ.setKey(new CellKey(elementPath, false));
                        newRowData.addSuccessor(succ);
                        if (nextData.isList()) {
                            int j = 0;
                            for (final Data nextNextData : nextData) {
                                Object child = atgFilter.getChild(atgFilterNode, j);
	                            if (child instanceof AtgFilterNode childNode) {
                                    final String nextPath = getNextPath(nextData, path);
                                    createTheLevel(succ, nextPath, nextNextData, selectionManager, atgFilter, childNode);
                                }
                                ++j;
                            }
                        }
                    }
                }
            }
        }
    }

    private static void createTheLevel(final RowSuccessor rowSuccessor, String path, final Data data, final SelectionManager selectionManager,
                                       final AtgFilter atgFilter, final AtgFilterNode atgFilterNode) {
        if (atgFilter.isLeaf(atgFilterNode)) {
            if (!atgFilterNode.isSuppressed()) {
                RowData newRowData = new RowData(null, selectionManager);
                String value = data.valueToString();
                newRowData.setValue(value);
                rowSuccessor.addSuccessor(newRowData);
                String nextPath = getNextPath(data, path);
                newRowData.setCellKey(new CellKey(nextPath, false));
            }
        } else {  // kein Leaf, also List oder Array
            if (data.isList()) {
                int i = 0;
                for (Data nextData : data) {
                    Object child = atgFilter.getChild(atgFilterNode, i);
	                if (child instanceof AtgFilterNode childNode) {
                        final String nextPath = getNextPath(nextData, path);
                        RowData newRowData = new RowData(null, selectionManager);
                        String value = data.valueToString();
                        newRowData.setValue(value);
                        rowSuccessor.addSuccessor(newRowData);
                        String elementPath = path + "[" + i + "]";
                        newRowData.setCellKey(new CellKey(elementPath, false));
                        createTheLevel(newRowData, nextPath, nextData, selectionManager, atgFilter, childNode);
                    }
                    ++i;
                }
            }
        }
    }

    /**
     * Gibt den CellKey des Objekts zurück.
     *
     * @return CellKey
     */
    public CellKey getCellKey() {
        return _cellKey;
    }

    /**
     * Setzt den CellKey des Objekts.
     *
     * @param key
     */
    private void setCellKey(CellKey key) {
        _cellKey = key;
    }

    /* Diese Methode war beim Weiteentwickeln manchmal hilfreich. */
    @SuppressWarnings("unused")
    private void dump(int level) {
        System.out.println("Level: " + level);
        System.out.println("Value: " + _value);
        System.out.println("CellKey: " + _cellKey);
        if (!_successors.isEmpty()) {
            Object object = _successors.get(0);
            if (object instanceof RowSuccessor) {
                for (Object successor : _successors) {
                    System.out.println("Successor!");
                    RowSuccessor rowSuccessor = (RowSuccessor) successor;
                    for (Object o : rowSuccessor.getSuccessors()) {
                        ((RowData) o).dump(level + 1);
                    }
                }
            } else if (object instanceof RowData) {
                for (Object o : _successors) {
                    ((RowData) o).dump(level + 1);
                }
            }
        }
    }

    /**
     * Gibt die Komponente zurück, die sich selbst und alle ihre Nachfolger darstellt.
     *
     * @return Komponente, die sich selbst und alle ihre Nachfolger darstellt
     */
    public JComponent getComponent() {
        return _component;
    }

    /**
     * Setzt die initiale Breite der Komponente, die sich selbst und alle Nachfolger darstellt.
     *
     * @param width die neue Breite dieser Komponente
     */
    public void setInitialWidth(int width) {
        _initialWidth = width;
    }

    /**
     * Erzeugt die Komponente, die sich selbst und alle Nachfolger darstellt.
     *
     * @return die Komponente, die sich selbst und alle Nachfolger darstellt
     */
    @SuppressWarnings({"OverlyLongMethod", "OverlyNestedMethod"})
    public JComponent createComponent() {
        if ((_dataTableObject == null) || (_dataTableObject.getData() != null)) {
            if (_successors.isEmpty()) {
                _component = new RowElement(_value);
                _cellKey.setCellText(_value);
            } else {
                if (!_isArray) { // handelt sich um eine Liste - Einträge sind RowData
                    JPanel panel = new JPanel(new BorderLayout());
                    GridBagLayout gbl = new GridBagLayout();
                    JPanel listPanel = new JPanel();
                    listPanel.setLayout(gbl);
                    int column = 0;

                    for (Object object : _successors) {
	                    if (object instanceof RowData rowData) {

                            JComponent row = rowData.createComponent();

                            GridBagConstraints gbc = makeGBC(column++, 0, 1, 1, 100., 100.);
                            gbc.fill = GridBagConstraints.BOTH;
                            gbl.setConstraints(row, gbc);
                            listPanel.add(row);
                        } else {
                            _debug.error("Daten müssen vom Typ RowData sein!");
                        }
                    }
                    panel.add(listPanel, BorderLayout.CENTER);
                    _component = panel;
                } else { // es handelt sich um ein Array (von Lists, von Attributen)
                    JPanel panel = new JPanel(new BorderLayout());
                    GridBagLayout gbl = new GridBagLayout();
                    JPanel arrayPanel = new JPanel();
                    arrayPanel.setLayout(gbl);
                    int row = 0;

                    for (Object object : _successors) {
	                    if (object instanceof RowSuccessor succ) {
                            GridBagLayout gbl2 = new GridBagLayout();
                            JPanel successorPanel = new JPanel();
                            successorPanel.setLayout(gbl2);
                            int column = 0;

                            final List<RowData> succs = succ.getSuccessors();
                            for (RowData rowData : succs) {
                                JComponent rowComp = rowData.createComponent();
                                GridBagConstraints gbc2 = makeGBC(column++, 0, 1, 1, 100., 100.);
                                gbc2.fill = GridBagConstraints.BOTH;
                                gbl2.setConstraints(rowComp, gbc2);
                                successorPanel.add(rowComp);
                            }
                            JPanel succPanel = new JPanel();
                            succPanel.setLayout(new BorderLayout());
                            succPanel.add(successorPanel, BorderLayout.CENTER);

                            GridBagConstraints gbc = makeGBC(0, row++, 1, 1, 100., 100.);
                            gbc.fill = GridBagConstraints.BOTH;
                            gbl.setConstraints(succPanel, gbc);
                            arrayPanel.add(succPanel);
                        } else {
                            _debug.error("Daten müssen vom Typ RowSuccessor sein!");
                        }
                    }
                    panel.add(arrayPanel, BorderLayout.CENTER);
                    _component = panel;
                }
            }
        } else {
            DataState dataState = _dataTableObject.getDataState();
            String textForState = DataTableObjectRenderer.getTextForState(dataState);
            final Color color = DataTableObjectRenderer.getColorForState(dataState);
            _component = new RowPanel(textForState, color);
            _cellKey.setCellText(textForState);
        }

        _component.addMouseListener(new MouseAdapter() {

            @Override
            public void mousePressed(MouseEvent e) {
                int modifiersEx = e.getModifiersEx();
                _selectionManager.mousePressed(_cellKey, modifiersEx);
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                int modifiersEx = e.getModifiersEx();

                if ((modifiersEx & InputEvent.CTRL_DOWN_MASK) > 0) {
                    _selectionManager.mouseReleased(_cellKey, InputEvent.CTRL_DOWN_MASK);
                } else if ((modifiersEx & InputEvent.SHIFT_DOWN_MASK) > 0) {
                    _selectionManager.mouseReleased(_cellKey, InputEvent.SHIFT_DOWN_MASK);
                } else {
                    _selectionManager.mouseReleased(_cellKey, 0);
                }

            }
        });

        _optimalColumnWidth = _component.getPreferredSize().width;

        if (_initialWidth > 0) {
            _component.setPreferredSize(new Dimension(_initialWidth, _component.getPreferredSize().height));
        }

        if (_cellKey != null) {
	        if (_component instanceof RowElement rowElement) {
                rowElement.setSelectionColors(_selectionManager.isCellKeySelected(_cellKey));
	        } else if (_component instanceof RowPanel rowPanel) {
                rowPanel.setSelectionBorder(_selectionManager.isCellKeySelected(_cellKey));
            }
        }
        return _component;
    }

    /**
     * Setzt das Flag, ob es sich hierbei um ein Array von Listen/Attributen handelt, oder nicht.
     *
     * @param isArray {@code true}, falls es ein Array von Listen/Attributen ist, sonst {@code false}
     */
    public void setIsArray(boolean isArray) {
        _isArray = isArray;
    }

    /* ############# implementiert die Methoden des RowListener-Interfaces ############## */

    /**
     * Gibt zurück, ob es sich um ein Array von Listen/Attributen handelt, oder nicht.
     *
     * @return {@code true}, falls es sich um ein Array von Listen/Attributen handelt, sonst {@code false}
     */
    public boolean isArray() {
        return _isArray;
    }

    /**
     * Gibt den anzuzeigenden Text zurück.
     *
     * @return anzuzeigender Text
     */
    public String getValue() {
        return _value;
    }

    /**
     * Setzt den anzuzeigenden Text.
     *
     * @param value darzustellenden Text
     */
    public void setValue(String value) {
        _value = value;
    }

    /**
     * Fügt einen Nachfolger vom Typ {@code RowSuccessor} oder {@code RowData} hinzu.
     *
     * @param object Nachfolger vom Typ {@code RowSuccessor} oder {@code RowData}
     *
     * @see RowSuccessor
     */
    private void addSuccessor(Object object) {
        _successors.add(object);
    }

    /**
     * Gibt alle Nachfolger zurück. Sie können vom Typ RowSuccessor oder vom Typ RowData sein, je nachdem, ob es sich um ein Array von
     * Listen/Attributen handelt, oder nicht.
     *
     * @return alle Nachfolger
     */
    public List<Object> getSuccessors() {
        return Collections.unmodifiableList(_successors);
    }

    /**
     * Gibt die für diese Komponente optimale Spaltenbreite zurück.
     *
     * @return die optimale Spaltenbreite
     */
    @Override
    public int getOptimalColumnWidth() {
        return _optimalColumnWidth;
    }

    /**
     * Setzt die Breite der Komponente, die diese Daten repräsentiert.
     *
     * @param width neue Breite der Komponente
     */
    @Override
    public void setWidth(int width) {
        if (_dataTableObject != null) {
            if (_dataTableObject.getData() == null) {
                _component.setPreferredSize(new Dimension(width, _component.getPreferredSize().height));
                _component.setMinimumSize(new Dimension(width, _component.getPreferredSize().height));
                _component.validate();
            }
        } else {
            Dimension size = new Dimension(width, _component.getSize().height);
            _component.setPreferredSize(size);
            _component.setSize(size);
            _component.setMinimumSize(size);
            _component.setMaximumSize(size);
            _component.validate();
        }
    }

    @Override
    public String toString() {
        return "RowData{" + "_value='" + _value + "'" + ", _isArray=" + _isArray + ", _component=" + _component + ", _initialWidth=" + _initialWidth +
               ", _optimalColumnWidth=" + _optimalColumnWidth + ", _successors=" + _successors + ", _dataTableObject=" + _dataTableObject + "}";
    }

    /**
     * Wandelt den Datensatz vom Datenverteiler in eine hierachische Struktur um.
     */
    private void initHierarchy() {
        final Data data = _dataTableObject.getData();
        final String firstDIVIDER = CellKey.getFIRST_SEPARATOR();
        final String cellKeyString = _dataTableObject.getObject().getPidOrId() + firstDIVIDER + _dataTableObject.getDataIndex() + firstDIVIDER +
                                     _dataTableObject.getFilterAttributeGroup().getPidOrId();

        setCellKey(new CellKey(cellKeyString, true));

        if (data == null) {
            return;
        }
        createNextLevel(this, cellKeyString, data, _selectionManager);
    }

    /* Hier beginnt eine eigene Initialisierung mit einem AtgFilter als Alternative zu initHierarchy.
     * ACHTUNG: diese Initialisierung ist noch nicht korrekt. Zumindest bei Arrays sollte es noch
     * Probleme geben. */
    @SuppressWarnings("unused")
    private void initFlat() {
        if (_dataTableObject == null) {
            return;
        }
        final AtgFilter atgFilter = _dataTableObject.getFilterAttributeGroup().getAtgFilter();
        if (atgFilter == null) {
            return;
        }
        final Data data = _dataTableObject.getData();
        final String firstDIVIDER = CellKey.getFIRST_SEPARATOR();
        final String cellKeyString = _dataTableObject.getObject().getPidOrId() + firstDIVIDER + _dataTableObject.getDataIndex() + firstDIVIDER +
                                     _dataTableObject.getFilterAttributeGroup().getPidOrId();
        setCellKey(new CellKey(cellKeyString, true));

        if (data == null) {
            return;
        }
        Object root = atgFilter.getRoot();
	    if (root != null && root instanceof AtgFilterNode atgFilterNode) {
            createTheLevel(this, cellKeyString, data, _selectionManager, atgFilter, atgFilterNode);
        }
    }
}
