/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.csv;

import de.bsvrz.pat.sysbed.dataview.filtering.FilterAttributeGroup;
import de.bsvrz.pat.sysbed.dataview.selectionManagement.CellKey;
import de.bsvrz.pat.sysbed.dataview.selectionManagement.CellKeyColumn;
import java.util.Objects;

/**
 * Eine CsvColumn ist entweder ein ansonsten leeres Objekt für eine Spalte, die noch Subspalten hat, oder sie kapselt die Informationen, die eine
 * Spalte in einer CSV-Datei benötigt, also Name und die Informationen des CellKeys, aus dem sie konstruiert wurde, die benötigt werden, um die
 * Inhalte von Datensätzen ermitteln zu können. Von einer CellKeyColumn unterscheidet sie sich dadurch, dass ein Array dort hier durch mehrere Spalten
 * repräsentiert wird.
 *
 * @author Kappich Systemberatung
 */
public class CsvColumn implements Comparable<CsvColumn> {

    /** Falls dies eine Spalte mit Subspalten ist, true; false sonst */
    final boolean _superColumn;

    /** Falls es eine Spalte ohne Subspalten ist, so der Index in der Spaltenliste */
    final Integer _cellKeyColumnIndex;

    /** Falls es eine Spalte ohne Subspalten ist, so AttributeNamePartArrayValues */
    final Integer[] _arrayIndexes;

    /** Falls es eine Spalte ohne Subspalten ist, so AttributePartsWithArrays */
    final String _name;

    /**
     * Eine CsvColumn wird aus einem CellKey konstruiert.
     *
     * @param cellKey der CellKey
     */
    public CsvColumn(FilterAttributeGroup filterAttributeGroup, final CellKey cellKey) {
        final CellKeyColumn cellKeyColumn = new CellKeyColumn(cellKey);
        _superColumn = cellKey.isSuperColumn();
        if (_superColumn) {
            _cellKeyColumnIndex = null;
            _arrayIndexes = null;
            _name = "";
        } else {
            _cellKeyColumnIndex = cellKeyColumn.getIndexInColumnList(filterAttributeGroup, true);
            _arrayIndexes = cellKey.getAttributeNamePartArrayValues();
            _name = cellKey.getAttributePartsWithArrays();
        }
    }

    @Override
    public int compareTo(CsvColumn o) {
        if (_superColumn && o._superColumn) {
            return 0;
        } else if (_superColumn) {
            return -1;
        } else if (o._superColumn) {
            return 1;
        } else {  // _superColumn und o._superColumn sind false
            if (!Objects.equals(_cellKeyColumnIndex, o._cellKeyColumnIndex)) {
                return Integer.compare(_cellKeyColumnIndex, o._cellKeyColumnIndex);
            }
            if ((_arrayIndexes != null) && (o._arrayIndexes != null)) {
                final int length = _arrayIndexes.length;
                if (length != o._arrayIndexes.length) {
                    return Integer.compare(length, o._arrayIndexes.length); // war RuntimeException
                }
                for (int index = 0; index < length; index++) {
                    if (!Objects.equals(_arrayIndexes[index], o._arrayIndexes[index])) {
                        return Integer.compare(_arrayIndexes[index], o._arrayIndexes[index]);
                    }
                }
                return 0;
            } // Die letzten Fälle dienen zur Sicherheit.
            else //noinspection VariableNotUsedInsideIf
                if (_arrayIndexes != null) {
                    return -1;
                } else if (o._arrayIndexes != null) {
                    return 1;
                } else {
                    return 0;
                }
        }
    }

    /**
     * Gibt den Namen zurück.
     *
     * @return den Namen
     */
    public String getName() {
        return _name;
    }

    @Override
    public boolean equals(Object o) {
	    if (!(o instanceof CsvColumn otherColumn)) {
            return false;
        }
        return compareTo(otherColumn) == 0;
    }

    @Override
    public int hashCode() {
        if (!_superColumn) {
            int h = _cellKeyColumnIndex * 10;
            for (final Integer arrayIndex : _arrayIndexes) {
                h += arrayIndex;
            }
            return h;
        } else {
            return 0;
        }
    }
}
