/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.csv;

import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.pat.sysbed.plugins.api.DataIdentificationChoice;
import java.awt.BorderLayout;
import java.awt.Component;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.SpringLayout;
import javax.swing.WindowConstants;

/**
 * @author Kappich Systemberatung
 */
public class CsvProgressDialogOnline extends JFrame implements CsvProgressDialog {

    private static final String TIME_FORMAT = "dd.MM.yyyy HH:mm:ss.SSS";    // ss.SSS: Punkt statt Komma

    private final String _fileName;
    private final DataIdentificationChoice _dataIdentificationChoice;
    private long _startTimestamp;
    private long _lastTimestamp;
    private JPanel _panel;
    private JLabel _startTimestampLabel;
    private JLabel _lastTimestampLabel;
    private JLabel _numberofDataTableObjectsLabel;
    private JPanel _centerPanel;
    private JLabel _dummyLabel;
    private JButton _stopButton;
    private Set<StopRequestListener> _stopListeners;
    private Set<InterruptRequestListener> _interruptListeners;
    private JProgressBar _progressBar;
    private JButton _interruptButton;

    public CsvProgressDialogOnline(String fileName, final AttributeGroup attributeGroup, final Aspect aspect, int simulationVariant,
                                   final List<SystemObject> objectList) {
        _fileName = fileName;
        _dataIdentificationChoice = new DataIdentificationChoice(attributeGroup, aspect, simulationVariant, objectList);
        _startTimestamp = Long.MAX_VALUE;
        _lastTimestamp = 0;
        initComponents();
        initLayout();
        initLogic();
        setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
    }

    public void newTimeStamp(long timestamp) {
        if (timestamp < _startTimestamp) {
            _startTimestamp = timestamp;
            DateFormat timeFormat = new SimpleDateFormat(TIME_FORMAT);
            _startTimestampLabel.setText(timeFormat.format(new Date(timestamp)));
        }
        if (timestamp > _lastTimestamp) {
            _lastTimestamp = timestamp;
            DateFormat timeFormat = new SimpleDateFormat(TIME_FORMAT);
            _lastTimestampLabel.setText(timeFormat.format(new Date(timestamp)));
        }
    }

    public void setNumberOfDataTableObjects(int numberOfDataTableObjects) {
        _numberofDataTableObjectsLabel.setText(Integer.toString(numberOfDataTableObjects));
    }

    private void initComponents() {
        _centerPanel = new JPanel();
        _startTimestampLabel = new JLabel("00.00.0000 00:00:00,000");
        _lastTimestampLabel = new JLabel("00.00.0000 00:00:00,000");
        _numberofDataTableObjectsLabel = new JLabel("0");
        _dummyLabel = new JLabel("");
        _stopButton = new JButton("Beenden");
        _interruptButton = new JButton("Abbrechen");
        _panel = new JPanel();
        _stopListeners = new HashSet<>();
        _interruptListeners = new HashSet<>();
        _progressBar = new JProgressBar(0, 100);
    }

    private void initLayout() {
        setTitle("Onlinedaten in " + _fileName + " speichern");
        _centerPanel.setLayout(new SpringLayout());

        JLabel label1 = new JLabel("Start-Zeitstempel: ");
        JLabel label2 = new JLabel("Letzter Zeitstempel: ");
        JLabel label3 = new JLabel("Anzahl Datensätze: ");

        _centerPanel.setBorder(BorderFactory.createEmptyBorder(20, 10, 0, 0));
        _centerPanel.add(label1);
        _centerPanel.add(_startTimestampLabel);
        _centerPanel.add(label2);
        _centerPanel.add(_lastTimestampLabel);
        _centerPanel.add(label3);
        _centerPanel.add(_numberofDataTableObjectsLabel);
        _centerPanel.add(_dummyLabel);
        _centerPanel.add(_stopButton);
        CsvUtils.makeCompactGrid(_centerPanel, 2, 10, 20);

        _panel.setLayout(new BorderLayout());
        _panel.add(_dataIdentificationChoice, BorderLayout.NORTH);
        _panel.add(_centerPanel, BorderLayout.CENTER);
        add(_panel);
        pack();
    }

    private void initLogic() {
        _stopButton.addActionListener(e -> {
            for (StopRequestListener listener : _stopListeners) {
                listener.stopRequested();
            }
        });
        _interruptButton.addActionListener(e -> {
            for (InterruptRequestListener listener : _interruptListeners) {
                listener.interruptRequested();
            }
        });
    }

    public void addPostProcessingLayout() {
        _stopButton.setEnabled(false);
        _centerPanel.remove(_dummyLabel);
        _centerPanel.remove(_stopButton);
        CsvUtils.makeCompactGrid(_centerPanel, 2, 10, 20);

        JLabel postProcessingLabel = new JLabel("Nachbearbeitung: ");

        _progressBar.setStringPainted(true);

        JPanel progressPanel = new JPanel();
        progressPanel.setLayout(new BorderLayout());

        JPanel centerProgressPanel = new JPanel();
        centerProgressPanel.setLayout(new BoxLayout(centerProgressPanel, BoxLayout.X_AXIS));
        centerProgressPanel.setBorder(BorderFactory.createEmptyBorder(0, 10, 30, 10));
        centerProgressPanel.add(postProcessingLabel);
        centerProgressPanel.add(Box.createHorizontalStrut(10));
        centerProgressPanel.add(_progressBar);
        progressPanel.add(centerProgressPanel, BorderLayout.CENTER);

        JPanel southProgressPanel = new JPanel();
        southProgressPanel.setBorder(BorderFactory.createEmptyBorder(0, 0, 20, 0));
        southProgressPanel.add(_interruptButton);
        progressPanel.add(southProgressPanel, BorderLayout.SOUTH);

        _panel.add(progressPanel, BorderLayout.SOUTH);
        pack();
        revalidate();
        repaint();
    }

    @Override
    public Component getComponent() {
        return this;
    }

    @Override
    public void disposeComponent() {
        dispose();
    }

    public void setPostProcessingProgress(int value) {
        _progressBar.setValue(Math.max(0, Math.min(value, 100)));
    }

    public void addStopRequestListener(final StopRequestListener listener) {
        _stopListeners.remove(listener);
        _stopListeners.add(listener);
    }

    @SuppressWarnings("unused")
    public boolean removeStopRequestListener(final StopRequestListener listener) {
        return _stopListeners.remove(listener);
    }

    @Override
    public void addInterruptRequestListener(final InterruptRequestListener listener) {
        _interruptListeners.remove(listener);
        _interruptListeners.add(listener);
    }

    @Override
    @SuppressWarnings("unused")
    public boolean removeInterruptRequestListener(final InterruptRequestListener listener) {
        return _interruptListeners.remove(listener);
    }

    @Override
    public String toString() {
        return "CsvProgressDialogOnline{" + "_fileName='" + _fileName + '\'' + ", _startTimestamp=" + _startTimestamp + ", _lastTimestamp=" +
               _lastTimestamp + '}';
    }

    public interface StopRequestListener {
        void stopRequested();
    }
}
