/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.filtering;

import de.bsvrz.dav.daf.main.config.Attribute;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeListDefinition;
import de.bsvrz.dav.daf.main.config.AttributeType;
import java.util.List;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeNode;

/**
 * Diese Klasse bildet die {@link TreeNode TreeNodes} der {@link AtgFilter}-Objekte. Jeder dieser Knoten steht für eine {@link AttributeGroup
 * Attributgruppe} oder ein {@link Attribute Attribut}. Ein AtgFilter-Objekt verwendet Knoten mit Attributgruppe als Wurzel, und die übrigen, um die
 * Attribut-Hierarchie dieser Attributgruppe abzubilden.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("NonSerializableFieldInSerializableClass")
public class AtgFilterNode extends DefaultMutableTreeNode {

    final String _objectName;   // Attributgruppen- oder Attributname
    boolean _suppressed;        // der eigentliche Filter
    String _pseudonym;          // der Name in der flachen Tabellenausgabe

    /**
     * Ein AtgFilterNode wird mit einem beliebigen Objekt initialisiert. Dieses Objekt ist das {@code UserObject} des {@link DefaultMutableTreeNode
     * DefaultMutableTreeNodes}.
     * <p>
     * In {@link AtgFilter} werden nur Objekte der Klassen {@link AttributeGroup} und {@link Attribute} verwendet.
     *
     * @param object
     */
    public AtgFilterNode(final Object object) {
        super(object);
	    if (object instanceof AttributeGroup atg) {
            _objectName = atg.getName();
	    } else if (object instanceof Attribute attribute) {
            _objectName = attribute.getName();
        } else {
            _objectName = object.toString();
        }
        _suppressed = true;       // am Anfang wird alles ignoriert
        _pseudonym = "";
    }

    /**
     * Kopiere die Eigenschaften des ersten Objekts in das zweite.
     *
     * @param source
     * @param target
     */
    public static void copyProperties(final AtgFilterNode source, final AtgFilterNode target) {
        target._suppressed = source._suppressed;
        target._pseudonym = source._pseudonym;
    }

    /**
     * Diese Methode erzeugt eine Kopie ohne Children.
     *
     * @return
     */
    public AtgFilterNode getCopy() {
        AtgFilterNode newNode = new AtgFilterNode(getUserObject());
        copyProperties(this, newNode);
        return newNode;
    }

    /**
     * Erzeugt rekursiv die Children-Hierarchie einer Attributgruppe bzw. eines Attributs
     */
    public void createChildren() {
        // Analog zu DataViewPanel.setHeaderSuccessors, damit die gleiche Hierarchie entsteht.
        Object object = getUserObject();
	    if (object instanceof AttributeGroup atg) {
            List<Attribute> attributes = atg.getAttributes();
            for (Attribute attribute : attributes) {
                AtgFilterNode child = new AtgFilterNode(attribute);
                add(child);
                child.createChildren();
            }
	    } else if (object instanceof Attribute attribute) {
            AttributeType attributeType = attribute.getAttributeType();
		    if (attributeType instanceof AttributeListDefinition attributeListDefinition) {
                final List<Attribute> attributes = attributeListDefinition.getAttributes();
                for (Attribute att : attributes) {
                    AtgFilterNode child = new AtgFilterNode(att);
                    add(child);
                    child.createChildren();
                }
            }
        }
    }

    /**
     * Gibt den Namen des Objekts zurück.
     *
     * @return ein Name
     */
    String getObjectName() {
        return _objectName;
    }

    /**
     * Gibt an, ob das entsprechende {@link Attribute} von dem Filter unterdrückt, also herausgefiltert wird oder nicht. Die Filter vom Typ {@link
     * AtgFilter} unterdrücken ausschließlich Blätter.
     *
     * @return s.o.
     */
    public boolean isSuppressed() {
        return _suppressed;
    }

    /**
     * Legt fest, ob das Attribut unterdrückt werden soll.
     *
     * @param suppressed
     */
    public void setSuppressed(final boolean suppressed) {
        _suppressed = suppressed;
    }

    /**
     * Gibt eine benutzerdefiniertes Pseudonym der Attributs zurück. Findet auch nur für Blaätter Anwendung (s. {@link AtgFilterDefDialog}.
     *
     * @return
     */
    public String getPseudonym() {
        if (_pseudonym == null) {
            return "";
        } else {
            return _pseudonym;
        }
    }

    /**
     * Setzt das Pseudonym.
     *
     * @param pseudonym
     */
    public void setPseudonym(final String pseudonym) {
        _pseudonym = pseudonym;
    }

    /**
     * Eine spezielle Methode zur Darstellung des Knoten im {@link AtgFilterDefDialog AtgFilterDefDialogs}.
     *
     * @return
     */
    public String getNodeText() {
        if (_pseudonym == null || _pseudonym.isEmpty()) {
            return getObjectName();
        } else {
            return getObjectName() + " (als " + _pseudonym + ")";
        }
    }

    /**
     * Ein {@code AtgFilterNode} ist im Sinne dieser Methode 'leer', falls unter diesem keine Daten erwartet werden können. D.h. wenn es sich um Blatt
     * handelt und es unterdrückt wird oder, wenn es sich um einen inneren Knoten handelt, und unterhalb dieses Knoten alle Knoten 'leer' sind.
     *
     * @return ob der Knoten 'leer' ist
     */
    public boolean isEmpty() {
        if (super.isLeaf()) {
            return _suppressed;
        } else {
            for (int i = 0; i < getChildCount(); ++i) {
                TreeNode node = getChildAt(i);
	            if (node instanceof AtgFilterNode atgFilterNode) {
                    if (!atgFilterNode.isEmpty()) {
                        return false;
                    }
                }
            }
            return true;
        }
    }

    /**
     * Bestimmt, ob der Knoten nicht-unterdrückte Nachfahren hat bzw. im Falls von {@code includeYourself} gleich {@code true}, ob auch der Knoten
     * selbst nicht unterdrückt ist.
     *
     * @param includeYourself
     *
     * @return s.o.
     */
    public boolean hasNonSuppressedAncestor(boolean includeYourself) {
        if (includeYourself) {
            if (!_suppressed) {
                return true;
            }
        }
        for (int i = 0; i < getChildCount(); ++i) {
            TreeNode node = getChildAt(i);
	        if (node instanceof AtgFilterNode atgFilterNode) {
                if (atgFilterNode.hasNonSuppressedAncestor(true)) {
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    public String toString() {
        String s = "";
        Object object = getUserObject();
        if (object instanceof AttributeGroup) {
            s += "AttributGruppe: " + _objectName;
        } else if (object instanceof Attribute) {
            s += "Attribut: " + _objectName;
        }
        if (_suppressed) {
            s += ", suppressed";
        } else {
            s += ", nicht suppressed";
        }
        if (getPseudonym().isEmpty()) {
            s += " ohne Pseudonym";
        } else {
            s += " mit Pseudonym " + _pseudonym;
        }
        return s;
    }
}
