/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.filtering;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Rectangle;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.EventObject;
import java.util.List;
import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.JTree;
import javax.swing.event.CellEditorListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.DefaultTreeSelectionModel;
import javax.swing.tree.TreeCellEditor;
import javax.swing.tree.TreeCellRenderer;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

/**
 * {@code CheckBoxTree} dekoriert einen JTree mit je einer {@link JCheckBox} pro Knoten bzw. Blatt. Dabei übernimmt {@code CheckBoxTree} den Renderer
 * der Original-JTree-Objekts und gegebenenfalls auch einen Editor. Das Interface {@code CheckBoxListener} dient dazu, bei Zustandsänderung einer
 * JCheckBox registrierte Listener davon zu unterrichten.
 * <p>
 * {@code CheckBoxTree} basiert der Idee und zum Teil auch dem Code nach auf einem Artikel von Santhosh Kumar Tekuri, vgl.
 * http://www.jroller.com/page/santhosh/20050610. Dieser Code wurde später von seinem Autor unter LGPL 2.1 oder höher (nach Wahl des Benutzers)
 * lizensiert. {@code CheckBoxTree} heißt dort {@code CheckTreeManager}. Dort wurde nur der Renderer 'gewrapped'. Statt dem dort verwendeten,
 * komplexen Selektions-Model für die Zustände der Check-Boxen, welches mit einer 3-Zustände-Check-Box interagiert,  kommt hier ein {@code
 * DefaultTreeSelectionModel} zum Einsatz.
 * <p>
 * Anwendung: eine typische Anwendung besteht aus ein bis vier Zeilen Code:
 * <pre>
 *              CheckTreeManager manager = new CheckTreeManager(tree, aCheckBoxInitializer);
 *              manager.setLeafsOnly(true);
 *              manager.suppressIcons();
 *              manager.addCheckBoxListener(aListener);
 *  </pre>
 * Da der {@code CheckBoxTree} keine {@code Component} ist, wird nicht er, sondern der dekorierte {@code JTree} in {@code Container} gesteckt.
 * <p>
 * Einschränkungen: (1) Im Konstruktor von {@code CheckBoxTree} wird der {@code TreeCellRenderer} und gegebenfalls der {@code TreeCellEditor} des
 * übergebenen {@code JTrees} durch einen speziellen Wrapper-Renderer und eine speziellen Wrapper-Editor ersetzt, die gewisse Aufgaben selber
 * übernehmen, aber andere an ihre Vorgänger deliegieren. Dementsprechend sind die Methoden {@code JTree.setCellRenderer} und {@code
 * JTree.setCellEditor} nach dem Konstruktor-Aufruf tabu. (2) Der Original-Renderer des übergebenen {@code JTrees} sollte - um vollen Funktionsunfang
 * zu haben - ein DefaultTreeCellRenderer sein oder einer Subklasse davon angehören (s. ignoreIcons). (3) Eine ähnliche Einschränkung gilt für das
 * {@code TableModel} des {@code JTrees}: ist dies ein {@code DefaultTableModel} oder davon abgeleitet, so wird der Knoten aktualisiert, wenn seine
 * Check-Box betätigt wird (s. {@code DefaultTableModel.reload(TreeNode)}). Dadurch wird eine eventuell vorhandene Veränderung der Darstellung
 * automatisch durchgeführt.</p>
 *
 * <p>Besonderheit: {@code CheckBoxTree} bietet dem Benutzer die Möglichkeit, jederzeit die Checkboxen
 * erneut zu initialisieren. Hierzu wird {@code reinitializeCheckBoxes} aufgerufen. Diese Methode benutzt den im Konstruktor angegebenen
 * Initialisierer. Damit steht dem Benutzer auch programmatisch die Möglichlkeit zu, die CheckBoxen nach der Konstruktion zu steuern.</p>
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("WeakerAccess")
public class CheckBoxTree extends MouseAdapter implements TreeSelectionListener {

    private static final int CHECKBOX_WIDTH = new JCheckBox().getPreferredSize().width;
    private final JTree _tree;
    private final TreeSelectionModel _checkBoxSelectionModel;
    private final TreeCellRendererWrapper _rendererWrapper;
    private final TreeCellEditorWrapper _editorWrapper;
    private final List<CheckBoxListener> _listeners = new ArrayList<>();
    private final CheckBoxInitializer _initializer;
    private boolean _leafsOnly;
    private boolean _ignoreEvents;

    /**
     * Der Konstruktor. Er benötigt einen {@code JTree} und ihm kann auch ein {@code CheckBoxInitializer} mitgegeben werden. Geschieht dies nicht, so
     * werden alle Check-Boxen unselektiert initialisiert.
     *
     * @param tree        ein JTree
     * @param initializer ein CheckBoxInitializer oder {@code null}
     */
    public CheckBoxTree(JTree tree, @Nullable CheckBoxInitializer initializer) {
        this._tree = tree;
        this._initializer = initializer;
        _checkBoxSelectionModel = new DefaultTreeSelectionModel();
        if (initializer != null) {
            initCheckBoxSelectionModel(tree, initializer);
        }
        _rendererWrapper = new TreeCellRendererWrapper(tree.getCellRenderer(), _checkBoxSelectionModel);
        tree.setCellRenderer(_rendererWrapper);
        _editorWrapper = new TreeCellEditorWrapper(tree.getCellEditor(), _checkBoxSelectionModel);
        tree.setCellEditor(_editorWrapper);
        addThisAsListener();
    }

    private static List<TreePath> getAllPaths(JTree tree) {
        List<TreePath> resultList = new ArrayList<>();
        if (tree.getModel().getRoot() != null) {
            addPaths(tree.getModel(), new TreePath(tree.getModel().getRoot()), resultList);
        }
        return resultList;
    }

    private static void addPaths(TreeModel model, TreePath path, List<TreePath> resultList) {
        resultList.add(path);
        Object object = path.getLastPathComponent();
        for (int i = 0; i < model.getChildCount(object); ++i) {
            addPaths(model, path.pathByAddingChild(model.getChild(object, i)), resultList);
        }
    }

    private void addThisAsListener() {
        _tree.addMouseListener(this);
        _tree.addMouseMotionListener(this);
        _checkBoxSelectionModel.addTreeSelectionListener(this);
    }

    /**
     * Diese Methode dient dazu, die CheckBoxen zu jeden beliebeigen Zeitpunkt neu initialisieren zu können.
     */
    public void reinitializeCheckBoxes() {
        if (_initializer != null) {
            initCheckBoxSelectionModel(_tree, _initializer);
        }
    }

    /**
     * Im Defaultzustand wird jeder Knoten mit einer {@code JCheckBox} dekoriert. Mit dieser Methode kann man dies auf alle Blätter einschränken und
     * auch wieder rückgängig machen.
     *
     * @param leafsOnly ein Boolean
     */
    public void setLeafsOnly(@SuppressWarnings("SameParameterValue") boolean leafsOnly) {
        this._leafsOnly = leafsOnly;
        _rendererWrapper.setLeafsOnly(leafsOnly);
        _editorWrapper.setLeafsOnly(leafsOnly);
    }

    /**
     * Nach dem Ausführen dieser Methode werden die üblichen Icons (für geöffnete und geschlossene Knoten, sowie Blätter) unterdrückt. Diese
     * Funktionalität steht nur für dann zur Verfügung, wenn der Original-Renderer ein {@code DefaultTreeCellRenderer} ist.
     */
    public void suppressIcons() {
        _rendererWrapper.suppressIcons();
    }

    /**
     * Registriert einen {@code CheckBoxListener}.
     *
     * @param l ein Listener
     */
    @SuppressWarnings("unused")
    public void addCheckBoxListener(CheckBoxListener l) {
        _listeners.add(l);
    }

    /**
     * De-registriert einen {@code CheckBoxListener} und gibt im  Erfolgsfall {@code true} zurück.
     *
     * @param l ein Listener
     *
     * @return der Erfolgswert
     */
    @SuppressWarnings("unused")
    public boolean removeCheckBoxListener(CheckBoxListener l) {
        return _listeners.remove(l);
    }

    private void notifyCheckBoxListeners(TreeNode node) {
        for (CheckBoxListener listener : _listeners) {
            listener.checkBoxStateSwitched(node);
        }
    }

    /**
     * Während das Selektionsverhalten dem {@code JTree} überlassen bleibt, wird das Betätigen der Check-Boxen hier im MouseListener abgearbeitet.
     *
     * @param e ein Maus-Event
     */
    @Override
    public void mousePressed(MouseEvent e) {
        updateCheckBoxSelection(e);
    }

    private void updateCheckBoxSelection(MouseEvent e) {
        TreePath path = _tree.getPathForLocation(e.getX(), e.getY());
        if (path == null) {
            return;
        }
        Rectangle bounds = _tree.getPathBounds(path);
        if (null != bounds) {
            if (e.getX() > bounds.x + CHECKBOX_WIDTH) {
                return;
            }
        }
        if (_leafsOnly) {
            if (!_tree.getModel().isLeaf(path.getLastPathComponent())) {
                return;
            }
        }

        boolean selected = _checkBoxSelectionModel.isPathSelected(path);
        try {
            _ignoreEvents = true;
            if (selected) {
                _checkBoxSelectionModel.removeSelectionPath(path);
            } else {
                _checkBoxSelectionModel.addSelectionPath(path);
            }
        } finally {
            _ignoreEvents = false;
            _tree.treeDidChange();
        }
        Object object = path.getLastPathComponent();
	    if (object instanceof TreeNode node) {
            notifyCheckBoxListeners(node);
            TreeModel model = _tree.getModel();
            if (model instanceof DefaultTreeModel) {
                ((DefaultTreeModel) model).reload(node);
            }
        }
    }

    @Override
    public void valueChanged(TreeSelectionEvent e) {
        if (!_ignoreEvents) {
            _tree.treeDidChange();
        }
    }

    @Override
    public String toString() {
        return "CheckBoxTree{" + "_leafsOnly=" + _leafsOnly + '}';
    }

    private void initCheckBoxSelectionModel(JTree tree, CheckBoxInitializer updater) {
        _checkBoxSelectionModel.clearSelection();
        List<TreePath> allPaths = getAllPaths(tree);
        for (TreePath path : allPaths) {
            if (updater.isCheckBoxEnabled(tree, path.getLastPathComponent())) {
                _checkBoxSelectionModel.addSelectionPath(path);
            }
        }
    }

    /**
     * Registrierte {@code CheckBoxListner} werden über Zustandsänderungen der {@code JCheckBoxes} informiert.
     */
    interface CheckBoxListener {
        void checkBoxStateSwitched(TreeNode node);
    }

    public interface CheckBoxInitializer {
        boolean isCheckBoxEnabled(JTree tree, Object node);
    }

    /*
    Der Check-Box-Wrapper für einen Renderer.
     */
    private static class TreeCellRendererWrapper extends JPanel implements TreeCellRenderer {

        private final TreeSelectionModel _selectionModel;
        private final TreeCellRenderer _delegate;
        private final JCheckBox _checkBox = new JCheckBox();
        private boolean _leafsOnly;

        /**
         * Der Konstruktor.
         *
         * @param delegate       ein TreeCellRenderer
         * @param selectionModel ein TreeSelectionModel für die Check-Boxen
         */
        public TreeCellRendererWrapper(TreeCellRenderer delegate, TreeSelectionModel selectionModel) {
            _delegate = delegate;
            _selectionModel = selectionModel;
            setLayout(new BorderLayout());
            setOpaque(false);
            _checkBox.setOpaque(false);
        }

        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean selected, boolean expanded, boolean leaf, int row,
                                                      boolean hasFocus) {
            removeAll();
            Component renderer = _delegate.getTreeCellRendererComponent(tree, value, selected, expanded, leaf, row, hasFocus);
            add(renderer, BorderLayout.CENTER);
            if (!_leafsOnly || leaf) {
                TreePath path = tree.getPathForRow(row);
                if (path != null) {
                    _checkBox.setSelected(_selectionModel.isPathSelected(path));
                }
                add(_checkBox, BorderLayout.WEST);
            }
            return this;
        }

        void setLeafsOnly(boolean leafsOnly) {
            this._leafsOnly = leafsOnly;
        }

        void suppressIcons() {
	        if (_delegate instanceof DefaultTreeCellRenderer renderer) {
                renderer.setLeafIcon(null);
                renderer.setClosedIcon(null);
                renderer.setOpenIcon(null);
            }
        }
    }

    /*
    Der Check-Box-Wrapper für einen Editor.
     */
    private static class TreeCellEditorWrapper extends JPanel implements TreeCellEditor {
        private final TreeCellEditor _delegate;
        private final TreeSelectionModel _checkBoxSelectionModel;
        private final JCheckBox _checkBox = new JCheckBox();
        private boolean _leafsOnly;

        /**
         * Der Konstruktor.
         *
         * @param editor                 ein Editor
         * @param checkBoxSelectionModel ein Selektions-Model
         */
        public TreeCellEditorWrapper(TreeCellEditor editor, TreeSelectionModel checkBoxSelectionModel) {
            this._delegate = editor;
            this._checkBoxSelectionModel = checkBoxSelectionModel;
            setLayout(new BorderLayout());
            _checkBox.setOpaque(true);
            _checkBox.addMouseListener(new MouseAdapter() {
                // Wenn der Editor aktiv ist, kann man auf dessen Checkbox drücken; dieses MouseEvent wird hier nur
                // zum Beenden des Editors genutzt. Denkbar ist die Weitergabe an den übergeordneten CheckBoxTree,
                // doch erlaubt dieser MouseEvent nicht die Bestimmung des TreePaths. IDEE: Vielleicht kann man
                // tree und row aus dem letzten getTreeCellEditorComponent-Aufruf verwenden?! (Der DefaultTreeCellEditor
                // hat Fields lastPath und lastRow, aber keine Zugriffsmethoden.)
                @Override
                public void mouseClicked(final MouseEvent e) {
                    _delegate.stopCellEditing();
                }
            });
            setOpaque(false);
        }

        @Nullable
        @Override
        public Component getTreeCellEditorComponent(JTree tree, Object value, boolean isSelected, boolean expanded, boolean leaf, int row) {
            if (null == _delegate) {
                return null; // No editing at all.
            }
            if (_leafsOnly && !leaf) {
                return null;
            }

            removeAll();
            _checkBox.setSelected(_checkBoxSelectionModel.isPathSelected(tree.getPathForRow(row)));
            _checkBox.setBackground(Color.WHITE);
            add(_checkBox, BorderLayout.WEST);
            add(_delegate.getTreeCellEditorComponent(tree, value, isSelected, expanded, leaf, row), BorderLayout.CENTER);
            return this;
        }

        @Override
        public Object getCellEditorValue() {
            return _delegate.getCellEditorValue();
        }

        /**
         * Hier wird die Entscheidung nur dann dem Original-Editor überlassen, wenn auch in dessen Komponente geklickt wurde. Mausklicks auf die
         * Check-Box führen hier nicht zu einem Editier-Vorgang.
         *
         * @param anEvent ein Event
         *
         * @return {@code true} falls das Editieren begonnen werden kann
         */
        @SuppressWarnings("OverlyNestedMethod")
        @Override
        public boolean isCellEditable(EventObject anEvent) {
	        if (anEvent instanceof MouseEvent me) {
                Object source = me.getSource();
		        if (source instanceof JTree tree) {
                    TreePath path = tree.getPathForLocation(me.getX(), me.getY());
                    if (path != null) {
                        Rectangle bounds = tree.getPathBounds(path);
                        if (bounds != null && me.getX() <= bounds.x + CHECKBOX_WIDTH) {
                            return false;
                        }
                        if (_leafsOnly) {
                            Object object = path.getLastPathComponent();
	                        if (object instanceof TreeNode node) {
                                if (!tree.getModel().isLeaf(node)) {
                                    return false;
                                }
                            }
                        }
                    }
                }
            }
            return _delegate.isCellEditable(anEvent);
        }

        @Override
        public boolean shouldSelectCell(EventObject anEvent) {
            return _delegate.shouldSelectCell(anEvent);
        }

        @Override
        public boolean stopCellEditing() {
            return _delegate.stopCellEditing();
        }

        @Override
        public void cancelCellEditing() {
            _delegate.cancelCellEditing();
        }

        @Override
        public void addCellEditorListener(CellEditorListener l) {
            _delegate.addCellEditorListener(l);
        }

        @Override
        public void removeCellEditorListener(CellEditorListener l) {
            _delegate.removeCellEditorListener(l);
        }

        public void setLeafsOnly(boolean leafsOnly) {
            this._leafsOnly = leafsOnly;
        }

    }
}
