/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.filtering;

import de.bsvrz.dav.daf.main.config.Attribute;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeListDefinition;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.pat.sysbed.dataview.selectionManagement.CellKey;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Diese Klasse kapselt eine Attributgruppe, die gegebenenfalls einen AtgFilter für sich besitzt. Demzufolge hat sie in erster Linien Methoden, die
 * aus dem Interface {@code Attributgroup} oder dessen Sub-Interfaces stammen, zwei Methode, um an Attributgruppe und Filter zu gelangen, und {@code
 * createColumnStrings}.
 *
 * @author Kappich Systemberatung
 */
public class FilterAttributeGroup {

    private static final String SEPARATOR = CellKey.getSECOND_SEPARATOR();

    private final AttributeGroup _attributeGroup;
    private final AtgFilter _atgFilter;

    /**
     * Ein Konstruktor mit Attributgruppe und einem Filter.
     *
     * @param attributeGroup eine Attributgruppe
     * @param atgFilter      ein Filter oder {@code null}
     */
    public FilterAttributeGroup(final AttributeGroup attributeGroup, @Nullable AtgFilter atgFilter) {
        _attributeGroup = attributeGroup;
        _atgFilter = atgFilter;
    }

    /**
     * Ein Konstruktor mit Attributgruppe und einem Filternamen.
     *
     * @param attributeGroup eine Attributgruppe
     * @param atgFilterName  ein Filtername oder {@code null}
     */
    public FilterAttributeGroup(final AttributeGroup attributeGroup, @Nullable String atgFilterName) {
        _attributeGroup = attributeGroup;
        _atgFilter = AtgFilterManager.getInstance().getFilter(atgFilterName);
    }

    private static void appendToColumnList(final String prefix, final Attribute attribute, final List<String> columnStringList,
                                           final boolean withColumnsForArrayListDefs) {
        final AttributeType attributeType = attribute.getAttributeType();
	    if (attributeType instanceof AttributeListDefinition ald) {
            if (withColumnsForArrayListDefs) {
                if (attribute.isArray()) {    // Arrays können leer sein, und dann ist es von Vorteil einen Eintrag zu haben.
                    columnStringList.add(prefix);
                }
            }
            for (Attribute subAttribute : ald.getAttributes()) {
                final String newPrefix = prefix + SEPARATOR + subAttribute.getName();
                appendToColumnList(newPrefix, subAttribute, columnStringList, withColumnsForArrayListDefs);
            }
        } else {
            columnStringList.add(prefix);
        }
    }

    @SuppressWarnings({"UseOfSystemOutOrSystemErr", "unused"})
    private static void dumpCollection(Collection<String> collection) {
        for (String item : collection) {
            System.out.println(item);
        }
    }

    /**
     * Gibt die Attributgruppe zurück.
     *
     * @return die Attrtibutgruppe
     */
    public AttributeGroup getAttributeGroup() {
        return _attributeGroup;
    }

    /**
     * Gibt den Namen, die Pid oder die Id der Attributgruppe zurück.
     *
     * @return Name, Pid oder Id der Attributgruppe
     */
    public String getNameOrPidOrId() {
        return _attributeGroup.getNameOrPidOrId();
    }

    /**
     * Gibt die Pid oder die Id der Attributgruppe zurück.
     *
     * @return Pid oder Id der Attributgruppe
     */
    public String getPidOrId() {
        return _attributeGroup.getPidOrId();
    }

    /**
     * Gibt die Pid der Attributgruppe zurück.
     *
     * @return Pid der Attributgruppe
     */
    public String getPid() {
        return _attributeGroup.getPid();
    }

    /**
     * Gibt den Filter zurück.
     *
     * @return den Filter oder {@code null}
     */
    @Nullable
    public AtgFilter getAtgFilter() {
        return _atgFilter;
    }

    /**
     * Gibt die Attribute der Filter-Attributgruppe zurück, d.h. im Falle eines vorhandenen Filters werden nur die nicht herausgefilterten Attribute
     * zurückgegeben.
     *
     * @return s.o.
     */
    public List<Attribute> getAttributes() {
        if (_atgFilter == null) {
            return _attributeGroup.getAttributes();
        } else {
            List<Attribute> attributes = new ArrayList<>();
            for (AtgFilterNode node : _atgFilter.getFilterAttributes()) {
                Object object = node.getUserObject();
                if (object instanceof Attribute) {
                    attributes.add((Attribute) object);
                }
            }
            return attributes;
        }
    }

    /**
     * Ein "ColumnString" meint hier einen String zur Beschreibung der Spalte bestehend aus Attribut-Gruppen-Pid und angehängt den Namen der Attribute
     * (durch Kommas separiert).
     *
     * @param withColumnsForArrayListDefs
     *
     * @return
     */
    public List<String> createColumnStrings(final boolean withColumnsForArrayListDefs) {
        if (_atgFilter == null) {
            return createColumnStringsNoFilter(withColumnsForArrayListDefs);
        } else {
            return createColumnStringsFilter();
        }
    }

    /* Diese und die folgende Methode entsprechen der alten filterlosen Implementation, die in
    CellKeyColumn durchgeführt wurde. */
    private List<String> createColumnStringsNoFilter(final boolean withColumnsForArrayListDefs) {
        List<String> columnStringList = new ArrayList<>();
        final List<Attribute> attributes = _attributeGroup.getAttributes();
        for (Attribute attribute : attributes) {
            final String prefix = _attributeGroup.getPidOrId() + SEPARATOR + attribute.getName();
            appendToColumnList(prefix, attribute, columnStringList, withColumnsForArrayListDefs);
        }
//		dumpCollection(columnStringList);
        return columnStringList;
    }

    /* Diese und die folgende Methode bilden die neue Implementation für Situationen mit Filter. */
    private List<String> createColumnStringsFilter() {
        List<String> columnStringList = new ArrayList<>();
        if (_atgFilter != null) {
            Object root = _atgFilter.getRoot();
	        if (root instanceof AtgFilterNode rootNode) {
                Object userObject = rootNode.getUserObject();
                if (userObject instanceof AttributeGroup) {
                    String prefix = ((AttributeGroup) userObject).getPidOrId();
                    if (!rootNode.isSuppressed()) {
                        columnStringList.add(prefix);
                    }
                    for (int i = 0; i < _atgFilter.getChildCount(root); ++i) {
                        workForChild(_atgFilter.getChild(root, i), prefix, columnStringList);
                    }
                }
            }

        }
//		dumpCollection(columnStringList);
        return columnStringList;
    }

    private void workForChild(@Nullable final Object child, final String prefix, final List<String> columnStringList) {
        if (child != null) {
	        if (child instanceof AtgFilterNode childNode) {
                Object userObject = childNode.getUserObject();
		        if (userObject instanceof Attribute attribute) {
                    String newPrefix = prefix + SEPARATOR + attribute.getName();
                    if (!childNode.isSuppressed()) {
                        columnStringList.add(newPrefix);
                    }
                    for (int i = 0; i < _atgFilter.getChildCount(child); ++i) {
                        workForChild(_atgFilter.getChild(child, i), newPrefix, columnStringList);
                    }
                }
            }
        }
    }

    @Override
    public String toString() {
        return "FilterAttributeGroup{" + "_attributeGroup=" + _attributeGroup + ", _atgFilter=" + _atgFilter + '}';
    }
}
