/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.selectionManagement;

import de.bsvrz.pat.sysbed.dataview.filtering.FilterAttributeGroup;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Eine CellKeyColumn steht für eine Spalte eines CellKeys.
 *
 * @author Kappich Systemberatung
 */
public class CellKeyColumn {

    private static final Map<FilterAttributeGroup, List<String>> _columnListsWithCfLD = new HashMap<>();
    // FilterAttributeGroup -> Column-String-Liste
    private static final Map<FilterAttributeGroup, Map<String, Integer>> _columnIndexMapsWithCfLD = new HashMap<>();
    // FilterAttributeGroup -> Map : Column-String -> Index in der Column-String-Liste
    private static final Map<FilterAttributeGroup, List<String>> _columnListsWithoutCfLD = new HashMap<>();
    // FilterAttributeGroup -> Column-String-Liste
    private static final Map<FilterAttributeGroup, Map<String, Integer>> _columnIndexMapsWithoutCfLD = new HashMap<>();
    private final String _columnString;
    private final boolean _superColumn;
    // FilterAttributeGroup -> Map : Column-String -> Index in der Column-String-Liste

    /**
     * Konstruiert die CellKeyColumn aus einem String, der keine Array-Informationen enthälten darf und einem Indikator, ob es sich um eine
     * übergreifende Spalte handelt.
     *
     * @param columnString die String-Darstellung
     * @param superColumn  ist die Spalte eine Super-Spalte?
     */
    public CellKeyColumn(final String columnString, final boolean superColumn) {
        _columnString = columnString;
        _superColumn = superColumn;
    }

    /**
     * Konstruiert die CellKeyColumn direkt aus einem CellKey dieser Spalte.
     *
     * @param key ein CellKey
     */
    public CellKeyColumn(final CellKey key) {
        _columnString = CellKey.removeArrays(key.getAttributeName());
        _superColumn = key.isSuperColumn();
    }

    /**
     * Gibt die Spaltenliste der Attributgruppe zurück, und zwar mit oder ohne die übergeordneten Spalten, die die Listen definieren.
     *
     * @param filterAttributeGroup        die Filter-Attributgruppe
     * @param withColumnsForArrayListDefs mit oder ohne übergeordneten Spalten
     *
     * @return die Spaltenliste mit Namen
     */
    public static List<String> getColumnList(final FilterAttributeGroup filterAttributeGroup,
                                             @SuppressWarnings("SameParameterValue") final boolean withColumnsForArrayListDefs) {
        List<String> columnList;
        if (withColumnsForArrayListDefs) {
            columnList = _columnListsWithCfLD.get(filterAttributeGroup);
        } else {
            columnList = _columnListsWithoutCfLD.get(filterAttributeGroup);
        }
        if (columnList == null) {
            initForAttributeGroup(filterAttributeGroup, withColumnsForArrayListDefs);
            if (withColumnsForArrayListDefs) {
                columnList = _columnListsWithCfLD.get(filterAttributeGroup);
            } else {
                columnList = _columnListsWithoutCfLD.get(filterAttributeGroup);
            }
        }
        return columnList;
    }

    private static void initForAttributeGroup(final FilterAttributeGroup filterAttributeGroup, final boolean withColumnsForArrayListDefs) {
        final List<String> columnList = filterAttributeGroup.createColumnStrings(withColumnsForArrayListDefs);
        if (withColumnsForArrayListDefs) {
            _columnListsWithCfLD.put(filterAttributeGroup, columnList);
        } else {
            _columnListsWithoutCfLD.put(filterAttributeGroup, columnList);
        }
        final int size = columnList.size();
        Map<String, Integer> columnIndexMap = new HashMap<>();
        for (int index = 0; index < size; index++) {
            columnIndexMap.put(columnList.get(index), index);
        }
        if (withColumnsForArrayListDefs) {
            _columnIndexMapsWithCfLD.put(filterAttributeGroup, columnIndexMap);
        } else {
            _columnIndexMapsWithoutCfLD.put(filterAttributeGroup, columnIndexMap);
        }
//		dumpForAttributeGroup( filterAttributeGroup, withColumnsForArrayListDefs);
    }

    @SuppressWarnings({"unused", "UseOfSystemOutOrSystemErr"})
    private static void dumpForAttributeGroup(final FilterAttributeGroup filterAttributeGroup, final boolean withColumnsForArrayListDefs) {
        final String nameOrPidOrId = filterAttributeGroup.getNameOrPidOrId();
        final List<String> columnList;
        if (withColumnsForArrayListDefs) {
            columnList = _columnListsWithCfLD.get(filterAttributeGroup);
        } else {
            columnList = _columnListsWithoutCfLD.get(filterAttributeGroup);
        }
        if (columnList == null) {
            System.out.println("Keine Column-Liste vorhanden!");
        } else {
            System.out.println("Column-Liste:");
            for (String column : columnList) {
                System.out.println(column);
            }
        }
        final Map<String, Integer> columnIndexMap;
        if (withColumnsForArrayListDefs) {
            columnIndexMap = _columnIndexMapsWithCfLD.get(filterAttributeGroup);
        } else {
            columnIndexMap = _columnIndexMapsWithoutCfLD.get(filterAttributeGroup);
        }
        if (columnIndexMap == null) {
            System.out.println("Column-Index-Map:");
        } else {
            System.out.println("Column-Index-Map:");
            for (final Map.Entry<String, Integer> entry : columnIndexMap.entrySet()) {
                System.out.println(entry.getKey() + ": " + entry.getValue());
            }
        }

    }

    /**
     * Gibt an, ob es sich um eine übergreifende Spalte handelt.
     *
     * @return ist die Spalte eine Super-Spalte?
     */
    public boolean isSuperColumn() {
        return _superColumn;
    }

    /**
     * Vergleicht die CellKeyColumn this mit dem String, ohne zunächst eine CellKeyColumn dafür anzulegen - und natürlich ohne den übergreifenden
     * Spaltenstatus zu berücksichtigen.
     *
     * @param columnString eine String-Darstellung einer CellKeyColumn
     *
     * @return sind die CellKeyColumns gmäß String-Darstellung gleich?
     */
    public boolean isEqualTo(final String columnString) {
        return _columnString.equals(columnString);
    }

    /**
     * Gibt den Index der CellKeyColumn in der Attributgruppe zurück, wobei dies der Wert in der entsprechenden Liste mit oder ohne Spalten für
     * Listen-Definitione ist. Wird kein Index gefunden, gibt die Methode {@code null} zurück.
     *
     * @param filterAttributeGroup        die Filter-Attributgruppe
     * @param withColumnsForArrayListDefs mit oder ohne übergeordneten Spalten
     *
     * @return ein Index oder {@code null}
     */
    @Nullable
    public Integer getIndexInColumnList(final FilterAttributeGroup filterAttributeGroup,
                                        @SuppressWarnings("SameParameterValue") final boolean withColumnsForArrayListDefs) {
        Map<String, Integer> columnIndexMap;
        if (withColumnsForArrayListDefs) {
            columnIndexMap = _columnIndexMapsWithCfLD.get(filterAttributeGroup);
        } else {
            columnIndexMap = _columnIndexMapsWithoutCfLD.get(filterAttributeGroup);
        }
        if (columnIndexMap == null) {
            initForAttributeGroup(filterAttributeGroup, withColumnsForArrayListDefs);
            if (withColumnsForArrayListDefs) {
                columnIndexMap = _columnIndexMapsWithCfLD.get(filterAttributeGroup);
            } else {
                columnIndexMap = _columnIndexMapsWithoutCfLD.get(filterAttributeGroup);
            }
        }
        final Integer index = columnIndexMap.get(_columnString);
        if (index == null) {
            for (final Map.Entry<String, Integer> entry : columnIndexMap.entrySet()) {
                if (entry.getKey().startsWith(_columnString)) {
                    return entry.getValue();
                }
            }
        }
        return index;
    }

    @Override
    public String toString() {
        return "CellKeyColumn{" + "_columnString='" + _columnString + '\'' + ", _superColumn=" + _superColumn + '}';
    }
}
