/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 * Copyright 2022 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.pat.sysbed.main;

import com.google.common.base.Splitter;
import de.bsvrz.dav.daf.main.ApplicationCloseActionHandler;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientDavParameters;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.pat.sysbed.dataview.filtering.AtgFilterManager;
import de.bsvrz.pat.sysbed.plugins.api.ExternalModule;
import de.bsvrz.pat.sysbed.plugins.archiveinfo.ArchiveInfoModule;
import de.bsvrz.pat.sysbed.plugins.archiverequest.StreamBasedArchiveRequestModule;
import de.bsvrz.pat.sysbed.plugins.configdata.ConfigurationDataModule;
import de.bsvrz.pat.sysbed.plugins.dataavailibility.DataAvailibilityModule;
import de.bsvrz.pat.sysbed.plugins.datareceiver.ShowCurrentDataModule;
import de.bsvrz.pat.sysbed.plugins.datasender.SendCurrentDataModule;
import de.bsvrz.pat.sysbed.plugins.datgen.DatGenModule;
import de.bsvrz.pat.sysbed.plugins.onlinetable.OnlineTableModule;
import de.bsvrz.pat.sysbed.plugins.onlprot.OnlineProtocolModule;
import de.bsvrz.pat.sysbed.plugins.parameditor.ParameterEditorModule;
import de.bsvrz.pat.sysbed.plugins.subscriptions.ShowSubscriptionInfo;
import de.bsvrz.pat.sysbed.plugins.sysprot.SystemProtocolModule;
import de.bsvrz.pat.sysbed.preselection.tree.TreeNodeObject;
import de.bsvrz.sys.funclib.application.AbstractGUIApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.configObjectAcquisition.ConfigurationHelper;
import de.bsvrz.sys.funclib.debug.Debug;

import javax.swing.*;
import javax.swing.border.EmptyBorder;
import java.awt.*;
import java.net.URL;
import java.util.List;
import java.util.*;

/**
 * Mit dieser Klasse wird die Anwendung "Generischer Test Monitor" gestartet. Die {@link de.bsvrz.pat.sysbed.plugins.api.ExternalModule Module}, die
 * zu der Applikation gehören sollen, können hier hinzugefügt werden. Der Auswahlbaum wird hier für den {@link
 * de.bsvrz.pat.sysbed.preselection.tree.PreselectionTree PreselectionTree} erstellt und an die Applikation übergeben. Außerdem wird die Verbindung
 * zum Datenverteiler und die Argumentliste, welche beim Aufruf der {@code main}-Methode angegeben wurde, übergeben.
 * <p>
 * Der Login-Dialog, welcher es ermöglicht, eine IP-Adresse mit Port, Benutzernamen und Passwort einzugeben, kann über den Aufrufparameter {@code
 * -autologin} ausgeschaltet werden. Allerdings werden dann die Parameter {@code -benutzer} und {@code -authentifizierung} benötigt.
 *
 * @author Kappich Systemberatung
 */
//public class GenericTestMonitor implements StandardApplication {
public class GenericTestMonitor extends AbstractGUIApplication implements ApplicationCloseActionHandler {

    /** Der Debug-Logger */
    private static final Debug _debug = Debug.getLogger();
    /** speichert die Argumente, die beim Aufruf übergeben wurden */
    private static final List<String> _argumentList = new LinkedList<>();
    /** speichert die Aufrufargumente für eventuelle Neustarts */
    private static String[] _args;
    /** speichert das "Logo" */
    private Icon _logo;
    /** speichert die Objekt-Pids der Argumente */
    private Iterable<String> _objects;

    /** speichert die Plugins */
    private Iterable<String> _plugins;

    /** speichert das Aufrufargument für die Onlinetabelle */
    private String _warningOnlineTable;

    /** speichert das Aufrufargument für die Archivanfrage */
    private String _warningArchiveRequest;

    /* ############### Methoden ################## */

    /** Öffentlichen Konstruktor dieser Klasse überschrieben, damit kein Objekt dieser Klasse erstellt werden kann. */
    GenericTestMonitor() {
    }

    /**
     * Die Applikation "Generischer Test Monitor" wird über die {@code main}-Methode gestartet.
     *
     * @param args Argumente, die beim Start der Applikation übergeben wurden
     */
    public static void main(String[] args) {
        //Menüs defaultmäßig in Mac-Menüzeile verschieben, wenn nichts anderes über VM-Argument -Dapple.laf.useScreenMenuBar=... vorgegeben wurde
        System.getProperties().put("apple.laf.useScreenMenuBar", System.getProperties().getOrDefault("apple.laf.useScreenMenuBar", "true"));
        _args = new String[args.length];
        System.arraycopy(args, 0, _args, 0, args.length);

        for (String arg : args) { // vor dem Aufruf von run - da sonst ein Teil der Argumente fehlen
            if (arg.startsWith("-autologin") || arg.startsWith("-plugins") || arg.startsWith("-warnungOnlineTabelle=") ||
                arg.startsWith("-warnungArchivAnfrage")) {
                // do nothing - Aufrufparameter wird hier ausgewertet und aus der Argumentliste rausgefiltert
            } else {
                _argumentList.add(arg);
            }
        }

        StandardApplicationRunner.run(new GenericTestMonitor(), args);
    }

    @Override
    public String getApplicationName() {
        return "Generischer Testmonitor";
    }

    /**
     * Mit der {@code main}-Methode übergebene Parameter können hier ausgewertet werden.
     *
     * @param argumentList die modifizierte Argumentliste von der Standardapplikation
     *
     * @throws Exception Falls ein ungültiges Argument gefunden wurde.
     */
    @Override
    public void parseArguments(ArgumentList argumentList) throws Exception {
        // der ConfigurationHelper könnte hier genutzt werden, allerdings wird dort das "," als Separator verwendet.
        String argument = argumentList.fetchArgument("-objekt=datenAuswahl.TestMenü01").asNonEmptyString();

        String plugins = argumentList.fetchArgument("-plugins=").asString();

        _plugins = Splitter.on(",").trimResults().omitEmptyStrings().split(plugins);
        _objects = Splitter.on(";").trimResults().omitEmptyStrings().split(argument);

        _warningOnlineTable = argumentList.fetchArgument("-warnungOnlineTabelle=1000").asString();
        _warningArchiveRequest = argumentList.fetchArgument("-warnungArchivAnfrage=2000").asString();
    }

    /**
     * Die Applikation wird erstellt, Module und Logo hinzugefügt und der Baum für die {@link de.bsvrz.pat.sysbed.preselection.panel.PreselectionPanel
     * Datenidentifikationsauswahl} wird erstellt. Anschließend wird die Anwendung gestartet.
     *
     * @param connection Verbindung zum Datenverteiler
     *
     * @throws Exception Falls es zu einer unerwarteten Ausnahme kommt.
     */
    @Override
    public void initialize(final ClientDavInterface connection) throws Exception {
        // die eingegebenen Werte des Login-Dialogs werden ausgelesen und die entsprechenden Aufrufargumente ersetzt.
        final ClientDavParameters davParameters = connection.getClientDavParameters();
        final String davString =
            "-datenverteiler=" + davParameters.getDavCommunicationAddress() + ":" + davParameters.getDavCommunicationSubAddress();
        final String userName = "-benutzer=" + connection.getLocalUser().getName();
        for (String s : _argumentList) {
            if (s.startsWith("-datenverteiler=")) {
                final int i = _argumentList.indexOf(s);
                _argumentList.set(i, davString);
            } else if (s.startsWith("-benutzer=")) {
                final int i = _argumentList.indexOf(s);
                _argumentList.set(i, userName);
            }
        }

        _debug.info("Durch Login-Dialog geänderte Aufrufargumente", _argumentList);

        try {
            URL url = GenericTestMonitorApplication.class.getResource("dtv-logo.png");
            if(url != null) {
                _logo = new ImageIcon(url);
            }
        } catch (Exception e) {
            _debug.warning("Logo konnte nicht geladen werden", e);
        }

        final Collection<Object> treeNodes = createTreeNodeObjects(connection);

        AtgFilterManager.getInstance(connection);

        connection.setCloseHandler(this);

        startGenericTestMonitor(connection, treeNodes);
    }

    /**
     * Erstellt anhand der Aufrufparameter den Vorauswahlbaum.
     *
     * @param connection Verbindung zum Datenverteiler
     *
     * @return der Vorauswahlbaum
     */
    private Collection<Object> createTreeNodeObjects(final ClientDavInterface connection) {
        DataModel dataModel = connection.getDataModel();

        TreeNodeObject treeNodeObject01 = new TreeNodeObject("Alles", "alles");

//		TreeNodeObject treeNodeObject02 = new TreeNodeObject("mq.a10.0000");
//		Filter filter04 = new Filter(Filter.OBJECT, new String[]{"mq.a10.0000"}, connection);
//		treeNodeObject02.addFilter(filter04);

        // Collection darf TreeNodeObject und SystemObject enthalten
        final Collection<Object> treeNodes = new LinkedList<>();
        treeNodes.add(treeNodeObject01);
//		treeNodes.add(treeNodeObject02);

        Set<SystemObject> allMenuObjects = new TreeSet<>();
        // Objekte des Aufrufparameters einfügen
        for (final String _object : _objects) {
            String object = _object.trim();
            try {
                List<SystemObject> objects = ConfigurationHelper.getObjects(object, dataModel);
                allMenuObjects.addAll(objects);
            } catch (IllegalArgumentException ignore) {
                if (!object.equals("datenAuswahl.TestMenü01")) {
                    _debug.warning("Objekt wird nicht ins Auswahlmenü aufgenommen, da '" + object + "' nicht aufgelöst werden konnte");
                }
            }
        }
        for (SystemObject menuObject : allMenuObjects) {
            if (menuObject.isOfType("typ.datenAuswahl")) {
                treeNodes.add(menuObject);
            } else {
                _debug.warning("Objekt wird nicht ins Auswahlmenü aufgenommen, da es nicht vom Typ 'typ.datenAuswahl' ist", menuObject);
            }
        }
        return treeNodes;
    }

    @Override
    public Class<?> getMainClass() {
        return this.getClass();
    }

    @Override
    public void close(String error) {
        Object[] options = {"Beenden", "Erneut starten"};
        int n = JOptionPane
            .showOptionDialog(null, "Die Verbindung zum Datenverteiler ist unterbrochen. Die Anwendung wird beendet.", "Verbindung unterbrochen",
                              JOptionPane.OK_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE, null, options, options[0]);
        if (n == 1) {
            StandardApplicationRunner.restart(new GenericTestMonitor(), _args);
        }
        System.exit(1);
    }

    /**
     * Startet den Generischen TestMonitor und übergibt die Verbindung zum Datenverteiler und den Vorauswahlbaum.
     *
     * @param connection Verbindung zum Datenverteiler
     * @param treeNodes  Knoten des Vorauswahlbaumes
     */
    private void startGenericTestMonitor(final ClientDavInterface connection, final Collection<Object> treeNodes) {

        final ProgressViewer progressViewer = new ProgressViewer();
        progressViewer.showProgress("Konfigurationsobjekte werden geladen");
        connection.getDataModel().getType("typ.konfigurationsObjekt").getObjects();
//		progressViewer.showProgress("Dynamische Objekte werden geladen");
//		connection.getDataModel().getType("typ.dynamischesObjekt").getObjects();
        progressViewer.close();

        SwingUtilities.invokeLater(() -> {
            try {
                Toolkit.getDefaultToolkit().setDynamicLayout(true);
                final GenericTestMonitorApplication application = new GenericTestMonitorApplication(getApplicationName(), connection, treeNodes);
                application.setArgumentList(_argumentList);
                if (_logo != null) {
                    application.addLogo(_logo);
                } else {
                    _debug.warning("Logo wurde nicht gefunden!");
                }

                // Liste mit Modulen, null == Separator
                final List<ExternalModule> modules = new ArrayList<>();

                modules.add(null);
                modules.add(new ParameterEditorModule());
                modules.add(new DataAvailibilityModule());
                modules.add(new OnlineTableModule(_warningOnlineTable));
                modules.add(new ConfigurationDataModule());
                modules.add(new ShowCurrentDataModule());
                modules.add(new SendCurrentDataModule());
                modules.add(new StreamBasedArchiveRequestModule(_warningArchiveRequest));
                modules.add(new ArchiveInfoModule());
                modules.add(null);
                modules.add(new SystemProtocolModule());
                modules.add(new OnlineProtocolModule());
                modules.add(new ShowSubscriptionInfo());
                modules.add(new DatGenModule());
                modules.add(null);

                // Plugins ermitteln und hinzufügen
                for (String plugin : _plugins) {
                    if (plugin.equals("-")) {
                        if (modules.isEmpty() || modules.get(modules.size() - 1) != null) {
                            modules.add(null);
                        }
                        continue;
                    }
                    if (!plugin.isEmpty()) {
                        try {
                            Class<ExternalModule> externalModuleClass = (Class<ExternalModule>) Class.forName(plugin);
                            if (modules.stream().anyMatch(it -> it != null && it.getClass() == externalModuleClass)) {
                                // Doppelte Module ignorieren
                                continue;
                            }
                            ExternalModule externalModule = externalModuleClass.newInstance();
                            modules.add(externalModule);
                        } catch (ClassNotFoundException ignore) {
                            JOptionPane.showMessageDialog(null, "Die von Ihnen angegebene Plugin-Klasse " + plugin + " existiert nicht.", "Warnung",
                                                          JOptionPane.ERROR_MESSAGE);
                        }
                    }
                }

                for (ExternalModule module : modules) {
                    if (module == null) {
                        application.addSeparator();
                    } else {
                        application.addModule(module);
                    }
                }

                application.start();
            } catch (Exception ex) {
                _debug.error("In der Initialisierungsphase ist eine unerwartete Ausnahme im GTM aufgetreten (siehe exception). Der GTM wird beendet.",
                             ex);
                ex.printStackTrace(); // Falls ein Fehler vorkommt, kann die Applikation abgebrochen werden.
                System.exit(1);
            }
        });
    }

    @Override
    public String toString() {
        return "GenericTestMonitor{" + "_logo=" + _logo + ", _objects=" + _objects + ", _plugins=" + _plugins + '}';
    }

    private static class ProgressViewer {

        private JDialog _dialog;

        private JLabel _label;

        public void showProgress(@SuppressWarnings("SameParameterValue") final String progressMessage) {
            if (!GraphicsEnvironment.isHeadless()) {
                if (_dialog == null) {
                    _dialog = new JDialog();
                    _label = new JLabel(progressMessage, UIManager.getIcon("OptionPane.informationIcon"), SwingConstants.LEFT);
                    _label.setBorder(new EmptyBorder(10, 10, 10, 10));
                    final Container contentPane = _dialog.getContentPane();
                    contentPane.add(_label, BorderLayout.CENTER);
                    final JProgressBar progressBar = new JProgressBar();
                    progressBar.setIndeterminate(true);
                    Box verticalBox = Box.createVerticalBox();
                    verticalBox.add(Box.createVerticalGlue());
                    verticalBox.add(progressBar);
                    verticalBox.setBorder(new EmptyBorder(10, 10, 10, 10));
                    contentPane.add(verticalBox, BorderLayout.SOUTH);
                    _dialog.pack();
                    Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
                    Rectangle abounds = _dialog.getBounds();
                    _dialog.setLocation((dim.width - abounds.width) / 2, (dim.height - abounds.height) / 2);
                    _dialog.setVisible(true);
                } else {
                    _label.setText(progressMessage);
                    _label.revalidate();
                }
            }
        }

        public void close() {
            if (_dialog != null) {
                _dialog.setVisible(false);
            }
        }
    }
}
