/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.plugins.api;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.LinkedListMultimap;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientDavParameters;
import de.bsvrz.dav.daf.main.authentication.AuthenticationFile;
import de.bsvrz.dav.daf.main.authentication.ClientCredentials;
import de.bsvrz.dav.daf.main.authentication.InteractiveGuiAuthentication;
import de.bsvrz.dav.daf.main.authentication.UserProperties;
import de.bsvrz.pat.sysbed.plugins.api.settings.SettingsData;
import de.bsvrz.sys.funclib.consoleProcessFrame.ConsoleProcessFrame;
import de.bsvrz.sys.funclib.consoleProcessFrame.ConsoleProcessPanel;
import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Supplier;
import javax.swing.JOptionPane;

/**
 * Basisklasse für Module, die einen externen Java-Prozess (DAF-Standardapplikation) starten. Diese Klasse unterstützt dabei, Datenverteiler,
 * Benutzer, Authentifizierung usw. zu erledigen. für die
 *
 * @author Kappich Systemberatung
 * @since 3.14
 */
public abstract class ExternalProcessModule extends ExternalModuleAdapter {

    private ConsoleProcessFrame _javaProcessFrame;

    private static String joinEntry(final Map.Entry<String, String> entry) {
        if (entry.getValue() == null) {
            return entry.getKey();
        }
        return entry.getKey() + "=" + entry.getValue();
    }

    /**
     * Gibt die Standardargumente zurück. Aktuell ist dies "-benutzer", "-authentifizierung", und "-datenverteiler". Es können in Zukunft weitere
     * Argumente ergänzt werden.
     * <p>
     * Diese Methode kann in Ausnahmefällen überschrieben werden, um eigene Werte zu setzen.
     *
     * @return Standardargumente
     */
    protected LinkedHashMap<String, String> getStandardArguments() {
        LinkedHashMap<String, String> result = new LinkedHashMap<>();
        ClientDavInterface connection = getConnection();
        String userName = connection.getLocalUser().getName();
        ClientDavParameters clientDavParameters = connection.getClientDavParameters();
        result.put("-benutzer", userName);
        result.put("-authentifizierung", "STDINDIRECT");
        result.put("-datenverteiler", clientDavParameters.getDavCommunicationAddress() + ":" + clientDavParameters.getDavCommunicationSubAddress());
        return result;
    }

    private Supplier<char[]> getPasswordSupplier() {
        UserProperties userProperties = getConnection().getClientDavParameters().getUserProperties();
        String name = getConnection().getLocalUser().getName();
        if (!(userProperties instanceof AuthenticationFile)) {
            userProperties = InteractiveGuiAuthentication.getInstance(_javaProcessFrame);
        }
        final UserProperties finalUserProperties = userProperties;
        return () -> {
            ClientCredentials clientCredentials = finalUserProperties.getClientCredentials(name);
            if (clientCredentials == null) {
                return null;
            }
            return clientCredentials.toString().toCharArray();
        };
    }

    /**
     * Startet einen Java-Prozess.
     *
     * @param settingsData Einstellungen dieses Moduls. Mit desen Einstellungen wird {@link #getParameterList(SettingsData)} aufgerufen.
     * @param mainClass    Java-Main-Klasse des zu startenden Prozesses
     * @param title        Titel des Fensters
     */
    public void startProcess(final SettingsData settingsData, final Class<?> mainClass, final String title) {
        final LinkedListMultimap<String, String> parameterList = getParameterList(settingsData);
        for (Map.Entry<String, String> entry : getStandardArguments().entrySet()) {
            parameterList.replaceValues(entry.getKey(), ImmutableList.of(entry.getValue()));
        }

        String[] arguments = parameterList.entries().stream().map(ExternalProcessModule::joinEntry).toArray(String[]::new);
        try {
            ConsoleProcessPanel panel = ConsoleProcessPanel.createJavaProcessPanel(mainClass.getName(), arguments, null, null);
            _javaProcessFrame = ConsoleProcessFrame.createConsoleProcessFrame(title, panel);
            panel.setPasswordSupplier(getPasswordSupplier(), "\u21a0");
            panel.start();
        } catch (IOException ex) {
            String message = ex.getMessage();
            JOptionPane
                .showMessageDialog(getApplication().getParent(), message, "Fehler beim Starten von \"" + title + "\"", JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Sammelt alle Einstellungen in einer Liste und gibt sie zurück. Die Liste enthält die Parameter für den Datengenerator. Einstellungen, die auch
     * in {@link #getStandardArguments()} erzeugt werden, werden ignoriert. Um diese zu überschreiben muss {@link #getStandardArguments()}
     * überschrieben werden.
     * <p>
     * Dieses Verhalten ist aus Gründen der Kompatibilität erforderlich, damit alte Einstellungen nicht den Verbindungsaufbau behindern, o. ä.
     * </p>
     *
     * @param settingsData Einstellungsdaten
     *
     * @return Liste aller Einstellungen dieses Dialogs
     */
    protected abstract LinkedListMultimap<String, String> getParameterList(SettingsData settingsData);
}
