/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.plugins.api;

import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.pat.sysbed.dataview.filtering.AtgFilter;
import de.bsvrz.pat.sysbed.dataview.filtering.AtgFilterDefDialog;
import de.bsvrz.pat.sysbed.dataview.filtering.AtgFilterManager;
import java.awt.Component;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

/**
 * @author Kappich Systemberatung
 */
public class FilterPanel extends JPanel implements AtgFilterManager.AtgFilterListener, DataIdentificationChoice.ChoiceListener {

    private static final String NEW_FILTER = "Neuer Filter";

    private final JButton _newAndEditFilterButton = new JButton(NEW_FILTER);
    private final JComboBox<AtgFilter> _filterSelectionComboBox = new JComboBox<>();

    private AttributeGroup _attributeGroup;

    private Component _parent;

    public FilterPanel() {
        super();
        _attributeGroup = null;
        _parent = null;
        initPanel();
        AtgFilterManager.getInstance().addListener(this);
    }

    public void setAttributeGroup(final AttributeGroup atg) {
        _attributeGroup = atg;
        _filterSelectionComboBox.setModel(new DefaultComboBoxModel<>(AtgFilterManager.getInstance().getAllFilters(atg)));
        if (_filterSelectionComboBox.getModel().getSize() > 0) {
            _filterSelectionComboBox.setSelectedIndex(0);
        }
    }

    public void setParent(final Component parent) {
        _parent = parent;
    }

    /**
     * Gibt den Namen des ausgewählten Filters zurück.
     *
     * @return der Name des ausgewählten Filters
     */
    public String getFilterName() {
        Object selected = _filterSelectionComboBox.getSelectedItem();
	    if (selected != null && selected instanceof AtgFilter atgFilter) {
            return atgFilter.getName();
        } else {
            return "";
        }
    }

    /**
     * Selektiert den Filter mit dem übergebenen Namen in der Combobox, falls ein solcher existiert.
     *
     * @param filterName ein Name
     */
    public void setFilterByName(final String filterName) {
        for (int i = 0; i < _filterSelectionComboBox.getModel().getSize(); ++i) {
            if (_filterSelectionComboBox.getModel().getElementAt(i).getName().equals(filterName)) {
                _filterSelectionComboBox.setSelectedIndex(i);
                break;
            }
        }
    }

    private void initPanel() {
        JPanel innerFilterPanel = new JPanel();
        innerFilterPanel.setLayout(new BoxLayout(innerFilterPanel, BoxLayout.X_AXIS));
        JLabel filterSelectionLabel = new JLabel("Filter: ");

        innerFilterPanel.add(filterSelectionLabel);
        innerFilterPanel.add(Box.createHorizontalStrut(5));
        innerFilterPanel.add(_filterSelectionComboBox);
        innerFilterPanel.add(Box.createHorizontalStrut(5));
        innerFilterPanel.add(_newAndEditFilterButton);

        setBorder(BorderFactory.createTitledBorder("Filterauswahl"));
        setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
        add(innerFilterPanel);

        /* Listener */
        _filterSelectionComboBox.addActionListener(e -> updateNewAndEditButton());

        _newAndEditFilterButton.addActionListener(e -> {
            if (_attributeGroup != null) {
                if (_newAndEditFilterButton.getText().equals(NEW_FILTER)) {
                    AtgFilterDefDialog defDialog = new AtgFilterDefDialog(_attributeGroup, _parent);
                    defDialog.setVisible(true);
                } else {  // Bearbeiten
                    Object selectedItem = _filterSelectionComboBox.getSelectedItem();
                    if (null != selectedItem) {
                        AtgFilter filter = (AtgFilter) selectedItem;
                        AtgFilterDefDialog defDialog = new AtgFilterDefDialog(filter.getCopy(), _parent);
                        defDialog.setVisible(true);
                    } else {
                        JOptionPane
                            .showMessageDialog(_parent, "Bitte wählen Sie einen Filter aus", "Kein Filter ausgewählt", JOptionPane.ERROR_MESSAGE);
                    }
                }
            } else { // Sollte bei richtiger Anwendung nicht passieren.
                JOptionPane.showMessageDialog(_parent, "Es wurde noch keine Attributgruppe gesetzt", "Keine Attributgruppe gesetzt",
                                              JOptionPane.ERROR_MESSAGE);
            }
        });
    }

    @Override
    public void filterAdded(AtgFilter newFilter) {
        if (newFilter.getAttributeGroup().equals(_attributeGroup)) {
            _filterSelectionComboBox.setModel(new DefaultComboBoxModel<>(AtgFilterManager.getInstance().getAllFilters(_attributeGroup)));
            _filterSelectionComboBox.setSelectedItem(newFilter);
        }
    }

    @Override
    public void filterChanged(AtgFilter changedFilter) {
        if (changedFilter.getAttributeGroup().equals(_attributeGroup)) {
            _filterSelectionComboBox.setModel(new DefaultComboBoxModel<>(AtgFilterManager.getInstance().getAllFilters(_attributeGroup)));
            _filterSelectionComboBox.setSelectedItem(changedFilter);
        }
    }

    @Override
    public void filterRemoved(AtgFilter removedFilter) {
        if (removedFilter.getAttributeGroup().equals(_attributeGroup)) {
            Object selectedItem = _filterSelectionComboBox.getSelectedItem();
            _filterSelectionComboBox.setModel(new DefaultComboBoxModel<>(AtgFilterManager.getInstance().getAllFilters(_attributeGroup)));
            _filterSelectionComboBox.setSelectedItem(selectedItem);
            updateNewAndEditButton();
        }
    }

    @Override
    public String toString() {
        return "FilterPanel{" + "_filterSelectionComboBox=" + _filterSelectionComboBox + ", _attributeGroup=" + _attributeGroup + ", _parent=" +
               _parent + '}';
    }

    @Override
    public void attributeGroupChanged(final AttributeGroup newAtg) {
        _attributeGroup = newAtg;
        _filterSelectionComboBox.setModel(new DefaultComboBoxModel<>(AtgFilterManager.getInstance().getAllFilters(_attributeGroup)));
        updateNewAndEditButton();
    }

    private void updateNewAndEditButton() {
        if (_filterSelectionComboBox.getSelectedItem() != null) {
            Object item = _filterSelectionComboBox.getSelectedItem();
	        if (item instanceof AtgFilter atgFilter) {
                if (!atgFilter.isValid()) {
                    _newAndEditFilterButton.setText(NEW_FILTER);
                } else {
                    _newAndEditFilterButton.setText("Bearbeiten");
                }
            }
        } else {
            _newAndEditFilterButton.setText(NEW_FILTER);
        }
    }
}
