/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.plugins.archiveinfo;

import de.bsvrz.dav.daf.main.archive.ArchiveQueryResult;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestManager;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.awt.BorderLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.io.IOException;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import java.util.function.Function;
import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

/**
 * Klasse für den Fortschrittsdialog
 *
 * @author Kappich Systemberatung
 */
public class ArchiveInfoRequester {
    private static final Debug _debug = Debug.getLogger();
    private final Window _parent;
    private final ArchiveRequestManager _requestManager;
    private ExecutorService _service = Executors.newSingleThreadExecutor();

    public ArchiveInfoRequester(@Nullable final Window parent, final ArchiveRequestManager requestManager) {
        _parent = parent;
        _requestManager = requestManager;
    }

    public Window getParent() {
        return _parent;
    }

    @Nullable
    public <E extends ArchiveQueryResult> E execute(Function<ArchiveRequestManager, E> function) {
        if (SwingUtilities.isEventDispatchThread()) {
            throw new IllegalStateException("Funktion darf nicht auf EDT ausgeführt werden");
        }
        E result = function.apply(_requestManager);
        FutureTask<E> task = new FutureTask<>(() -> {
            if (result.isRequestSuccessful()) {
                return result;
            } else {
                throw new IOException(result.getErrorMessage());
            }
        });
        _service.submit(task);
        AbortTaskDialog dialog;
        dialog = new AbortTaskDialog(_parent, task, result);
        SwingUtilities.invokeLater(() -> {
            dialog.setVisible(true);
        });
        try {
            return task.get();
        } catch (CancellationException | InterruptedException ignored) {
            return null;
        } catch (ExecutionException e) {
            SwingUtilities.invokeLater(() -> {
                _debug.warning("Die Anfrage konnte nicht durchgeführt werden", e);
                JOptionPane.showMessageDialog(_parent, "Die Anfrage konnte nicht durchgeführt werden:\n" + e.toString());
            });
            return null;
        } finally {
            SwingUtilities.invokeLater(dialog::dispose);
        }
    }

    public <E extends ArchiveQueryResult> void executeAsync(Function<ArchiveRequestManager, E> function) {
        new Thread(() -> {
            E result = execute(function);
            if (result == null) {
                return;
            }
            try {
                String errorMessage = result.getErrorMessage();
                SwingUtilities.invokeLater(() -> JOptionPane.showMessageDialog(_parent, errorMessage));
            } catch (InterruptedException ignored) {
            }
        }).start();
    }

    private static final class AbortTaskDialog extends JDialog {
        public <E extends ArchiveQueryResult> AbortTaskDialog(final Window parent, final FutureTask<E> task, final E result) {
            super(parent);
            JPanel contentPane = new JPanel(new BorderLayout());
            setContentPane(contentPane);
            JLabel label = new JLabel("Anfrage wird durchgeführt...");
            label.setBorder(BorderFactory.createEmptyBorder(15, 15, 15, 15));
            contentPane.add(label, BorderLayout.CENTER);
            JPanel buttonBar = new JPanel();
            buttonBar.setLayout(new BoxLayout(buttonBar, BoxLayout.X_AXIS));
            JButton cancelButton = new JButton();
            buttonBar.add(Box.createHorizontalGlue());
            buttonBar.add(cancelButton);
            buttonBar.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
            cancelButton.setAction(new AbstractAction("Abbrechen") {
                @Override
                public void actionPerformed(final ActionEvent e) {
                    task.cancel(true);
                }
            });

            contentPane.add(buttonBar, BorderLayout.SOUTH);
            pack();
            setLocationRelativeTo(parent);
        }
    }
}
