/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.plugins.archiveinfo;

import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKindCombination;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.dav.daf.main.archive.ArchiveOrder;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestOption;
import de.bsvrz.dav.daf.main.archive.ArchiveTimeSpecification;
import de.bsvrz.dav.daf.main.archive.TimingType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.pat.sysbed.plugins.api.settings.SettingsData;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Backend-Klasse für die Einstellungen einer Info-Anfrage
 *
 * @author Kappich Systemberatung
 */
public class ArchiveInfoSettings {

    private static final String KEY_DATAKIND = "archivetype";
    private static final String KEY_FROM = "from";
    private static final String KEY_TO = "to";
    private static final String KEY_TIMING = "timingoption";
    private static final String KEY_TIMING_TYPE = "timingtype";
    private static final String KEY_DELAYED_ORDER = "order";
    private static final String KEY_ARCHIVE = "archive";

    private final List<SystemObject> _objects;
    private final Set<ArchiveDataKind> _dataKinds = new HashSet<>();
    private final ArchiveInfoModule _archiveInfoModule;
    private final List<SystemObjectType> _objectTypes;
    private DataDescription _dataDescription;
    private long _timeTo;
    private long _timeFrom;
    private TimingOption _timeOption;
    private boolean _queryByArchiveTime;
    private boolean _sortDelayedByDataTime;
    private String _archive;

    public ArchiveInfoSettings(final ArchiveInfoModule archiveInfoModule, final SettingsData settingsData) {
        _archiveInfoModule = archiveInfoModule;
        _objectTypes = settingsData.getObjectTypes();
        _dataDescription =
            new DataDescription(settingsData.getAttributeGroup(), settingsData.getAspect(), (short) settingsData.getSimulationVariant());
        _objects = new ArrayList<>(settingsData.getObjects());
        Map<String, String> settingsMap = settingsData.getSettingsMap();
        String datakinds = settingsMap.getOrDefault(KEY_DATAKIND, "oa on na nn");
        for (String dk : datakinds.split(" ")) {
            switch (dk) {
                case "oa":
                    _dataKinds.add(ArchiveDataKind.ONLINE);
                    break;
                case "on":
                    _dataKinds.add(ArchiveDataKind.ONLINE_DELAYED);
                    break;
                case "na":
                    _dataKinds.add(ArchiveDataKind.REQUESTED);
                    break;
                case "nn":
                    _dataKinds.add(ArchiveDataKind.REQUESTED_DELAYED);
                    break;
            }
        }
        _timeFrom = Long.parseLong(settingsMap.getOrDefault(KEY_FROM, "-1"));
        _timeTo = Long.parseLong(settingsMap.getOrDefault(KEY_TO, "-1"));
        _timeOption = TimingOption.valueOf(settingsMap.getOrDefault(KEY_TIMING, "Everything"));
        _queryByArchiveTime = "archive".equals(settingsMap.get(KEY_TIMING_TYPE));
        _sortDelayedByDataTime = "data".equals(settingsMap.get(KEY_DELAYED_ORDER));
        _archive = settingsMap.getOrDefault(KEY_ARCHIVE, "");
    }

    public List<SystemObject> getObjects() {
        return _objects;
    }

    public Set<ArchiveDataKind> getDataKinds() {
        return _dataKinds;
    }

    public ArchiveInfoModule getArchiveInfoModule() {
        return _archiveInfoModule;
    }

    public DataDescription getDataDescription() {
        return _dataDescription;
    }

    public void setDataDescription(final DataDescription dataDescription) {
        _dataDescription = dataDescription;
    }

    public long getTimeTo() {
        return _timeTo;
    }

    public void setTimeTo(final long timeTo) {
        _timeTo = timeTo;
    }

    public long getTimeFrom() {
        return _timeFrom;
    }

    public void setTimeFrom(final long timeFrom) {
        _timeFrom = timeFrom;
    }

    public TimingOption getTimeOption() {
        return _timeOption;
    }

    public void setTimeOption(final TimingOption timeOption) {
        _timeOption = timeOption;
    }

    public boolean isQueryByArchiveTime() {
        return _queryByArchiveTime;
    }

    public void setQueryByArchiveTime(final boolean queryByArchiveTime) {
        _queryByArchiveTime = queryByArchiveTime;
    }

    public boolean isSortDelayedByDataTime() {
        return _sortDelayedByDataTime;
    }

    public void setSortDelayedByDataTime(final boolean sortDelayedByDataTime) {
        _sortDelayedByDataTime = sortDelayedByDataTime;
    }

    public SettingsData getSettings(final String title) {
        SettingsData settingsData =
            new SettingsData(_archiveInfoModule.getModuleName(), _archiveInfoModule.getClass(), _objectTypes, _dataDescription.getAttributeGroup(),
                             _dataDescription.getAspect(), _objects);
        settingsData.setTitle(title);
        Map<String, String> settingsMap = new HashMap<>();
        StringBuilder dataKinds = new StringBuilder();
        if (_dataKinds.contains(ArchiveDataKind.ONLINE)) {
            dataKinds.append(" oa");
        }
        if (_dataKinds.contains(ArchiveDataKind.ONLINE_DELAYED)) {
            dataKinds.append(" on");
        }
        if (_dataKinds.contains(ArchiveDataKind.REQUESTED)) {
            dataKinds.append(" na");
        }
        if (_dataKinds.contains(ArchiveDataKind.REQUESTED_DELAYED)) {
            dataKinds.append(" nn");
        }
        settingsMap.put(KEY_DATAKIND, dataKinds.toString().trim());
        settingsMap.put(KEY_FROM, String.valueOf(_timeFrom));
        settingsMap.put(KEY_TO, String.valueOf(_timeTo));
        settingsMap.put(KEY_TIMING, _timeOption.name());
        settingsMap.put(KEY_TIMING_TYPE, _queryByArchiveTime ? "archive" : "data");
        settingsMap.put(KEY_DELAYED_ORDER, _sortDelayedByDataTime ? "data" : "archive");
        settingsMap.put(KEY_ARCHIVE, _archive);
        settingsData.setSettingsMap(settingsMap);
        return settingsData;
    }

    public List<ArchiveDataSpecification> getDataSpecifications() {
        boolean relative;
        long from;
        long to;
        long actualTo;
        if (_timeTo == -1) {
            to = Long.MAX_VALUE;
            actualTo = System.currentTimeMillis();
        } else {
            to = _timeTo;
            actualTo = _timeTo;
        }
	    from = switch (_timeOption) {
		    case DataSets -> {
                relative = true;
			    yield _timeFrom;
		    }
		    case Minutes -> {
                relative = false;
			    yield minus (actualTo, ChronoUnit.MINUTES);
		    }
		    case Hours -> {
                relative = false;
			    yield minus (actualTo, ChronoUnit.HOURS);
		    }
		    case Days -> {
                relative = false;
			    yield minus (actualTo, ChronoUnit.DAYS);
		    }
		    case Weeks -> {
                relative = false;
			    yield minus (actualTo, ChronoUnit.WEEKS);
		    }
		    case Months -> {
                relative = false;
			    yield minus (actualTo, ChronoUnit.MONTHS);
		    }
		    case Years -> {
                relative = false;
			    yield minus (actualTo, ChronoUnit.YEARS);
		    }
		    case Everything -> {
                relative = false;
			    yield 0;
		    }
		    case Absolute -> {
                relative = false;
			    yield _timeFrom;
		    }
		    default -> throw new UnsupportedOperationException();
	    };
        TimingType timingType = _queryByArchiveTime ? TimingType.ARCHIVE_TIME : TimingType.DATA_TIME;
        ArchiveTimeSpecification timeSpec = new ArchiveTimeSpecification(timingType, relative, from, to);
        ArchiveOrder order = _sortDelayedByDataTime ? ArchiveOrder.BY_DATA_TIME : ArchiveOrder.BY_INDEX;

        final List<ArchiveDataSpecification> result = new ArrayList<>(_objects.size());
        for (SystemObject object : _objects) {
            result.add(new ArchiveDataSpecification(timeSpec, new ArchiveDataKindCombination(_dataKinds), order, ArchiveRequestOption.NORMAL,
                                                    _dataDescription, object));
        }
        return result;
    }

    private long minus(final long to, final ChronoUnit chronoUnit) {
        LocalDateTime localDateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(to), ZoneId.systemDefault());
        LocalDateTime result = localDateTime.minus(_timeFrom, chronoUnit);
        return result.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
    }

    public String getArchive() {
        return _archive;
    }

    public void setArchive(final String archive) {
        _archive = archive;
    }

    public List<SystemObjectType> getObjectTypes() {
        return _objectTypes;
    }
}
