/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.plugins.archiveinfo;

import static de.bsvrz.pat.sysbed.plugins.archiveinfo.ArchiveInfoResultDialog.DATE_TIME_FORMATTER;


import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;
import com.google.common.collect.TreeRangeSet;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.dav.daf.main.archive.ArchiveInformationResult;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GradientPaint;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;
import java.time.Instant;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.TreeSet;
import javax.swing.JViewport;

/**
 * Swing-Komponente für die Datenansicht
 *
 * @author Kappich Systemberatung
 */
public class DataView extends Component {
    public static final Color AVAILABLE_COLOR_TOP = new Color(90, 240, 70, 180);
    public static final Color AVAILABLE_COLOR_BOTTOM = new Color(60, 160, 40, 180);
    public static final Color AVAILABLE_COLOR_HILIGHT = new Color(100, 250, 80, 180);
    public static final Color AVAILABLE_COLOR_SHADOW = new Color(50, 150, 30, 180);

    public static final Color DELETED_COLOR_TOP = new Color(240, 240, 240, 180);
    public static final Color DELETED_COLOR_BOTTOM = new Color(160, 160, 160, 180);
    public static final Color DELETED_COLOR_HILIGHT = new Color(250, 250, 250, 180);
    public static final Color DELETED_COLOR_SHADOW = new Color(150, 150, 150, 180);
    public static final int ITEM_HEIGHT = 16;
    private static final int X_AXIS_HEIGHT = 96;
    private static final int Y_AXIS_WIDTH = 16;
    private final TreeSet<Long> _timestamps;
    private final Map<ArchiveDataSpecification, List<ArchiveInformationResult>> _data;
    private final RangeSet<Long> _selection = TreeRangeSet.create();
    private final JViewport _parentViewport;
    private TreeMap<Long, Integer> _timestampToPosition;
    private TreeMap<Integer, Long> _positionToTimestamp;
    private Range<Long> _mainSelection;

    public DataView(final TreeSet<Long> timestamps, final Map<ArchiveDataSpecification, List<ArchiveInformationResult>> data,
                    final JViewport parentViewport) {
        _parentViewport = parentViewport;
        parentViewport.addChangeListener(e -> repaint());
        _timestamps = timestamps;
        _timestampToPosition = new TreeMap<>();
        _positionToTimestamp = new TreeMap<>();
        Long previous = _timestamps.first();
        int position = 0;
        _timestampToPosition.put(previous, position);
        _positionToTimestamp.put(position, previous);
        for (Long timestamp : timestamps.tailSet(previous, false)) {
            double distance = timestamp - previous;
            distance = distance / 3600_000;
            distance = Math.max(distance, 10 * Math.sqrt(distance));
            distance = Math.max(distance, 1);
            distance = Math.min(distance, 128);
            position += distance;
            _timestampToPosition.put(timestamp, position);
            _positionToTimestamp.put(position, timestamp);
            previous = timestamp;
        }

        _data = data;
        int width = mapTimestampToX(timestamps.last());
        Dimension size = new Dimension(Y_AXIS_WIDTH + width + 1, data.size() * ITEM_HEIGHT + X_AXIS_HEIGHT + 1);
        setMinimumSize(size);
        setPreferredSize(size);
        setMaximumSize(size);

        MouseAdapter mouseAdapter = new MouseAdapter() {

            long pressedX = -1;

            @Override
            public void mouseMoved(final MouseEvent e) {
                setMainSelection(Range.singleton(mapXToTimestamp(e.getX() - Y_AXIS_WIDTH)));
            }

            @Override
            public void mousePressed(final MouseEvent e) {
                pressedX = mapXToTimestamp(e.getX() - Y_AXIS_WIDTH);
                setMainSelection(Range.singleton(pressedX));
            }

            @Override
            public void mouseDragged(final MouseEvent e) {
                long draggedX = mapXToTimestamp(e.getX() - Y_AXIS_WIDTH);
                setMainSelection(Range.encloseAll(Arrays.asList(pressedX, draggedX)));
            }

            @Override
            public void mouseReleased(final MouseEvent e) {
                long releasedX = mapXToTimestamp(e.getX() - Y_AXIS_WIDTH);
                Range<Long> range = Range.encloseAll(Arrays.asList(pressedX, releasedX));
                boolean empty = false;
                if (range.isEmpty() || range.lowerEndpoint().equals(range.upperEndpoint())) {
                    empty = true;
                }
                if (e.isShiftDown()) {
                    if (!empty) {
                        _selection.add(range);
                    }
                } else if (e.isControlDown()) {
                    if (!empty) {
                        _selection.remove(range);
                    }
                } else {
                    _selection.clear();
                    if (!empty) {
                        _selection.add(range);
                    }
                }
                selectionChanged();
                setMainSelection(null);
                repaint();
            }
        };

        addMouseListener(mouseAdapter);
        addMouseMotionListener(mouseAdapter);
    }

    protected void selectionChanged() {

    }

    public int mapTimestampToX(final Long timestamp) {
        return _timestampToPosition.get(timestamp);
    }

    public long mapXToTimestamp(final Integer x) {
        Map.Entry<Integer, Long> floorEntry = _positionToTimestamp.floorEntry(x);
        Map.Entry<Integer, Long> ceilEntry = _positionToTimestamp.ceilingEntry(x);
        if (ceilEntry == null || (floorEntry != null && x - floorEntry.getKey() < ceilEntry.getKey() - x)) {
            return floorEntry.getValue();
        } else {
            return ceilEntry.getValue();
        }
    }

    @Override
    public void paint(final Graphics g) {
        super.paint(g);
        Graphics2D graphics2D = (Graphics2D) g;
        graphics2D.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
        graphics2D.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_OFF);
        Rectangle bounds = graphics2D.getClipBounds();
        long fromTime = _positionToTimestamp.floorEntry(bounds.x).getValue();
        Map.Entry<Integer, Long> ceilingEntry = _positionToTimestamp.ceilingEntry(bounds.width + bounds.x);
        if (ceilingEntry == null) {
            ceilingEntry = _positionToTimestamp.lastEntry();
        }
        long toTime = ceilingEntry.getValue();
        int rowIdx = 0;
        for (List<ArchiveInformationResult> row : _data.values()) {
            for (ArchiveInformationResult col : row) {
                if (col.getIntervalEnd() > fromTime && col.getIntervalStart() < toTime) {
                    int x = mapTimestampToX(col.getIntervalStart());
                    int width = mapTimestampToX(col.getIntervalEnd()) - x - 1;
                    if (!col.isDataGap()) {
                        graphics2D.setPaint(new GradientPaint(0, rowIdx * ITEM_HEIGHT, AVAILABLE_COLOR_TOP, 0, (1 + rowIdx) * ITEM_HEIGHT - 1,
                                                              AVAILABLE_COLOR_BOTTOM));
                        graphics2D.fillRect(Y_AXIS_WIDTH + x, rowIdx * ITEM_HEIGHT, width, ITEM_HEIGHT - 1);
                        graphics2D.setPaint(AVAILABLE_COLOR_HILIGHT);
                        graphics2D.drawLine(Y_AXIS_WIDTH + x, rowIdx * ITEM_HEIGHT, Y_AXIS_WIDTH + x, (rowIdx + 1) * ITEM_HEIGHT - 1);
                        graphics2D.drawLine(Y_AXIS_WIDTH + x, rowIdx * ITEM_HEIGHT, Y_AXIS_WIDTH + x + width, rowIdx * ITEM_HEIGHT);
                        graphics2D.setPaint(AVAILABLE_COLOR_SHADOW);
                        graphics2D.drawLine(Y_AXIS_WIDTH + x + width, rowIdx * ITEM_HEIGHT, Y_AXIS_WIDTH + x + width, (rowIdx + 1) * ITEM_HEIGHT - 1);
                        graphics2D
                            .drawLine(Y_AXIS_WIDTH + x, (rowIdx + 1) * ITEM_HEIGHT - 1, Y_AXIS_WIDTH + x + width, (rowIdx + 1) * ITEM_HEIGHT - 1);
                    } else if (col.getVolumeIdTypB() != -1) {
                        graphics2D.setPaint(
                            new GradientPaint(0, rowIdx * ITEM_HEIGHT, DELETED_COLOR_TOP, 0, (1 + rowIdx) * ITEM_HEIGHT - 1, DELETED_COLOR_BOTTOM));
                        graphics2D.fillRect(Y_AXIS_WIDTH + x, rowIdx * ITEM_HEIGHT, width, ITEM_HEIGHT - 1);
                        graphics2D.setPaint(DELETED_COLOR_HILIGHT);
                        graphics2D.drawLine(Y_AXIS_WIDTH + x, rowIdx * ITEM_HEIGHT, Y_AXIS_WIDTH + x, (rowIdx + 1) * ITEM_HEIGHT - 1);
                        graphics2D.drawLine(Y_AXIS_WIDTH + x, rowIdx * ITEM_HEIGHT, Y_AXIS_WIDTH + x + width, rowIdx * ITEM_HEIGHT);
                        graphics2D.setPaint(DELETED_COLOR_SHADOW);
                        graphics2D.drawLine(Y_AXIS_WIDTH + x + width, rowIdx * ITEM_HEIGHT, Y_AXIS_WIDTH + x + width, (rowIdx + 1) * ITEM_HEIGHT - 1);
                        graphics2D
                            .drawLine(Y_AXIS_WIDTH + x, (rowIdx + 1) * ITEM_HEIGHT - 1, Y_AXIS_WIDTH + x + width, (rowIdx + 1) * ITEM_HEIGHT - 1);
                    }
                    graphics2D.setPaint(Color.black);
                    int volumeIdTypB = col.getVolumeIdTypB();
                    if (volumeIdTypB != -1) {
                        graphics2D.drawString(String.valueOf(volumeIdTypB), x, (rowIdx + 1) * ITEM_HEIGHT);
                    }
                }
            }
            rowIdx++;
        }
        graphics2D.setPaint(new Color(0x7FFFFFFF));
        int xAxisY = _parentViewport.getHeight() - X_AXIS_HEIGHT + _parentViewport.getViewPosition().y;
        graphics2D.fillRect(0, xAxisY, getWidth(), X_AXIS_HEIGHT);
        graphics2D.setPaint(Color.BLACK);
        int oldX = -16;
        for (Map.Entry<Integer, Long> entry : _positionToTimestamp.entrySet()) {
            Integer x = entry.getKey();
            if (x - oldX < 16) {
                continue;
            }
            graphics2D.drawLine(Y_AXIS_WIDTH + x, xAxisY, Y_AXIS_WIDTH + x, xAxisY + 2);
            drawDate(graphics2D, xAxisY, x, entry.getValue());
            oldX = x;
        }
        graphics2D.drawLine(0, 0, 0, xAxisY);

        graphics2D.drawLine(0, xAxisY, getWidth(), xAxisY);
        Range<Long> mainSelection = _mainSelection;
        if (mainSelection != null) {
            graphics2D.setPaint(Color.red);
            int xx = mapTimestampToX(mainSelection.lowerEndpoint());
            int xxx = mapTimestampToX(mainSelection.upperEndpoint());
            graphics2D.drawLine(Y_AXIS_WIDTH + xx, 0, Y_AXIS_WIDTH + xx, xAxisY);
            graphics2D.drawLine(Y_AXIS_WIDTH + xx, xAxisY, Y_AXIS_WIDTH + xx, xAxisY + 2);
            drawDate(graphics2D, xAxisY, xx, mainSelection.lowerEndpoint());
            if (xxx != xx) {
                drawDate(graphics2D, xAxisY, xxx, mainSelection.upperEndpoint());
                graphics2D.drawLine(Y_AXIS_WIDTH + xxx, 0, Y_AXIS_WIDTH + xxx, xAxisY);
                graphics2D.setPaint(new Color(255, 0, 0, 32));
                graphics2D.fillRect(Y_AXIS_WIDTH + xx, 0, xxx - xx, xAxisY);
            }
        }
        for (Range<Long> range : _selection.asRanges()) {
            graphics2D.setPaint(Color.blue);
            int xx = mapTimestampToX(range.lowerEndpoint());
            int xxx = mapTimestampToX(range.upperEndpoint());
            graphics2D.drawLine(Y_AXIS_WIDTH + xx, 0, Y_AXIS_WIDTH + xx, xAxisY);
            if (xxx != xx) {
                graphics2D.drawLine(Y_AXIS_WIDTH + xxx, 0, Y_AXIS_WIDTH + xxx, xAxisY);
                graphics2D.setPaint(new Color(0, 0, 255, 32));
                graphics2D.fillRect(Y_AXIS_WIDTH + xx, 0, xxx - xx, xAxisY);
            }
        }
    }

    private void drawDate(final Graphics2D graphics2D, final int xAxisY, final int xPos, final long timestamp) {
        AffineTransform transform = graphics2D.getTransform();
        graphics2D.rotate(Math.PI / 4, xPos, xAxisY + 2);
        Paint oldPaint = graphics2D.getPaint();
        graphics2D.setPaint(Color.white);
        graphics2D.drawString(Instant.ofEpochMilli(timestamp).atZone(ZoneId.systemDefault()).format(DATE_TIME_FORMATTER), xPos + 13,
                              -Y_AXIS_WIDTH + xAxisY + 15);
        graphics2D.drawString(Instant.ofEpochMilli(timestamp).atZone(ZoneId.systemDefault()).format(DATE_TIME_FORMATTER), xPos + 13,
                              -Y_AXIS_WIDTH + xAxisY + 13);
        graphics2D.drawString(Instant.ofEpochMilli(timestamp).atZone(ZoneId.systemDefault()).format(DATE_TIME_FORMATTER), xPos + 15,
                              -Y_AXIS_WIDTH + xAxisY + 13);
        graphics2D.drawString(Instant.ofEpochMilli(timestamp).atZone(ZoneId.systemDefault()).format(DATE_TIME_FORMATTER), xPos + 15,
                              -Y_AXIS_WIDTH + xAxisY + 15);
        graphics2D.setPaint(oldPaint);
        graphics2D.drawString(Instant.ofEpochMilli(timestamp).atZone(ZoneId.systemDefault()).format(DATE_TIME_FORMATTER), xPos + 14,
                              -Y_AXIS_WIDTH + xAxisY + 14);
        graphics2D.setTransform(transform);
    }

    public void setMainSelection(final Range<Long> mainSelection) {
        if (Objects.equals(_mainSelection, mainSelection)) {
            return;
        }
        _mainSelection = mainSelection;
        repaint();
    }

    public List<ArchiveInformationResult> getSelection() {
        final List<ArchiveInformationResult> results = new ArrayList<>();
        for (List<ArchiveInformationResult> list : _data.values()) {
            for (ArchiveInformationResult archiveInformationResult : list) {
                long intervalStart = archiveInformationResult.getIntervalStart();
                long intervalEnd = archiveInformationResult.getIntervalEnd();
                if (intervalStart < intervalEnd) {
                    if (_selection.intersects(Range.closed(intervalStart, intervalEnd))) {
                        results.add(archiveInformationResult);
                    }
                }
            }
        }
        return results;
    }

    public RangeSet<Long> getSelectionInterval() {
        return _selection;
    }
}
