/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.plugins.dataavailibility;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.pat.sysbed.main.ApplicationInterface;
import de.bsvrz.pat.sysbed.main.GenericTestMonitorApplication;
import de.bsvrz.pat.sysbed.preselection.lists.PreselectionLists;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableModel;

/**
 * Implementiert den Dialog, der dem Benutzer die Datenverfügbarkeit anzeigt und eine Objektauswahl gestattet.
 *
 * @author Kappich Systemberatung
 */
public class AvailibilityViewer {

    private static final String TIME_FORMAT = "dd.MM.yyyy HH:mm:ss,SSS";
    private static final String ONLINEDATA = "Onlinedaten verfügbar";
    private static final String NO_DATA = "Keine Daten";
    private static final String NO_RIGHTS = "Keine Rechte";
    private static final String NO_SOURCE = "Keine Quelle";
    private static final String POSSIBLE_GAP = "Mögliche Lücke";
    private static final String UNKNOWN_STATE = "Unbekannter Status";
    private final Debug _debug = Debug.getLogger();
    private final UnsubscriberFrame _frame;

    private final ApplicationInterface _application;

    private final ClientDavInterface _connection;

    private final List<SystemObject> _objects;

    private final AttributeGroup _attributeGroup;

    private final Aspect _aspect;

    private final DataDescription _dataDescription;

    private final ReceiveOptions _receiveOptions = ReceiveOptions.normal();
    private final Map<SystemObject, Integer> _rowMap;
    private final Map<Integer, SystemObject> _systemObjectMap;
    private ReceiverRole _receiverRole = ReceiverRole.receiver();
    private JTable _table;
    private JLabel _onlineDataLabel;
    private JLabel _noDataLabel;
    private JLabel _noRightsLabel;
    private JLabel _noSourceLabel;
    private JLabel _unknownStateLabel;

    /**
     * Konstruktor, der anhand der Datenidentifikation sich beim Datenverteiler anmeldet und die Daten in Tabellenform darstellt.
     *
     * @param connection        Verbindung zum Datenverteiler
     * @param objects           die zu betrachtenden Systemobjekte
     * @param attributeGroup    die Attributgruppe
     * @param aspect            der zu betrachtende Aspekt
     * @param simulationVariant die Simulationsvariante
     */
    public AvailibilityViewer(final ApplicationInterface application, final ClientDavInterface connection, final List<SystemObject> objects,
                              final AttributeGroup attributeGroup, final Aspect aspect, int simulationVariant) {
        _application = application;
        _connection = connection;
        _objects = objects;
        _attributeGroup = attributeGroup;
        _aspect = aspect;

        if (simulationVariant != -1) {
            _dataDescription = new DataDescription(_attributeGroup, _aspect, (short) simulationVariant);
        } else {
            _dataDescription = new DataDescription(_attributeGroup, _aspect);
        }

        System.getProperties().put("apple.laf.useScreenMenuBar", "true");

        _frame = new UnsubscriberFrame(_connection, _objects, _dataDescription);
        _frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

        _rowMap = new HashMap<>(_objects.size());
        _systemObjectMap = new HashMap<>(_objects.size());

        Container pane = _frame.getContentPane();
        pane.setLayout(new BorderLayout());

        pane.add(getThePanel(), BorderLayout.CENTER);
//		_frame.pack();  // sieht vor dem setSize blöd aus, beseitigt aber ein gelegentliches Artefakt
        _frame.setSize(700, 400);
    }

    /**
     * Setzt die Empfängerrolle.
     *
     * @param receiverRole die Empfängerrolle
     */
    public void setReceiverRole(final ReceiverRole receiverRole) {
        _receiverRole = receiverRole;
    }

    /**
     * Zeigt den Viewer an.
     */
    public void show() {
        final String title =
            "Datenverfügbarkeit (Attributgruppe: " + _attributeGroup.getNameOrPidOrId() + ", Aspekt: " + _aspect.getNameOrPidOrId() + ")";
        _frame.setTitle(GenericTestMonitorApplication.getTitle(title, _connection));
        _frame.setVisible(true);
        try {
            ClientReceiverInterface receiver = new DataReceiver();
            _connection.subscribeReceiver(receiver, _objects, _dataDescription, _receiveOptions, _receiverRole);
            _frame.setReceiver(receiver);
        } catch (RuntimeException ex) {
            _frame.setVisible(false);
            _frame.dispose();
            _debug.error(
                "Beim Öffnen der Datenverfügbarkeitsprüfung ist bei der Anmeldung der gewünschten Datenidentifikationen ein Fehler aufgetreten " +
                "(siehe Exception)", ex);
            throw new IllegalStateException(ex.getMessage(), ex);
        }
    }

    @SuppressWarnings("OverlyLongMethod")
    private JPanel getThePanel() {
        JPanel thePanel = new JPanel();
        thePanel.setLayout(new BorderLayout());
        thePanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
        initTable();
        thePanel.add(new JScrollPane(_table), BorderLayout.CENTER);

        class CheckBoxActionListener implements ActionListener {
            @Override
            public void actionPerformed(final ActionEvent e) {
                Object source = e.getSource();
	            if (source instanceof JCheckBox checkBox) {
                    for (int row = 0; row < _table.getModel().getRowCount(); ++row) {
                        Object value = _table.getModel().getValueAt(row, 2);
	                    if (value != null && value instanceof String valueAsStr) {
                            if (valueAsStr.equals(checkBox.getText())) {
                                _table.getModel().setValueAt(checkBox.isSelected(), row, 3);
                            }
                        }
                    }
                }
            }
        }

        _onlineDataLabel = new JLabel("(0)");
        _noDataLabel = new JLabel("(0)");
        _noRightsLabel = new JLabel("(0)");
        _noSourceLabel = new JLabel("(0)");
        _unknownStateLabel = new JLabel("(" + _objects.size() + ")");

        JCheckBox onlineDataCheckBox = new JCheckBox(ONLINEDATA);
        onlineDataCheckBox.addActionListener(new CheckBoxActionListener());
        JCheckBox noDataCheckBox = new JCheckBox(NO_DATA);
        noDataCheckBox.addActionListener(new CheckBoxActionListener());
        JCheckBox noRightsCheckBox = new JCheckBox(NO_RIGHTS);
        noRightsCheckBox.addActionListener(new CheckBoxActionListener());
        JCheckBox noSourceCheckBox = new JCheckBox(NO_SOURCE);
        noSourceCheckBox.addActionListener(new CheckBoxActionListener());
        JCheckBox unknownStateCheckBox = new JCheckBox(UNKNOWN_STATE);
        unknownStateCheckBox.addActionListener(new CheckBoxActionListener());

        JPanel onlineDataPanel = new JPanel(new FlowLayout(FlowLayout.LEFT, 5, 0));
        JPanel noDataPanel = new JPanel(new FlowLayout(FlowLayout.LEFT, 5, 0));
        JPanel noRightsPanel = new JPanel(new FlowLayout(FlowLayout.LEFT, 5, 0));
        JPanel noSourcePanel = new JPanel(new FlowLayout(FlowLayout.LEFT, 5, 0));
        JPanel unknownStatePanel = new JPanel(new FlowLayout(FlowLayout.LEFT, 5, 0));

        onlineDataPanel.add(onlineDataCheckBox);
        onlineDataPanel.add(_onlineDataLabel);
        noDataPanel.add(noDataCheckBox);
        noDataPanel.add(_noDataLabel);
        noRightsPanel.add(noRightsCheckBox);
        noRightsPanel.add(_noRightsLabel);
        noSourcePanel.add(noSourceCheckBox);
        noSourcePanel.add(_noSourceLabel);
        unknownStatePanel.add(unknownStateCheckBox);
        unknownStatePanel.add(_unknownStateLabel);

        JPanel checkBoxPanel = new JPanel();
        checkBoxPanel.setLayout(new BoxLayout(checkBoxPanel, BoxLayout.Y_AXIS));
        checkBoxPanel.add(onlineDataPanel);
        checkBoxPanel.add(noDataPanel);
        checkBoxPanel.add(noRightsPanel);
        checkBoxPanel.add(noSourcePanel);
        checkBoxPanel.add(unknownStatePanel);

        JPanel buttonPanel = new JPanel();
        buttonPanel.setLayout(new BorderLayout());
        buttonPanel.add(checkBoxPanel, BorderLayout.CENTER);
        buttonPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));

        JPanel okPanel = new JPanel();
        okPanel.setLayout(new BoxLayout(okPanel, BoxLayout.Y_AXIS));
        okPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
        okPanel.setAlignmentY(Component.BOTTOM_ALIGNMENT);
        okPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 0, 10));
        JButton okButton = new JButton("OK");
        okButton.addActionListener(e -> {
            List<SystemObject> selectedSystemObjects = new ArrayList<>();
            for (int row = 0; row < _table.getModel().getRowCount(); ++row) {
                Object value = _table.getModel().getValueAt(row, 3);
	            if (value != null && value instanceof Boolean selected) {
                    if (selected) {
                        selectedSystemObjects.add(_systemObjectMap.get(row));
                    }
                }
            }
	        if (_application instanceof GenericTestMonitorApplication genericTestMonitorApplication) {
                PreselectionLists preselectionLists = genericTestMonitorApplication.getPreselectionLists();
                preselectionLists.setPreselectedObjects(selectedSystemObjects);
                List<AttributeGroup> atgList = new ArrayList<>(1);
                atgList.add(_attributeGroup);
                preselectionLists.setPreselectedAttributeGroups(atgList);
                List<Aspect> aspectList = new ArrayList<>();
                aspectList.add(_aspect);
                preselectionLists.setPreselectedAspects(aspectList);
                //noinspection ConstantConditions
                preselectionLists.setPreselectedObjectTypes(null);
            }
            _frame.dispose();
        });
        okPanel.add(Box.createHorizontalGlue());
        okPanel.add(Box.createVerticalGlue());
        okPanel.add(okButton);
        buttonPanel.add(okPanel, BorderLayout.EAST);

        thePanel.add(buttonPanel, BorderLayout.SOUTH);
        return thePanel;
    }

    private void initTable() {
        String[] columnNames = {"Objekt", "Zeit", "Verfügbarkeit", "Auswahl"};
        Object[][] data = new Object[_objects.size()][4];
        for (int i = 0; i < _objects.size(); ++i) {
            _rowMap.put(_objects.get(i), i);
            _systemObjectMap.put(i, _objects.get(i));
            data[i][0] = _objects.get(i).getNameOrPidOrId();
            data[i][1] = "unbekannt";
            data[i][2] = "unbekannt";
            data[i][3] = Boolean.FALSE;
        }

        _table = new JTable(new MyTableModel(columnNames, data));
        _table.setDefaultRenderer(Boolean.class, new CheckBoxRenderer());
        _table.setRowHeight(30);

        _table.getColumnModel().getColumn(3).setMaxWidth(70);
    }

    @Override
    public String toString() {
        return "AvailibilityViewer{" + "_objects=" + _objects + ", _attributeGroup=" + _attributeGroup + ", _aspect=" + _aspect +
               ", _dataDescription=" + _dataDescription + ", _receiveOptions=" + _receiveOptions + ", _receiverRole=" + _receiverRole + '}';
    }

    private static class MyTableModel extends DefaultTableModel {

        MyTableModel(String[] columnNames, Object[][] data) {
            super(data, columnNames);
        }

        /*
         * Achtung: JTable benutzt diese Methode, um Default-Renderer und Default-Editor der Zellen zu bestimmen;
         * wird sie nicht überschrieben, so würde die Zellen mit Checkboxen für boolsche Werte durch die
         * Textanzeige 'true' oder 'false' ersetzt, weil der falsche Renderer benutzt würde.
         */
        @Nullable
        @Override
        public Class<?> getColumnClass(int c) {
            Object o = getValueAt(0, c);
            if (o != null) {
                return o.getClass();
            } else {
                return null;
            }
        }

        @Override
        public boolean isCellEditable(int row, int col) {
            return col == 3;
        }

    }

    private static class CheckBoxRenderer extends JPanel implements TableCellRenderer {
        public CheckBoxRenderer() {
            JCheckBox checkBox = new JCheckBox();
            checkBox.setOpaque(false);
            this.add(checkBox);
        }

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            if (isSelected) {
                setBackground(table.getSelectionBackground());
            } else {
                setBackground(table.getBackground());
            }
            JCheckBox checkBox = (JCheckBox) getComponent(0);
            if (null != value) {
                checkBox.setSelected((Boolean) value);
            }
            return this;
        }
    }

    /*
     * Die Klasse verarbeitet die Daten, die vom Datenverteiler gesandt werden, und kommt bei Darstellung zum Zug.
     */
    private class DataReceiver implements ClientReceiverInterface {

        public DataReceiver() {
        }

        @Override
        public void update(ResultData[] results) {
            for (final ResultData result : results) {
                Integer rowIndex = _rowMap.get(result.getObject());
                _table.getModel().setValueAt(getDataTime(TIME_FORMAT, result.getDataTime()), rowIndex, 1);
                DataState dataState = result.getDataState();
                if (dataState.equals(DataState.DATA)) {
                    _table.getModel().setValueAt(ONLINEDATA, rowIndex, 2);
                } else if (dataState.equals(DataState.NO_DATA)) {
                    _table.getModel().setValueAt(NO_DATA, rowIndex, 2);
                } else if (dataState.equals(DataState.NO_RIGHTS)) {
                    _table.getModel().setValueAt(NO_RIGHTS, rowIndex, 2);
                } else if (dataState.equals(DataState.NO_SOURCE)) {
                    _table.getModel().setValueAt(NO_SOURCE, rowIndex, 2);
                } else if (dataState.equals(DataState.POSSIBLE_GAP)) {
                    _table.getModel().setValueAt(POSSIBLE_GAP, rowIndex, 2);
                } else {
                    _table.getModel().setValueAt(UNKNOWN_STATE, rowIndex, 2);
                }
            }
            SwingUtilities.invokeLater(this::updateStatistics);
        }

        public String getDataTime(String format, long dataTime) {
            DateFormat timeFormat = new SimpleDateFormat(format);
            return timeFormat.format(dataTime);
        }

        private void updateStatistics() {
            TableModel tableModel = _table.getModel();
            int onlineData = 0;
            int noData = 0;
            int noRights = 0;
            int noSource = 0;
            int unknownState = 0;
            for (int row = 0; row < tableModel.getRowCount(); ++row) {
                String s = (String) tableModel.getValueAt(row, 2);
                switch (s) {
                    case ONLINEDATA:
                        ++onlineData;
                        break;
                    case NO_DATA:
                        ++noData;
                        break;
                    case NO_RIGHTS:
                        ++noRights;
                        break;
                    case NO_SOURCE:
                        ++noSource;
                        break;
                    case UNKNOWN_STATE:
                        ++unknownState;
                        break;
                }
            }
            _onlineDataLabel.setText("(" + onlineData + ")");
            _noDataLabel.setText("(" + noData + ")");
            _noRightsLabel.setText("(" + noRights + ")");
            _noSourceLabel.setText("(" + noSource + ")");
            _unknownStateLabel.setText("(" + unknownState + ")");
        }
    }
}
