/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.plugins.dataavailibility;

import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.pat.sysbed.help.GtmHelp;
import de.bsvrz.pat.sysbed.plugins.api.ButtonBar;
import de.bsvrz.pat.sysbed.plugins.api.DataIdentificationChoice;
import de.bsvrz.pat.sysbed.plugins.api.DialogInterface;
import de.bsvrz.pat.sysbed.plugins.api.ExternalModuleAdapter;
import de.bsvrz.pat.sysbed.plugins.api.settings.SettingsData;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.awt.Container;
import java.awt.FlowLayout;
import java.awt.Window;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

/**
 * Implementiert das Modul für die Datenverfügbarkeitsprüfung. Dieses Modul stellt für die ausgewählte Datenidentifikation die Verfügbarkeit von
 * Online-Daten dar und gibt dem Benutzer die Möglichkeit, eine Auswahl von Objekten zu treffen.
 *
 * @author Kappich Systemberatung
 */
public class DataAvailibilityModule extends ExternalModuleAdapter {

    /** speichert den Dialog der Onlinetabelle */
    private static DataAvailibilityDialog _dialog;
    /** speichert den Text des Tooltips */
    private String _tooltipText;

    /* ############ Methoden ############ */

    /**
     * Gibt den Namen des Moduls zurück.
     *
     * @return der Name des Moduls
     */
    public String getModuleName() {
        return "Datenverfügbarkeitsprüfung";
    }

    /**
     * Gibt den Text des Buttons zurück.
     *
     * @return Text des Buttons
     */
    public String getButtonText() {
        return "Datenverfügbarkeitsprüfung";
    }

    /**
     * Gibt den Text des Tooltips zurück.
     *
     * @return Text des Tooltips
     */
    public String getTooltipText() {
        return _tooltipText;
    }

    /**
     * Diese Methode erhält die ausgewählte Datenidentifikation und stellt die Datenverfügbarkeits-Dialog dar.
     *
     * @param settingsData enthält die ausgewählte Datenidentifikation
     */
    public void startModule(final SettingsData settingsData) {
        _dialog = new DataAvailibilityDialog(getApplication().getParent());
        _dialog.setDataIdentification(settingsData);
    }

    /**
     * Diese Methode erhält alle Einstellungen für die Anzeige des Datenverfügbarkeits-Dialog.
     *
     * @param settingsData die Einstellungsdaten
     */
    public void startSettings(final SettingsData settingsData) {
        _dialog = new DataAvailibilityDialog(getApplication().getParent());
        _dialog.startSettings(settingsData);
    }

    /**
     * Diese Methode erhält alle Einstellungen für die Datenverfügbarkeitsprüfung und startet den {@link DataAvailibilityDialog Dialog}. Dieser wird
     * mit den Einstellungsdaten gefüllt.
     *
     * @param settingsData die Einstellungsdaten
     */
    public void change(final SettingsData settingsData) {
        _dialog = new DataAvailibilityDialog(getApplication().getParent());
        _dialog.setSettings(settingsData);
    }

    /**
     * Gibt an, ob die Vorauswahl den Anforderungen der Datenverfügbarkeitsprüfung genügen.
     *
     * @param settingsData enthält die ausgewählte Datenidentifikation
     *
     * @return gibt an, ob die Vorauswahl den Anforderungen der Datenverfügbarkeitsprüfung genügen
     */
    public boolean isPreselectionValid(final SettingsData settingsData) {
        if (!super.isPreselectionValid(settingsData)) {
            _tooltipText = "Genau eine Attributgruppe, ein Aspekt und mindestens ein Objekt müssen ausgewählt sein.";
            return false;
        }

        // ATGV prüfen
        final AttributeGroupUsage atgUsage = settingsData.getAttributeGroup().getAttributeGroupUsage(settingsData.getAspect());
        if (atgUsage == null || atgUsage.isConfigurating()) {
            _tooltipText = "Es muss eine Online-Attributgruppenverwendung ausgewählt werden.";
            return false;
        }
        _tooltipText = "Auswahl übernehmen";
        return true;
    }


    /* ############### Klasse DataAvailibilityDialog ############## */

    /**
     * Stellt einen Dialog zur Verfügung, mit dem Einstellungen für die Datenverfügbarkeitsprüfung gemacht werden können. Diese Einstellungen können
     * gespeichert werden. Durch betätigen des "OK"-Buttons wird die Datenverfügbarkeitsprüfung gestartet.
     */
    private class DataAvailibilityDialog implements DialogInterface {

        /** gibt an, welche Rollen zur Verfügung stehen */
        private final String[] _roleUnit = {"Empfänger"};
        /** speichert eine Parent-Komponente */
        private final Window _parent;
        /** speichert den Dialog */
        private JDialog _dialog;
        /** speichert eine Instanz der Datenidentifikationsauswahl */
        private DataIdentificationChoice _dataIdentificationChoice;
        /** hier kann die Rolle ausgewählt werden */
        private JComboBox<String> _roleComboBox;

        /* ############### Methoden ################### */

        /** Standardkonstruktor. Ein Objekt der Klasse wird angelegt. */
        public DataAvailibilityDialog(@Nullable final Window parent) {
            _parent = parent;
        }

        /**
         * Mit dieser Methode können die Datenidentifikationsdaten übergeben werden. Der Dialog wird mit Default-Werten dargestellt.
         *
         * @param data enthält die ausgewählte Datenidentifikation
         */
        public void setDataIdentification(final SettingsData data) {
            if (_dialog == null) {
                createDialog();
            }
            _dataIdentificationChoice.setDataIdentification(data.getObjectTypes(), data.getAttributeGroup(), data.getAspect(), data.getObjects(),
                                                            data.getSimulationVariant());
            _dataIdentificationChoice.showTree(getApplication().getTreeNodes(), getApplication().getConnection(), data.getTreePath());
            showDialog();
        }

        /**
         * Diese Methode zeigt den Dialog an und trägt die Einstellungsdaten in die entsprechenden Felder ein.
         *
         * @param data die Einstellungsdaten
         */
        public void setSettings(final SettingsData data) {
            if (_dialog == null) {
                createDialog();
            }
            _dataIdentificationChoice.setDataIdentification(data.getObjectTypes(), data.getAttributeGroup(), data.getAspect(), data.getObjects(),
                                                            data.getSimulationVariant());
            _dataIdentificationChoice.showTree(getApplication().getTreeNodes(), getApplication().getConnection(), data.getTreePath());
            Map<String, String> settingsMap = data.getSettingsMap();
            if (settingsMap.containsKey("rolle")) {
                setRole(settingsMap.get("rolle"));
            }
            showDialog();
        }

        /**
         * Startet die Onlinetabelle anhand der Einstellungsdaten.
         *
         * @param settingsData die Einstellungsdaten
         */
        public void startSettings(final SettingsData settingsData) {
            List<SystemObject> objects = settingsData.getObjects();
            AttributeGroup atg = settingsData.getAttributeGroup();
            Aspect asp = settingsData.getAspect();
            int simulationVariant = settingsData.getSimulationVariant();

            AvailibilityViewer viewer = new AvailibilityViewer(getApplication(), getConnection(), objects, atg, asp, simulationVariant);
            Map<String, String> settingsMap = settingsData.getSettingsMap();
            if (settingsMap.containsKey("rolle")) {
                String value = settingsMap.get("rolle");
                if (value.equals("empfaenger")) {
                    viewer.setReceiverRole(ReceiverRole.receiver());
                } else if (value.equals("senke")) {
                    viewer.setReceiverRole(ReceiverRole.drain());
                }
            }
            try {
                viewer.show();
            } catch (Exception ex) {
                JOptionPane.showMessageDialog(_dialog, ex.getMessage(), "Fehler bei der Anmeldung", JOptionPane.ERROR_MESSAGE);
            }
        }

        /** Erstellt den Dialog. Bestandteil ist die Datenidentifikation und die Anmeldeoption, bestehend aus der Rolle und der Anmeldeart. */
        private void createDialog() {
            _dialog = new JDialog(_parent);
            _dialog.setTitle(getButtonText());
            _dialog.setResizable(false);

            Container pane = _dialog.getContentPane();
            pane.setLayout(new BoxLayout(pane, BoxLayout.Y_AXIS));

            // Datenidentifikationsauswahl-Panel
            final List<SystemObjectType> types = new LinkedList<>();
            DataModel configuration = getConnection().getDataModel();
            types.add(configuration.getType("typ.konfigurationsObjekt"));
            types.add(configuration.getType("typ.dynamischesObjekt"));
            _dataIdentificationChoice = new DataIdentificationChoice(null, types);
            pane.add(_dataIdentificationChoice);

            // Anmeldeoptionen
            JPanel applyPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
            applyPanel.setBorder(BorderFactory.createTitledBorder("Anmeldeoptionen"));
            JLabel roleLabel = new JLabel("Rolle: ");
            _roleComboBox = new JComboBox<>(_roleUnit);
            _roleComboBox.setSelectedIndex(0);
            roleLabel.setLabelFor(_roleComboBox);
            applyPanel.add(roleLabel);
            applyPanel.add(_roleComboBox);
            applyPanel.add(Box.createHorizontalStrut(10));

            pane.add(applyPanel);

            // untere Buttonleiste
            final ButtonBar buttonBar = new ButtonBar(this);
            _dialog.getRootPane().setDefaultButton(buttonBar.getAcceptButton());
            pane.add(buttonBar);
        }

        /** Durch diese Methode wird der Dialog angezeigt. */
        private void showDialog() {
            _dialog.setLocation(70, 70);
            _dialog.pack();
            _dialog.setVisible(true);
        }

        /**
         * Gibt die ausgewählte Rolle für die Onlinetabelle zurück.
         *
         * @return die ausgewählte Rolle
         */
        private String getRole() {
            String item = (String) _roleComboBox.getSelectedItem();
            if (item.equals(_roleUnit[0])) {
                return "empfaenger";
            } else {
                return "senke";
            }
        }

        /**
         * Setzt die Rolle für die Onlinetabelle.
         *
         * @param role die Rolle für die Onlinetabelle
         */
        private void setRole(final String role) {
            int index = 0;
            if (role.equals("empfaenger")) {
                index = 0;
            } else if (role.equals("senke")) {
                index = 1;
            }
            _roleComboBox.setSelectedIndex(index);
        }

        /**
         * Erstellt die Einstellungsdaten.
         *
         * @param title der Name für die Einstellungen
         *
         * @return die Einstellungsdaten
         */
        private SettingsData getSettings(String title) {
	        Class<DataAvailibilityModule> moduleClass = DataAvailibilityModule.class;
            List<SystemObjectType> objectTypes = _dataIdentificationChoice.getObjectTypes();
            AttributeGroup atg = _dataIdentificationChoice.getAttributeGroup();
            Aspect asp = _dataIdentificationChoice.getAspect();
            List<SystemObject> objects = _dataIdentificationChoice.getObjects();

            SettingsData settingsData = new SettingsData(getModuleName(), moduleClass, objectTypes, atg, asp, objects);
            settingsData.setTitle(title);
            settingsData.setSimulationVariant(_dataIdentificationChoice.getSimulationVariant());
            settingsData.setTreePath(_dataIdentificationChoice.getTreePath());
            settingsData.setSettingsMap(getSettingsMap());

            return settingsData;
        }

        /**
         * Sammelt alle Parameter des Dialogs.
         *
         * @return Map aller Parameter des Dialogs
         */
        private Map<String, String> getSettingsMap() {
            Map<String, String> map = new HashMap<>();
            map.put("rolle", getRole());
            return map;
        }

        /** Durch betätigen des "OK"-Buttons wird die Onlinetabelle gestartet und dieser Dialog wird geschlossen. Die Parameter werden gespeichert. */
        public void doOK() {
            SettingsData settingsData = getSettings("");
            startSettings(settingsData);
            doCancel();
            saveSettings(settingsData);
        }

        /** diese Methode schließt den Dialog */
        public void doCancel() {
            _dialog.setVisible(false);
            _dialog.dispose();
        }

        /**
         * diese Methode speichert die Parameter
         *
         * @param title Titel dieser Konfiguration
         */
        public void doSave(String title) {
            SettingsData settingsData = getSettings(title);
            saveSettings(settingsData);
        }

        /**
         * Durch Betätigen des "Hilfe"-Buttons wird die Kontexthilfe geöffnet.
         */
        @Override
        public void openHelp() {
            GtmHelp.openHelp("#Datenverfügbarkeitsprüfung");
        }
    }
}
