/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.plugins.onlinetable;

import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.pat.sysbed.dataview.DataViewFrame;
import de.bsvrz.pat.sysbed.dataview.filtering.FilterAttributeGroup;
import de.bsvrz.pat.sysbed.help.GtmHelp;
import de.bsvrz.pat.sysbed.plugins.api.ButtonBar;
import de.bsvrz.pat.sysbed.plugins.api.DataIdentificationChoice;
import de.bsvrz.pat.sysbed.plugins.api.DialogInterface;
import de.bsvrz.pat.sysbed.plugins.api.ExternalModuleAdapter;
import de.bsvrz.pat.sysbed.plugins.api.FilterPanel;
import de.bsvrz.pat.sysbed.plugins.api.settings.SettingsData;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.awt.Container;
import java.awt.FlowLayout;
import java.awt.Window;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

/**
 * Implementiert das Modul für die Onlinetabelle. Dieses Modul stellt für die ausgewählte Datenidentifikation alle aktuellen Daten vom Datenverteiler
 * in Tabellendarstellung dar.
 *
 * @author Kappich Systemberatung
 */
public class OnlineTableModule extends ExternalModuleAdapter {

    /** speichert den Dialog der Onlinetabelle */
    private static OnlineTableDialog _dialog;
    /** speichert den Wert für die Warnung bei großen Anfragen */
    private final String _warningLimitString;
    /** speichert den Text des Tooltips */
    private String _tooltipText;

    /**
     * Der Konstruktor ohne Argumente.
     */
    public OnlineTableModule() {
        _warningLimitString = null;
    }

    /**
     * Der Konstruktor mit einer Grenze für Warnungen bei großen Anfragen.
     */
    public OnlineTableModule(final String warningLimit) {
        _warningLimitString = warningLimit;
    }


    /* ############ Methoden ############ */

    /**
     * Gibt den Namen des Moduls zurück.
     *
     * @return der Name des Moduls
     */
    @Override
    public String getModuleName() {
        return "Onlinetabelle";
    }

    /**
     * Gibt den Text des Buttons zurück.
     *
     * @return Text des Buttons
     */
    @Override
    public String getButtonText() {
        return "Onlinetabelle anzeigen";
    }

    /**
     * Gibt den Text des Tooltips zurück.
     *
     * @return Text des Tooltips
     */
    @Override
    public String getTooltipText() {
        return _tooltipText;
    }

    /**
     * Diese Methode erhält die ausgewählte Datenidentifikation und stellt die Onlinetabelle dar.
     *
     * @param settingsData enthält die ausgewählte Datenidentifikation
     */
    @Override
    public void startModule(final SettingsData settingsData) {
        _dialog = new OnlineTableDialog(_warningLimitString, getApplication().getParent());
        _dialog.setDataIdentification(settingsData);
    }

    /**
     * Diese Methode erhält alle Einstellungen für die Anzeige der OnlineTabelle.
     *
     * @param settingsData die Einstellungsdaten
     */
    @Override
    public void startSettings(final SettingsData settingsData) {
        _dialog = new OnlineTableDialog(_warningLimitString, getApplication().getParent());
        _dialog.startSettings(settingsData);
    }

    /**
     * Diese Methode erhält alle Einstellungen für die Onlinetabelle und startet den {@link OnlineTableDialog Dialog}. Dieser wird mit den
     * Einstellungsdaten gefüllt.
     *
     * @param settingsData die Einstellungsdaten
     */
    @Override
    public void change(final SettingsData settingsData) {
        _dialog = new OnlineTableDialog(_warningLimitString, getApplication().getParent());
        _dialog.setSettings(settingsData);
    }

    /**
     * Gibt an, ob die Vorauswahl den Anforderungen der Onlinetabelle genügen.
     *
     * @param settingsData enthält die ausgewählte Datenidentifikation
     *
     * @return gibt an, ob die Vorauswahl den Anforderungen der Onlinetabelle genügen
     */
    @Override
    public boolean isPreselectionValid(final SettingsData settingsData) {
        if (!super.isPreselectionValid(settingsData)) {
            _tooltipText = "Genau eine Attributgruppe, ein Aspekt und mindestens ein Objekt müssen ausgewählt sein.";
            return false;
        }

        // ATGV prüfen
        final AttributeGroupUsage atgUsage = settingsData.getAttributeGroup().getAttributeGroupUsage(settingsData.getAspect());
        if (atgUsage == null || atgUsage.isConfigurating()) {
            _tooltipText = "Es muss eine Online-Attributgruppenverwendung ausgewählt werden.";
            return false;
        }
        _tooltipText = "Auswahl übernehmen";
        return true;
    }


    /* ############### Klasse OnlineTableDialog ############## */

    /**
     * Stellt einen Dialog zur Verfügung, mit dem Einstellungen für die Onlinetabelle gemacht werden können. Diese Einstellungen können gespeichert
     * werden. Durch betätigen des "OK"-Buttons wird die Onlinetabelle gestartet.
     */
    private class OnlineTableDialog implements DialogInterface {

        /** gibt an, welche Rollen zur Verfügung stehen */
        private final String[] _roleUnit = {"Empfänger", "Senke"};
        /** gibt an, welche Anmeldearten zur Verfügung stehen */
        private final String[] _applyModeEntries = {"Online", "Nur geänderte Datensätze", "Auch nachgelieferte Datensätze"};
        /** gibt an, welche Darstellungsoptionen zur Verfügung stehen */
        private final String[] _displayOptions = {"Aktuelle Daten unten anhängen",
                                                  "Aktuelle Daten oben einfügen",
                                                  "Nur aktuellste Daten anzeigen",
                                                  "Daten nur in einer CSV-Datei speichern"};
        /** gibt an, welche Zeichenkodierungen zur Verfügung stehen */
        private final String[] _encodingOptions = {"ISO-8859-1", "UTF-8", "MacRoman"};
        /** hier kann eine Parent-Komponente gespeichert werden */
        private final Window _parent;
        /** speichert den Dialog */
        private JDialog _dialog;
        /** speichert eine Instanz der Datenidentifikationsauswahl */
        private DataIdentificationChoice _dataIdentificationChoice;
        /** hier kann die Rolle ausgewählt werden */
        private JComboBox<String> _roleComboBox;
        /** hier kann die Anmeldeart ausgewählt werden */
        private JComboBox<String> _applyModeCombo;
        /** hier kann die Darstellungsoption ausgewählt werden */
        private JComboBox<String> _displayOptionsComboBox;
        /** hier kann die Zeichenkodierung für den CSV-Export festgelegt werden */
        private JComboBox<String> _encodingComboBox;
        /** hier kann ein Filter ausgewählt und bearbeitet werden */
        private FilterPanel _filterPanel;
        /** das Limit, ab dem der Benutzer vor einer langen Abfrage gewarnt wird */
        private int _warningLimit = Integer.MAX_VALUE;


        /* ############### Methoden ################### */

        /** Standardkonstruktor. Ein Objekt der Klasse wird angelegt. */
        public OnlineTableDialog(@Nullable String warningLimitString, @Nullable final Window parent) {
            if (warningLimitString != null) {
                try {
	                _warningLimit = Integer.parseInt(warningLimitString);
                } catch (NumberFormatException ignore) {
                }
            }
            _parent = parent;
        }

        /**
         * Mit dieser Methode können die Datenidentifikationsdaten übergeben werden. Der Dialog wird mit Default-Werten dargestellt.
         *
         * @param data enthält die ausgewählte Datenidentifikation
         */
        public void setDataIdentification(final SettingsData data) {
            if (_dialog == null) {
                createDialog();
            }
            _dataIdentificationChoice.setDataIdentification(data.getObjectTypes(), data.getAttributeGroup(), data.getAspect(), data.getObjects(),
                                                            data.getSimulationVariant());
            _dataIdentificationChoice.showTree(getApplication().getTreeNodes(), getApplication().getConnection(), data.getTreePath());
            _filterPanel.setAttributeGroup(data.getAttributeGroup());
            _filterPanel.setParent(_dialog);
            _dataIdentificationChoice.addChoiceListener(_filterPanel);
            showDialog();
        }

        /**
         * Diese Methode zeigt den Dialog an und trägt die Einstellungsdaten in die entsprechenden Felder ein.
         *
         * @param data die Einstellungsdaten
         */
        public void setSettings(final SettingsData data) {
            if (_dialog == null) {
                createDialog();
            }
            _dataIdentificationChoice.setDataIdentification(data.getObjectTypes(), data.getAttributeGroup(), data.getAspect(), data.getObjects(),
                                                            data.getSimulationVariant());
            _dataIdentificationChoice.showTree(getApplication().getTreeNodes(), getApplication().getConnection(), data.getTreePath());
            Map<String, String> settingsMap = data.getSettingsMap();
            if (settingsMap.containsKey("rolle")) {
                setRole(settingsMap.get("rolle"));
            }
            if (settingsMap.containsKey("option")) {
                setApplyMode(settingsMap.get("option"));
            }
            if (settingsMap.containsKey("display")) {
                setDisplayOption(settingsMap.get("display"));
            }
            if (settingsMap.containsKey("csvencoding")) {
                _encodingComboBox.setSelectedItem(settingsMap.get("csvencoding"));
            }

            _filterPanel.setAttributeGroup(data.getAttributeGroup());
            _filterPanel.setParent(_dialog);
            _dataIdentificationChoice.addChoiceListener(_filterPanel);

            if (settingsMap.containsKey("filter")) {
                _filterPanel.setFilterByName(settingsMap.get("filter"));
            }

            // csv & encoding

            showDialog();
        }

        /**
         * Startet die Onlinetabelle anhand der Einstellungsdaten.
         *
         * @param settingsData die Einstellungsdaten
         */
        public void startSettings(final SettingsData settingsData) {
            List<SystemObject> objects = settingsData.getObjects();
            AttributeGroup atg = settingsData.getAttributeGroup();
            Aspect asp = settingsData.getAspect();
            int simulationVariant = settingsData.getSimulationVariant();

            // Der Filter muss schon im DataViewFrame-Konstruktor bekannt sein:
            Map<String, String> settingsMap = settingsData.getSettingsMap();
            String filterName = "";
            if (settingsMap.containsKey("filter")) {
                filterName = settingsMap.get("filter");
            }
            FilterAttributeGroup filterAttributeGroup = new FilterAttributeGroup(atg, filterName);

            DataViewFrame dataViewFrame = new DataViewFrame(getConnection(), objects, filterAttributeGroup, asp, simulationVariant);

            int displayOptions = 0;
            if (settingsMap.containsKey("rolle")) {
                String value = settingsMap.get("rolle");
                if (value.equals("empfaenger")) {
                    dataViewFrame.setReceiverRole(ReceiverRole.receiver());
                } else if (value.equals("senke")) {
                    dataViewFrame.setReceiverRole(ReceiverRole.drain());
                }
            }
            if (settingsMap.containsKey("option")) {
                String value = settingsMap.get("option");
                switch (value) {
                    case "online":
                        dataViewFrame.setReceiveOptions(ReceiveOptions.normal());
                        break;
                    case "delta":
                        dataViewFrame.setReceiveOptions(ReceiveOptions.delta());
                        break;
                    case "nachgeliefert":
                        dataViewFrame.setReceiveOptions(ReceiveOptions.delayed());
                        break;
                }
            }
            if (settingsMap.containsKey("display")) {
                displayOptions = Integer.parseInt(settingsMap.get("display"));
            }
            String charsetName = Charset.defaultCharset().displayName();
            if (settingsMap.containsKey("csvencoding")) {
                charsetName = settingsMap.get("csvencoding");
            }
            try {
                dataViewFrame.setParent(_dialog);
                dataViewFrame.showOnlineData(displayOptions, charsetName, settingsData);
            } catch (Exception ex) {
                JOptionPane.showMessageDialog(_dialog, ex.getMessage(), "Fehler bei der Anmeldung", JOptionPane.ERROR_MESSAGE);
            }
        }

        /** Erstellt den Dialog. Bestandteil ist die Datenidentifikation und die Anmeldeoption, bestehend aus der Rolle und der Anmeldeart. */
        private void createDialog() {
            _dialog = new JDialog(_parent);
            _dialog.setTitle(getButtonText());
            _dialog.setResizable(false);

            Container pane = _dialog.getContentPane();
            pane.setLayout(new BoxLayout(pane, BoxLayout.Y_AXIS));

            // Datenidentifikationsauswahl-Panel
            final List<SystemObjectType> types = new LinkedList<>();
            DataModel configuration = getConnection().getDataModel();
            types.add(configuration.getType("typ.konfigurationsObjekt"));
            types.add(configuration.getType("typ.dynamischesObjekt"));
            _dataIdentificationChoice = new DataIdentificationChoice(null, types);
            pane.add(_dataIdentificationChoice);

            // Anmeldeoptionen
            JPanel applyPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
            applyPanel.setBorder(BorderFactory.createTitledBorder("Anmeldeoptionen"));
            JLabel roleLabel = new JLabel("Rolle: ");
            _roleComboBox = new JComboBox<>(_roleUnit);
            _roleComboBox.setSelectedIndex(0);
            roleLabel.setLabelFor(_roleComboBox);
            applyPanel.add(roleLabel);
            applyPanel.add(_roleComboBox);
            applyPanel.add(Box.createHorizontalStrut(10));

            JLabel applyModeLabel = new JLabel("Anmeldeart: ");
            _applyModeCombo = new JComboBox<>(_applyModeEntries);
            _applyModeCombo.setEditable(false);
            applyModeLabel.setLabelFor(_applyModeCombo);
            applyPanel.add(applyModeLabel);
            applyPanel.add(_applyModeCombo);

            // Darstellungsoptionen
            JPanel displayOptionsPanel = new JPanel();
            displayOptionsPanel.setLayout(new BoxLayout(displayOptionsPanel, BoxLayout.Y_AXIS));
            displayOptionsPanel.setBorder(BorderFactory.createTitledBorder("Darstellungsoptionen"));

            JPanel firstLinePanel = new JPanel();
            firstLinePanel.setLayout(new BoxLayout(firstLinePanel, BoxLayout.X_AXIS));
            JLabel displayOptionsLabel = new JLabel("Darstellung: ");
            _displayOptionsComboBox = new JComboBox<>(_displayOptions);
            _displayOptionsComboBox.setSelectedIndex(0);    // Default-Einstellung
            _displayOptionsComboBox.addActionListener(e -> {
                _encodingComboBox.setEnabled(_displayOptionsComboBox.getSelectedIndex() == 3);
            });
            displayOptionsLabel.setLabelFor(_displayOptionsComboBox);
            firstLinePanel.add(displayOptionsLabel);
            firstLinePanel.add(Box.createHorizontalStrut(5));
            firstLinePanel.add(_displayOptionsComboBox);

            JPanel secondLinePanel = new JPanel();
            secondLinePanel.setLayout(new BoxLayout(secondLinePanel, BoxLayout.X_AXIS));
            JLabel encodingLabel = new JLabel("Zeichenkodierung: ");
            _encodingComboBox = new JComboBox<>(_encodingOptions);
            _encodingComboBox.setEnabled(_displayOptionsComboBox.getSelectedIndex() == 3);
            String defaultCharsetName = Charset.defaultCharset().displayName();
            if (defaultCharsetName.equals("UTF-8")) {
                _encodingComboBox.setSelectedIndex(1);
            } else {
                _encodingComboBox.setSelectedIndex(0);
            }
            secondLinePanel.add(encodingLabel);
            secondLinePanel.add(Box.createHorizontalStrut(5));
            secondLinePanel.add(_encodingComboBox);

            displayOptionsPanel.add(firstLinePanel);
            displayOptionsPanel.add(Box.createVerticalStrut(5));
            displayOptionsPanel.add(secondLinePanel);

            // Filterauswahl
            _filterPanel = new FilterPanel();
            _dataIdentificationChoice.addChoiceListener(_filterPanel);

            pane.add(applyPanel);
            pane.add(displayOptionsPanel);
            pane.add(_filterPanel);

            // untere Buttonleiste
            final ButtonBar buttonBar = new ButtonBar(this);
            _dialog.getRootPane().setDefaultButton(buttonBar.getAcceptButton());
            pane.add(buttonBar);
        }

        /** Durch diese Methode wird der Dialog angezeigt. */
        private void showDialog() {
            _dialog.setLocation(70, 70);
            _dialog.pack();
            _dialog.setVisible(true);
        }

        /**
         * Gibt die ausgewählte Rolle für die Onlinetabelle zurück.
         *
         * @return die ausgewählte Rolle
         */
        private String getRole() {
            String item = (String) _roleComboBox.getSelectedItem();
            if (item.equals(_roleUnit[0])) {
                return "empfaenger";
            } else {
                return "senke";
            }
        }

        /**
         * Setzt die Rolle für die Onlinetabelle.
         *
         * @param role die Rolle für die Onlinetabelle
         */
        private void setRole(final String role) {
            int index = 0;
            if (role.equals("empfaenger")) {
                index = 0;
            } else if (role.equals("senke")) {
                index = 1;
            }
            _roleComboBox.setSelectedIndex(index);
        }

        /**
         * Gibt die Anmeldeart zurück.
         *
         * @return die Anmeldeart
         */
        private String getApplyMode() {
            int index = _applyModeCombo.getSelectedIndex();
	        String mode = switch (index) {
		        case 0 -> "online";
		        case 1 -> "delta";
		        case 2 -> "nachgeliefert";
		        default -> "";
	        };
            return mode;
        }

        /**
         * Mit dieser Methode kann die Anmeldeart gesetzt werden.
         *
         * @param mode Anmeldeart
         */
        private void setApplyMode(final String mode) {
	        int index = switch (mode) {
		        case "online" -> 0;
		        case "delta" -> 1;
		        case "nachgeliefert" -> 2;
		        default -> 0;
	        };
            _applyModeCombo.setSelectedIndex(index);
        }

        /**
         * Gibt die ausgewählte Darstellungsoption zurück.
         *
         * @return die ausgewählte Darstellungsoption
         */
        private String getDisplayOption() {
            return String.valueOf(_displayOptionsComboBox.getSelectedIndex());
        }

        /**
         * Setzt die Darstellungsoption.
         *
         * @param index der Index der Darstellungsoption
         */
        private void setDisplayOption(final String index) {
            _displayOptionsComboBox.setSelectedIndex(Integer.parseInt(index));
        }

        /**
         * Erstellt die Einstellungsdaten.
         *
         * @param title der Name für die Einstellungen
         *
         * @return die Einstellungsdaten
         */
        private SettingsData getSettings(String title) {
            Class<?> moduleClass = OnlineTableModule.class;
            List<SystemObjectType> objectTypes = _dataIdentificationChoice.getObjectTypes();
            AttributeGroup atg = _dataIdentificationChoice.getAttributeGroup();
            Aspect asp = _dataIdentificationChoice.getAspect();
            List<SystemObject> objects = _dataIdentificationChoice.getObjects();

            SettingsData settingsData = new SettingsData(getModuleName(), moduleClass, objectTypes, atg, asp, objects);
            settingsData.setTitle(title);
            settingsData.setSimulationVariant(_dataIdentificationChoice.getSimulationVariant());
            settingsData.setTreePath(_dataIdentificationChoice.getTreePath());
            settingsData.setSettingsMap(getSettingsMap());

            return settingsData;
        }

        /**
         * Sammelt alle Parameter des Dialogs.
         *
         * @return Map aller Parameter des Dialogs
         */
        private Map<String, String> getSettingsMap() {
            Map<String, String> map = new HashMap<>();
            map.put("rolle", getRole());
            map.put("option", getApplyMode());
            map.put("display", getDisplayOption());
            map.put("filter", _filterPanel.getFilterName());
            Object object = _encodingComboBox.getSelectedItem();
            if (object != null) {
                map.put("csvencoding", (String) object);
            }
            return map;
        }

        /** Durch betätigen des "OK"-Buttons wird die Onlinetabelle gestartet und dieser Dialog wird geschlossen. Die Parameter werden gespeichert. */
        @Override
        public void doOK() {
            SettingsData settingsData = getSettings("");
            if (settingsData.getObjects().size() > _warningLimit) {
                Object[] objects = {"Durchführen", "Abbrechen"};
                int n = JOptionPane.showOptionDialog(_dialog, "Umfangreiche Aktion: es werden Daten zu " + settingsData.getObjects().size() +
                                                              " Objekten angefordert.", "Warnung", JOptionPane.OK_CANCEL_OPTION,
                                                     JOptionPane.WARNING_MESSAGE, null, objects, objects[1]);
                if (n != 0) {    // n kann 0, 1 und -1 sein!
                    return;
                }
            }
            startSettings(settingsData);
            doCancel();
            saveSettings(settingsData);
        }

        /** diese Methode schließt den Dialog */
        @Override
        public void doCancel() {
            _dialog.setVisible(false);
            _dialog.dispose();
        }

        /**
         * diese Methode speichert die Parameter
         *
         * @param title Titel dieser Konfiguration
         */
        @Override
        public void doSave(String title) {
            SettingsData settingsData = getSettings(title);
            saveSettings(settingsData);
        }

        /**
         * Durch Betätigen des "Hilfe"-Buttons wird die Kontexthilfe geöffnet.
         */
        @Override
        public void openHelp() {
            GtmHelp.openHelp("#Onlinetabelle");
        }
    }
}
