/*
 * Copyright 2004 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.preselection.tree;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.DynamicObjectType;
import de.bsvrz.dav.daf.main.config.InvalidationListener;
import de.bsvrz.dav.daf.main.config.ObjectTimeSpecification;
import de.bsvrz.dav.daf.main.config.Pid;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.TimeSpecificationType;
import de.bsvrz.pat.sysbed.preselection.treeFilter.standard.Filter;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import javax.swing.BoxLayout;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTree;
import javax.swing.SpinnerDateModel;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

/**
 * Die Klasse {@code PreselectionTree} ist ein Teil der Datenidentifikationsauswahl. Sie stellt die spezifizierte Vorauswahl in Form eines Baumes zur
 * Verfügung.
 * <p>
 * Durch die spezifizierte Vorauswahl wird die Anzahl der durch den Benutzer auswählbaren Datenidentifikationen durch verschiedene Filter
 * eingeschränkt.<p> Die Objekte werden nach der Filterung wieder zur Verfügung gestellt und können beispielsweise mit Hilfe der Klasse {@link
 * de.bsvrz.pat.sysbed.preselection.lists.PreselectionLists} dargestellt und weiter eingeschränkt werden.
 *
 * @author Kappich Systemberatung
 * @see #PreselectionTree
 * @see #addPreselectionListener
 */
public final class PreselectionTree extends JPanel {

    private final DataModel _dataModel;
    /** speichert ein Objekt der Klasse {@code PreselectionTreeHandler} */
    private final PreselectionTreeHandler _preselectionTreeHandler;
    /** speichert angemeldete Listener-Objekte */
    private final List<PreselectionTreeListener> _listenerList = new LinkedList<>();
    /** speichert einen JTree */
    private JTree _tree;
    /** Speichert alle aktuellen Systemobjekte. Die Collection wird automatisch beim Erzeugen und Löschen von dynamischen Objekten aktualisiert. */
    private Collection<SystemObject> _currentSystemObjects;
    /**
     * Speichert alle Systemobjekte die dem ausgewählten Zeitbereich entsprechen. In der Regel mit {@link #_currentSystemObjects} identisch, außer es
     * wurde ein Zeitbereich angegeben
     */
    private Collection<SystemObject> _selectedSystemObjects;
    /**
     * Speichert den Stand der zuletzt für die Filterung verwendeten Collection der Systemobjekte. Referenz auf das Objekt, das bei der letzten
     * Filterung in der Variablen {@link #_selectedSystemObjects} enthalten war.
     */
    private Collection<SystemObject> _lastUsedSystemObjects;
    /** speichert die gefilterten Systemobjekte */
    private Collection<SystemObject> _filterObjects;
    /** speichert kommaseparierte PIDs, die den Pfad im Baum angeben */
    private String _treePath;
    private TreePath _selectedTreePath;
    private JButton _updateButton;

    /**
     * Aktuelle Intervallauswahl für die Objektgültigkeit
     */
    private ObjectTimeSpecification _currentInterval = ObjectTimeSpecification.valid();

    /**
     * Bei der letzten Aktualisierung verwendete Intervallauswahl für die Objektgültigkeit
     */
    private ObjectTimeSpecification _lastInterval = ObjectTimeSpecification.valid();
    private JComboBox<TimeSpecificationType> _timeSpecificationTypeJComboBox;
    private JSpinner _fromAbsoluteDate;
    private JSpinner _toAbsoluteDate;

    /**
     * Der Konstruktor erstellt ein Objekt der Klasse {@code PreselectionTree}.
     *
     * @param connection Verbindung zum Datenverteiler
     * @param treeNodes  ein Parameter zur Spezifizierung der Vorauswahl (Baum), bestehend aus Systemobjekten und {@link TreeNodeObject
     *                   Knotenobjekten}
     *
     * @see #createAndShowGui()
     */
    public PreselectionTree(ClientDavInterface connection, Collection<Object> treeNodes) {
        _dataModel = connection.getDataModel();
        _preselectionTreeHandler = new PreselectionTreeHandler(this, connection);
        createAndShowGui();
        // erst die Oberfläche erstellen, dann die Anmeldung beim DaV und das Darstellen der Knoten
        registerDynamicObjectType();
        _preselectionTreeHandler.setTreeNodes(treeNodes);
    }

    /**
     * Die Methode wird vom Konstruktor aufgerufen und stellt einen JTree für die spezifizierte Vorauswahl zur Verfügung. Bei Auswahl eines Knotens im
     * Baum werden alle Filter auf dem Pfad von der Wurzel bis zum Knoten auf die Systemobjekte angewendet.
     */
    private void createAndShowGui() {
        _updateButton = new JButton("Aktualisieren");
        _updateButton.setEnabled(false);
        _updateButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                _updateButton.setEnabled(false);
                updateSelectedObjects();
                filterObjects(_selectedTreePath);
                notifyTreeSelectionChanged();
                updateUpdateButton();
            }
        });

        _timeSpecificationTypeJComboBox = new JComboBox<>(TimeSpecificationType.values());
        _timeSpecificationTypeJComboBox.setRenderer(new DefaultListCellRenderer() {
            @Override
            public Component getListCellRendererComponent(final JList<?> list, final Object value, final int index, final boolean isSelected,
                                                          final boolean cellHasFocus) {
                return super.getListCellRendererComponent(list, formatValue((TimeSpecificationType) value), index, isSelected, cellHasFocus);
            }

            private String formatValue(final TimeSpecificationType value) {
	            return switch (value) {
		            case VALID -> "Aktuelle Objekte";
		            case VALID_AT_TIME -> "Am Zeitpunkt";
		            case VALID_IN_PERIOD -> "Im Zeitbereich";
		            case VALID_DURING_PERIOD -> "Im ges. Zeitbereich";
	            };
            }
        });

        JPanel fromPanel = new JPanel();
        JPanel toPanel = new JPanel();
        fromPanel.setLayout(new BoxLayout(fromPanel, BoxLayout.X_AXIS));
        toPanel.setLayout(new BoxLayout(toPanel, BoxLayout.X_AXIS));
        _fromAbsoluteDate = new JSpinner(new SpinnerDateModel());
        _toAbsoluteDate = new JSpinner(new SpinnerDateModel());
        JLabel from = new JLabel("von:");
        fromPanel.add(from);
        fromPanel.add(_fromAbsoluteDate);
        JLabel to = new JLabel("bis:");
        toPanel.add(to);
        toPanel.add(_toAbsoluteDate);
        from.setMinimumSize(to.getPreferredSize());
        to.setMinimumSize(from.getPreferredSize());

        _timeSpecificationTypeJComboBox.setSelectedItem(TimeSpecificationType.VALID);
        _timeSpecificationTypeJComboBox.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(final ActionEvent e) {
                updateTimeSpecCombo(from, fromPanel, toPanel);
            }
        });

        updateTimeSpecCombo(from, fromPanel, toPanel);

        _fromAbsoluteDate.addChangeListener(e -> updateDateSelection());
        _toAbsoluteDate.addChangeListener(e -> updateDateSelection());

        JPanel refreshPanel = new JPanel();
        refreshPanel.setLayout(new GridBagLayout());
        GridBagConstraints cons = new GridBagConstraints();
        cons.fill = GridBagConstraints.HORIZONTAL;
        cons.weightx = 1;
        cons.gridx = 0;
        refreshPanel.add(_timeSpecificationTypeJComboBox, cons);
        refreshPanel.add(fromPanel, cons);
        refreshPanel.add(toPanel, cons);
        refreshPanel.add(_updateButton, cons);

        _tree = new JTree();
        _tree.setModel(new DefaultTreeModel(null));
        _tree.setEditable(false);
        _tree.setRootVisible(false);
        _tree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
//		_tree.getSelectionModel().setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
        _tree.setShowsRootHandles(true);
        _tree.addTreeSelectionListener(new TreeSelectionListener() {
            public void valueChanged(TreeSelectionEvent e) {
						/*
						{
							_filterObjects = _systemObjects;
							// einen Knoten ausgewählt:
							//  -> Filter von jedem Knoten auf dem selektierten Pfad auf die Objekte anwenden
							//     (brauche ich dazu erst alle Objekte? Oder gibt mir
							//     der Filter vor, welche Objekte ich laden soll?
							//  -> _systemObjects auf neuen Wert setzen
							//  -> notifyTreeSelectionChanged aufrufen

							// 1. Objekte vom Pfad mit Filter holen
							TreePath tp = e.getNewLeadSelectionPath();
//				long start = System.currentTimeMillis();
							if(tp != null) {
								Object[] objects = tp.getPath();
								//_debug.finest(" Selektiert: " + tp.toString());
								for(int i = 0; i < objects.length; i++) {
									TreeNodeObject treeNodeObject = (TreeNodeObject)objects[i];
									//System.out.println("Name: " + treeNodeObject.getName() + "   Pid: " + treeNodeObject.getPid());
									//System.out.println("Knoten [" + i + "]: " + treeNodeObject.getName());
									// nehme Objektmenge und wende Filter darauf an
									Collection filters = treeNodeObject.getFilters();
									if(!filters.isEmpty()) {	   // Filter anwenden
										for(Iterator iterator = filters.iterator(); iterator.hasNext();) {
											Filter filter = (Filter)iterator.next();
											// hier wird der Filter angewandt
											_filterObjects = filter.filterObjects(_filterObjects);
										}
									}
								}
//					_debug.finest("neue Objekte bestimmen: " + (System.currentTimeMillis() - start));
								// Filter anwenden
								notifyTreeSelectionChanged();
							}
						}
						*/
                filterObjects(e.getNewLeadSelectionPath());
                notifyTreeSelectionChanged();
            }
        });
        JScrollPane treeScrollPane = new JScrollPane(_tree);
        setLayout(new BorderLayout());
        setMinimumSize(new Dimension(0, 0));
        setPreferredSize(new Dimension(200, 200));
        add(treeScrollPane, BorderLayout.CENTER);
        add(refreshPanel, BorderLayout.SOUTH);
    }

    public void updateTimeSpecCombo(final JLabel from, final JPanel fromPanel, final JPanel toPanel) {
        from.setVisible(true);
        fromPanel.setVisible(true);
        toPanel.setVisible(true);
        TimeSpecificationType selectedItem = (TimeSpecificationType) _timeSpecificationTypeJComboBox.getSelectedItem();
        if (selectedItem == null) {
            return;
        }
        switch (selectedItem) {
            case VALID:
                fromPanel.setVisible(false);
                toPanel.setVisible(false);
                break;
            case VALID_AT_TIME:
                toPanel.setVisible(false);
                from.setVisible(false);
                break;
        }
        fromPanel.revalidate();
        toPanel.revalidate();
        updateDateSelection();
    }

    private void updateDateSelection() {
        TimeSpecificationType selectedItem = (TimeSpecificationType) _timeSpecificationTypeJComboBox.getSelectedItem();
        if (selectedItem == null) {
            return;
        }
        long fromTime = ((Date) _fromAbsoluteDate.getValue()).getTime();
        long toTime = ((Date) _toAbsoluteDate.getValue()).getTime();

        switch (selectedItem) {
            case VALID:
                _currentInterval = ObjectTimeSpecification.valid();
                break;
            case VALID_AT_TIME:
                _currentInterval = ObjectTimeSpecification.valid(fromTime);
                break;
            case VALID_IN_PERIOD:
                _currentInterval = ObjectTimeSpecification.validInPeriod(fromTime, toTime);
                break;
            case VALID_DURING_PERIOD:
                _currentInterval = ObjectTimeSpecification.validDuringPeriod(fromTime, toTime);
                break;
        }
        updateUpdateButton();
    }

    private void updateSelectedObjects() {
        if (_currentInterval == ObjectTimeSpecification.valid()) {
            _selectedSystemObjects = _currentSystemObjects;
        } else {
            if (_currentInterval.getType() == TimeSpecificationType.VALID_IN_PERIOD ||
                _currentInterval.getType() == TimeSpecificationType.VALID_DURING_PERIOD) {
                if (_currentInterval.getEndTime() < _currentInterval.getStartTime()) {
                    JOptionPane
                        .showMessageDialog(this, "Ungültiges Zeitintervall. Intervallende vor Intervallanfang.", "Fehler", JOptionPane.ERROR_MESSAGE);
                    return;
                }
            }
            _selectedSystemObjects = _preselectionTreeHandler.getAllObjects(_currentInterval);
        }
        _lastInterval = _currentInterval;
    }

    private void registerDynamicObjectType() {
        final String pid = Pid.Type.DYNAMIC_OBJECT;
        final DynamicObjectType dynamicObjectType = (DynamicObjectType) _dataModel.getType(pid);
        final DynamicObjectTypeListener listener = new DynamicObjectTypeListener();
        dynamicObjectType.addInvalidationListener(listener);
        dynamicObjectType.addObjectCreationListener(listener);
    }

    /**
     * Filtert die Objekte nach der Auswahl im Auswahlbaum.
     *
     * @param tp selektierter Pfad im Auswahlbaum
     */
    private void filterObjects(final TreePath tp) {
        _lastUsedSystemObjects = _selectedSystemObjects;
        _selectedTreePath = tp;
        // einen Knoten ausgewählt:
        //  -> Filter von jedem Knoten auf dem selektierten Pfad auf die Objekte anwenden
        //     (brauche ich dazu erst alle Objekte? Oder gibt mir
        //     der Filter vor, welche Objekte ich laden soll?
        //  -> _systemObjects auf neuen Wert setzen
        //  -> notifyTreeSelectionChanged aufrufen

        // 1. Objekte vom Pfad mit Filter holen
        if (tp != null) {
            _filterObjects = _lastUsedSystemObjects;
            Object[] objects = tp.getPath();
            for (final Object object : objects) {
                TreeNodeObject treeNodeObject = (TreeNodeObject) object;
                // nehme Objektmenge und wende Filter darauf an
                final Collection<Filter> filters = treeNodeObject.getFilters();
                if (!filters.isEmpty()) {       // Filter anwenden
                    for (final Filter filter : filters) {
                        // hier wird der Filter angewandt
                        _filterObjects = filter.filterObjects(_filterObjects);
                    }
                }
            }
        } else {
            _filterObjects = new LinkedList<>();
        }
        updateUpdateButton();
    }

    private void updateUpdateButton() {
        final boolean objectsUpToDate =
            (_selectedTreePath == null) || (_selectedSystemObjects == _lastUsedSystemObjects && _currentInterval == _lastInterval);
        _updateButton.setEnabled(!objectsUpToDate);
    }

    /**
     * Ändert den aktuellen Baum in der JTree-Komponente.
     *
     * @param newModel das TreeModel, welches angezeigt werden soll
     */
    void setTreeData(TreeModel newModel) {
//		_debug.finest("setTreeData");
//		long start = System.currentTimeMillis();

        // Den alten Pfad merken
        TreePath selectionPath = _tree.getSelectionPath();
        if (selectionPath != null) {
            Object[] path = selectionPath.getPath();
            if (path.length > 1) {
                StringBuilder oldPath = new StringBuilder();
                for (int i = 1; i < path.length; i++) {
                    final Object o = path[i];
                    oldPath.append(((TreeNodeObject) o).getPid());
                    oldPath.append(",");
                }
                _treePath = oldPath.substring(0, oldPath.length() - 1);
            }
        }
        _tree.setModel(newModel);
//		_debug.finest("ModelTime: " + (System.currentTimeMillis() - start));
        _preselectionTreeHandler.initDataLists();
//		_debug.finest("initDataLists: " + (System.currentTimeMillis() - start));
        _currentSystemObjects = _preselectionTreeHandler.getAllObjects();
        _selectedSystemObjects = _currentSystemObjects;
        _filterObjects = new LinkedList<>();
        try {
            selectTreePath();
        } catch (Exception ignore) {
        }
//		_debug.finest("selectTreePath(): " + (System.currentTimeMillis() - start));
        notifyTreeSelectionChanged();
        updateUpdateButton();
//		_debug.finest("notify: " + (System.currentTimeMillis() - start));
    }

    /** Selektiert anhand des Strings _treePath (enthält kommaseparierte PIDs) den Pfad im Baum. */
    private void selectTreePath() {
        if (_treePath != null) {
            String[] paths = _treePath.split(",");
            List<TreeNodeObject> treeNodeObjects = new ArrayList<>();
            TreeNodeObject root = (TreeNodeObject) _tree.getModel().getRoot();
            if (root == null) {
                return;
            }
            treeNodeObjects.add(root);
            TreeNodeObject node = root;
            for (final String path : paths) {
                for (int i = 0, n = node.getChildCount(); i < n; i++) {
                    TreeNodeObject nodeObject = node.getChild(i);
                    if (nodeObject.getPid().equals(path)) {
                        treeNodeObjects.add(nodeObject);
                        node = nodeObject;
                        break;
                    }
                }
            }
            _tree.setSelectionPath(new TreePath(treeNodeObjects.toArray()));
        }
    }

    /**
     * Gibt die Parameter für die Vorauswahl (Baum) zurück. Die Collection enthält Systemobjekte und {@link TreeNodeObject Knotenobjekte}. Anhand der
     * Objekte wird der Baum für die Vorauswahl erzeugt.
     *
     * @return die Sammlung von System- und Knotenobjekten
     */
    public Collection<Object> getTreeNodes() {
        return Collections.unmodifiableCollection(_preselectionTreeHandler.getTreeNodes());
    }

    /**
     * Gibt den selektierten Pfad des Baums als kommaseparierten String zurück. Jedes Objekt wird durch eine PID repräsentiert.
     *
     * @return Pfad des Baums als kommaseparierten String
     */
    public String getSelectedTreePath() {
        TreePath treePath = _tree.getSelectionPath();
        String path = "";
        if (treePath != null) {
            Object[] objects = treePath.getPath();
            for (int i = 0; i < objects.length; i++) {
                TreeNodeObject treeNodeObject = (TreeNodeObject) objects[i];
                String pid = treeNodeObject.getPid();
                if ("Wurzel".equals(treeNodeObject.getName())) {
                    continue;
                }
                if (path.isEmpty() && i == 1) {
                    path += pid;
                } else {
                    path += "," + pid;
                }
            }
        }
        return path;
    }

    /**
     * Kommaseparierte PIDs werden als String übergeben, die einen Pfad im Baum des PreselectionTrees darstellen. Ist der Pfad vorhanden, dann wird er
     * selektiert.
     *
     * @param treePath Pfad des Baums als kommaseparierten String
     */
    public void setSelectedTreePath(final String treePath) {
//		_debug.finest("treePath = " + treePath);
        _treePath = treePath;
        if (treePath != null && !treePath.isEmpty()) {
            try {
                selectTreePath();
            } catch (Exception ignore) {
            }
        } else {    // Selektion aufheben
            _tree.clearSelection();
        }
    }

    /**
     * Fügt einen {@code PreselectionTreeListener} hinzu.
     *
     * @param listener ein Objekt, welches den Listener implementiert
     */
    public void addPreselectionListener(PreselectionTreeListener listener) {
        _listenerList.add(listener);
    }

    /**
     * Entfernt einen {@code PreselectionTreeListener}.
     *
     * @param listener ein Objekt, welches den Listener implementiert
     */
    public void removePreselectionListener(PreselectionTreeListener listener) {
        _listenerList.remove(listener);
    }

    /**
     * Gibt dem Listener-Objekt bekannt, ob ein Koten im Baum angewählt wurde. Die gefilterten Systemobjekte werden dann an das Listener-Objekt
     * übergeben.
     */
    private void notifyTreeSelectionChanged() {
        final Collection<SystemObject> unmodifiableCollection = Collections.unmodifiableCollection(_filterObjects);
        for (PreselectionTreeListener preselectionTreeListener : _listenerList) {
            preselectionTreeListener.setObjects(unmodifiableCollection);
        }
    }

    private final class DynamicObjectTypeListener implements InvalidationListener, DynamicObjectType.DynamicObjectCreatedListener {

        public DynamicObjectTypeListener() {
        }

        public void invalidObject(DynamicObject dynamicObject) {
            // Objekt entfernen
            if (_currentSystemObjects != null) {
                Set<SystemObject> allSystemObjects = new HashSet<>(_currentSystemObjects);
                allSystemObjects.remove(dynamicObject);
                _currentSystemObjects = allSystemObjects;
                updateSelectedObjects();
            }
            updateUpdateButton();
        }

        public void objectCreated(DynamicObject createdObject) {
            // Objekt hinzufügen
            if (_currentSystemObjects != null) {
                final List<SystemObject> newList = new ArrayList<>(_currentSystemObjects.size() + 1);
                newList.addAll(_currentSystemObjects);
                newList.add(createdObject);
                _currentSystemObjects = newList;
                updateSelectedObjects();
            }
            updateUpdateButton();
        }
    }
}
