/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.kappich.
 *
 * de.bsvrz.sys.funclib.kappich is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.kappich is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.kappich; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.kappich.collections;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.SetMultimap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Diese Klasse erlaubt es, eine Guava {@link Multimap} dort zu verwenden, wo eine {@link MultiValueMap} erwartet wird.
 *
 * @author Kappich Systemberatung
 */
public final class MultimapAdapter<K, V> implements MultiValueMap<K, V> {

    private final Multimap<K, V> _map;

    public MultimapAdapter(Multimap<K, V> map) {
        _map = map;
    }

    private MultimapAdapter(MultiValueMap<? extends K, ? extends V> map) {
        if (map.isUsingUniqueValues()) {
            _map = HashMultimap.create();
        } else {
            _map = ArrayListMultimap.create();
        }
        addAll(map);
    }

    public static <K, V> MultimapAdapter<K, V> copyOf(MultiValueMap<K, V> map) {
        return new MultimapAdapter<>(map);
    }

    @Override
    public int size() {
        return _map.size();
    }

    @Override
    public boolean isEmpty() {
        return _map.isEmpty();
    }

    @Override
    public boolean contains(final Object o) {
	    if (o instanceof Map.Entry<?, ?> entry) {
            return _map.containsEntry(entry.getKey(), entry.getValue());
        }
        return false;
    }

    @Override
    public boolean add(final Map.Entry<K, V> entry) {
        return _map.put(entry.getKey(), entry.getValue());
    }

    @Override
    public boolean containsKey(final Object key) {
        return _map.containsKey(key);
    }

    @Override
    public boolean containsValue(final Object value) {
        return _map.containsValue(value);
    }

    @Override
    public Collection<V> get(final K key) {
        return _map.get(key);
    }

    @Override
    public boolean add(final K key, final V value) {
        return _map.put(key, value);
    }

    @Override
    public boolean addAll(final K key, final Collection<? extends V> value) {
        return _map.putAll(key, value);
    }

    @Override
    public boolean remove(final Object key, final Object value) {
        return _map.remove(key, value);
    }

    @Override
    public Collection<V> removeAll(final Object key) {
        return _map.removeAll(key);
    }

    @Override
    public void clear() {
        _map.clear();
    }

    @Override
    public Set<K> keySet() {
        return _map.keySet();
    }

    @Override
    public List<V> values() {
        return new ArrayList<>(_map.values());
    }

    @Override
    public Set<V> valueSet() {
        return new HashSet<>(_map.values());
    }

    @Override
    public Set<Map.Entry<K, Collection<V>>> entrySet() {
        return _map.asMap().entrySet();
    }

    @Override
    public Set<K> findKey(final V value) {
        return _map.entries().stream().filter(it -> Objects.equals(it.getValue(), value)).map(Map.Entry::getKey).collect(Collectors.toSet());
    }

    @Override
    public Set<K> findKeys(final Collection<? extends V> values) {
        return _map.asMap().entrySet().stream().filter(it -> it.getValue().stream().anyMatch(values::contains)).map(Map.Entry::getKey)
            .collect(Collectors.toSet());
    }

    @Override
    public void addAll(final Map<? extends K, Collection<? extends V>> map) {
        for (Map.Entry<? extends K, Collection<? extends V>> entry : map.entrySet()) {
            _map.putAll(entry.getKey(), entry.getValue());
        }
    }

    @Override
    public void addAll(final MultiValueMap<? extends K, ? extends V> multiValueMap) {
        for (Map.Entry<? extends K, ? extends Collection<? extends V>> entry : multiValueMap.entrySet()) {
            _map.putAll(entry.getKey(), entry.getValue());
        }
    }

    @Override
    public boolean isUsingUniqueValues() {
        return _map instanceof SetMultimap;
    }

    /**
     * Gibt eine View auf die dahinterliegende Multimap zurück.
     *
     * @return Multimap
     */
    public Multimap<K, V> asMultimap() {
        return _map;
    }

    /**
     * Gibt eine View auf die dahinterliegende Map zurück. Entspricht {@link #asMultimap()}.{@link Multimap#asMap() asMap()}.
     *
     * @return Map-View
     */
    public Map<K, Collection<V>> asMap() {
        return _map.asMap();
    }
}
