/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.kappich.
 *
 * de.bsvrz.sys.funclib.kappich is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.kappich is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.kappich; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.kappich.collections;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.swing.ComboBoxModel;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

/**
 * Ein einfaches ListModel, das eine Menge von Elementen darstellt und diese Objekte nach einem Comparator sortiert. Der Comparator wird im
 * Konstruktor übergeben.
 * <p>
 * Das ListModel unterstützt außerdem eine schnelle indexOf()-Implementierung.
 * <p>
 * Wenn als {@link Comparator} ein {@link CollatorComparator} angegeben wird, wird eine optimierte Sortierung mit {@link java.text.CollationKey}s
 * verwendet.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("ALL")
public class SortedListModel<E> implements ComboBoxModel<E> {

    private final CopyOnWriteArrayList<ListDataListener> _listeners = new CopyOnWriteArrayList<>();

    private Object _selectedItem;

    private Object[] _data = new Object[0];

    private Comparator<? super E> _comparator;

    private Collection<? extends E> _elements;

    public SortedListModel(final Comparator<? super E> comparator) {
        _comparator = comparator;
    }

    private void fireContentsChanged(final int type, final int index0, final int index1) {
        if (_listeners.isEmpty()) {
            return;
        }
        ListDataEvent e = new ListDataEvent(this, type, index0, index1);
        for (ListDataListener listener : _listeners) {
            listener.contentsChanged(e);
        }
    }

    @Override
    public Object getSelectedItem() {
        return _selectedItem;
    }

    @Override
    public void setSelectedItem(final Object anItem) {
        _selectedItem = anItem;
        fireContentsChanged(ListDataEvent.CONTENTS_CHANGED, -1, -1);
    }

    @Override
    public int getSize() {
        if (_data == null) {
            return _elements.size();
        }
        return _data.length;
    }

    @Override
    public E getElementAt(final int index) {
        lazyInit();
        return (E) _data[index];
    }

    private void lazyInit() {
        if (_data == null) {
            if (_comparator instanceof CollatorComparator) {
                CollatorComparator comparator = (CollatorComparator<E>) _comparator;
//				_data = _elements.parallelStream().map(it -> comparator.getSortKey(it)).sorted().map(it -> it.getObject()).toArray();
                final List<CollatorComparator.SortKey> tmp = new ArrayList<>(_elements.size());
                for (E element : _elements) {
                    tmp.add(comparator.getSortKey(element));
                }
                Collections.sort(tmp);
                _data = new Object[tmp.size()];
                for (int i = 0; i < _data.length; i++) {
                    _data[i] = tmp.get(i).getObject();
                }
            } else {
//				_data = _elements.parallelStream().sorted(_comparator).toArray();
                _data = _elements.toArray();
                Arrays.sort((E[]) _data, _comparator);
            }
            _elements = null;
        }
    }

    public void setElements(Collection<? extends E> elements) {
        int oldLength = _data.length;
        _data = new Object[0];
        fireContentsChanged(ListDataEvent.INTERVAL_REMOVED, 0, oldLength);
        _elements = elements;
        _data = null;
        fireContentsChanged(ListDataEvent.INTERVAL_ADDED, 0, _elements.size());
    }

    public int indexOf(E element) {
        if (element == null) {
            return -1;
        }

        lazyInit();

        // Binäre Suche
        int i = Arrays.binarySearch((E[]) _data, element, _comparator);

        // Im primitiven Fall könnte man i zurückgegeben, aber es könnte mehrere Werte geben, die laut dem Collator gleich sind,
        // aber nicht per equals, und wir wollen den Index des richtigen Elements zurückgeben.

        if (i < 0) {
            return -1;
        }
        for (int f = i; f >= 0; f--) {
            // Nach vorne suchen, der Fall dass i bereits richtig ist wird gleich mitbehandelt
            if (element.equals(_data[f])) {
                return f;
            }
            if (_comparator.compare((E) _data[f], element) != 0) {
                // Der Collator meldet einen Unterschied, wir sind also aus dem Bereich der in Frage kommenden Objekte auf jeden Fall raus)
                break;
            }
        }
        for (int f = i + 1; f < _data.length; f++) {
            // Nach hinten suchen
            if (_comparator.compare((E) _data[f], element) != 0) {
                // Der Collator meldet einen Unterschied, wir sind also aus dem Bereich der in Frage kommenden Objekte auf jeden Fall raus)
                break;
            }
            if (element.equals(_data[f])) {
                return f;
            }
        }
        return i;
    }

    @Override
    public void addListDataListener(final ListDataListener l) {
        _listeners.add(l);
    }

    @Override
    public void removeListDataListener(final ListDataListener l) {
        _listeners.remove(l);
    }

    public E get(final int index) {
        lazyInit();

        return (E) _data[index];
    }

    public Collection<E> getElementsUnsorted() {
        if (_data == null) {
            return Collections.unmodifiableCollection(_elements);
        } else {
            return Arrays.asList((E[]) _data);
        }
    }
}
