/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.kappich.
 *
 * de.bsvrz.sys.funclib.kappich is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.kappich is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.kappich; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.kappich.onlinehelp;

import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.kappich.filechooser.AwtFileChooser;
import java.awt.Component;
import java.awt.Toolkit;
import java.awt.event.KeyEvent;
import java.awt.image.BufferedImage;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.URL;
import java.net.URLConnection;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.imageio.ImageIO;
import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.KeyStroke;
import javax.swing.filechooser.FileNameExtensionFilter;

/**
 * Eine Klasse zur Erstellung eines Hilfemenüs.
 *
 * @author Kappich Systemberatung
 */
public final class HelpMenu {

    private static final Debug _debug = Debug.getLogger();

    private static final Pattern HTML = Pattern.compile(".html", Pattern.LITERAL);
    private static final Pattern COMPILE = Pattern.compile("<img src=\"", Pattern.LITERAL);

    // Keine Objekte dieser Klasse.
    private HelpMenu() {
    }

    /**
     * Mit Hilfe dieser Methode kann ein Hilfe-Menü erzeugt werden.
     *
     * @param help   eine {@link Help}-Objekt
     * @param parent eine Parent-{@link Component Komponente}, bzgl der die Dialoge angezeigt werden
     *
     * @return ein Hilfe-Menü
     */
    @SuppressWarnings("OverlyLongMethod")
    public static JMenu createHelpMenu(final Help help, final String resourceName, @Nullable final Component parent) {
        // Hilfe-Menue
        JMenu helpMenu = new JMenu("Hilfe");

        helpMenu.setMnemonic(KeyEvent.VK_H);
        helpMenu.getAccessibleContext().setAccessibleDescription("Das Hilfe-Menue");

        helpMenu.add(getOpenHelpItem(help));
        helpMenu.add(getSaveHelpItem(help, resourceName, parent));
        return helpMenu;
    }

    /**
     * Mit Hilfe dieser Methode kann ein {@link JMenuItem} zum Öffnen der Online-Hilfe oder Bedienungsanleitung erzeugt werden.
     *
     * @param help ein {@link Help}-Objekt
     *
     * @return ein ein {@link JMenuItem} zum Öffnen der Online-Hilfe oder Bedienungsanleitung
     */
    @SuppressWarnings("WeakerAccess")
    public static JMenuItem getOpenHelpItem(final Help help) {
        JMenuItem menuItem = new JMenuItem("Online-Hilfe öffnen");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Öffnet ein Browser-Fenster mit der Online-Hilfe");
        menuItem.setEnabled(true);
        menuItem.addActionListener(event -> help.open("#logo"));
        return menuItem;
    }

    /**
     * Mit Hilfe dieser Methode kann ein {@link JMenuItem} zum Speichern der Online-Hilfe oder Bedienungsanleitung erzeugt werden.
     *
     * @param help ein {@link Help}-Objekt
     *
     * @return ein ein {@link JMenuItem} zum Speichern der Online-Hilfe oder Bedienungsanleitung
     */
    @SuppressWarnings("WeakerAccess")
    public static JMenuItem getSaveHelpItem(final Help help, final String resourceName, @Nullable final Component parent) {
        JMenuItem menuItem = new JMenuItem("Online-Hilfe speichern");
        menuItem.getAccessibleContext().setAccessibleDescription("Speichert die Online-Hilfe in einer HTML-Datei");
        menuItem.setEnabled(true);

        menuItem.addActionListener(e -> {
            // Schritt 1: frage den Benutzer nach dem Verzeichnis, wo die Daten gesichert werden sollen
            final String path = HelpMenu.getPath(parent);
            if (path == null) {
                return;
            }
            // Schritt 2: öffne das Verzeichnis
            File directory = openDirectory(path, parent);
            if (directory == null) {
                return;
            }
            // Schritt 3: speichere die PNGs
            if (!storeAllPng(help, directory)) {
                return;
            }
            // Schritt 4: schreibe die HTML-Datei
            writeHtml(help, resourceName, parent, path, directory);
        });
        return menuItem;
    }

    @Nullable
    private static String getPath(@Nullable final Component parent) {
        final JFileChooser fileChooser;
        if (System.getProperty("os.name").toLowerCase().startsWith("mac")) {
            fileChooser = AwtFileChooser.createFileChooser();
        } else {
            fileChooser = new JFileChooser();
        }
        FileNameExtensionFilter filter = new FileNameExtensionFilter("HTML-Datei", "html", "htm");
        fileChooser.setFileFilter(filter);
        fileChooser.setDialogTitle("Online-Hilfe speichern");
        fileChooser.setApproveButtonText("Speichern");

        String path;

        while (true) {
            int showSaveDialog = fileChooser.showSaveDialog(parent);
            if (!(showSaveDialog == JFileChooser.CANCEL_OPTION)) {
                File selectedFile = fileChooser.getSelectedFile();
                path = selectedFile.getPath();
                if (!path.toLowerCase().endsWith(".html")) {
                    //noinspection StringConcatenationInLoop
                    path += ".html";
                }
                File htmlFile = new File(path);
                if (htmlFile.exists()) {
                    int n = JOptionPane
                        .showConfirmDialog(fileChooser, "Die Datei '" + htmlFile.getName() + "' existiert bereits.\nDatei überschreiben?", "Warnung",
                                           JOptionPane.YES_NO_OPTION);
                    if (n == JOptionPane.YES_OPTION) {
                        break;
                    }
                } else {
                    break;
                }
            } else {
                return null;
            }
        }
        return path;
    }

    @Nullable
    private static File openDirectory(final String path, @Nullable Component parent) {
        File directory = new File(HTML.matcher(path).replaceAll(Matcher.quoteReplacement("")) + "-Dateien");
        // Ordner anlegen mit dem Namen des Files
        if (!directory.mkdir()) {
            JOptionPane.showMessageDialog(parent, "Das Verzeichnis '" + path + "' konnte nicht angelegt werden.", "Fehlermeldung",
                                          JOptionPane.ERROR_MESSAGE);
            return null;
        }
        return directory;
    }

    private static boolean storeAllPng(final Help help, final File directory) {
        Map<String, BufferedImage> images = HelpPage.getImages(help.getClass());

        for (final Map.Entry<String, BufferedImage> stringBufferedImageEntry : images.entrySet()) {
            BufferedImage image = stringBufferedImageEntry.getValue();
            try {
                ImageIO.write(image, "png", new File(directory.getPath() + "/" + stringBufferedImageEntry.getKey()));
            } catch (IOException e) {
                _debug.error("Die Abbildungen konten nicht gespeichert werden", e);
                return false;
            }
        }
        return true;
    }

    private static void writeHtml(final Help help, final String resourceName, @Nullable final Component parent, final String path,
                                  final File directory) {
        try {
            StringBuilder completeText = new StringBuilder();
            final URL helpPage = help.getClass().getResource(resourceName);
            URLConnection urlConnection = helpPage.openConnection();
            BufferedReader reader = null;
            try {
                reader = new BufferedReader(new InputStreamReader(urlConnection.getInputStream(), "ISO-8859-1"));
                String inputLine;
                String lineSeparator = System.getProperty("line.separator");
                while ((inputLine = reader.readLine()) != null) {
                    completeText.append(inputLine).append(lineSeparator);
                }
            } catch (IOException ignore) {
                JOptionPane
                    .showMessageDialog(parent, "Das Dokument der Online-Hilfe konnte nicht geöffnet werden.", "Fehler", JOptionPane.ERROR_MESSAGE);
            } finally {
                try {
                    if (reader != null) {
                        reader.close();
                    }
                } catch (IOException e) {
                    _debug.error("Fehler beim Schließen des BufferedReaders der HTML-Datei", e);
                }
            }

            BufferedWriter bufferedWriter = null;
            try {
                File htmlFile = new File(path);
                bufferedWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(htmlFile), "ISO-8859-1"));
                bufferedWriter
                    .write(COMPILE.matcher(completeText.toString()).replaceAll(Matcher.quoteReplacement("<img src=\"" + directory.getName() + "/")));

            } catch (IOException ignore) {
                JOptionPane
                    .showMessageDialog(parent, "Das Dokument der Online-Hilfe konnte nicht gespeichert werden.", "Fehler", JOptionPane.ERROR_MESSAGE);
            } finally {
                if (bufferedWriter != null) {
                    try {
                        bufferedWriter.close();
                    } catch (IOException e1) {
                        _debug.error("Fehler beim Schließen des BufferedWriters der HTML-Datei.", e1);
                    }
                }
            }
        } catch (IOException e2) {
            _debug.error("IO-Fehler beim Schreiben der HTML-Datei.", e2);
        }
    }
}
