/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.kappich.
 *
 * de.bsvrz.sys.funclib.kappich is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.kappich is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.kappich; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.kappich.onlinehelp;

import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.awt.MediaTracker;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.net.JarURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;
import javax.imageio.ImageIO;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.event.HyperlinkEvent;
import javax.swing.text.Element;
import javax.swing.text.View;
import javax.swing.text.ViewFactory;
import javax.swing.text.html.HTMLEditorKit;
import javax.swing.text.html.ImageView;

/**
 * Das Fenster zur Darstellung der Online-Hilfe.
 *
 * @author Kappich Systemberatung
 */
public final class HelpPage {

    private static final Debug _debug = Debug.getLogger();
    private static final HTMLEditorKit.HTMLFactory FACTORY = new HTMLEditorKit.HTMLFactory();

    /**
     * <p>Diese Methode gibt eine {@link JEditorPane Textkomponente} mit {@code HTML}-Support zurück. Diese Textkomponente zeigt den Text,
     * der durch Resource {@code resourceName} der Klasse {@code resourceClass} festgelegt ist.</p>
     *
     * @param resourceClass eine Klasse
     * @param resourceName  ein Name einer Resource
     * @param href          eine Hypertext-Referenz
     */
    @Nullable
    public static JEditorPane getHelpPane(Class<?> resourceClass, final String resourceName, final String href) {
        URL url;
        try {
            URL documentationPage = resourceClass.getResource(resourceName);
            url = new URL(documentationPage, href);
        } catch (MalformedURLException e) {
            final Throwable cause = e.getCause();
            if (cause != null && cause.getMessage() != null) {
                _debug.error("HelpPage.getHelpPane(): MalformedURLException mit Ursache: " + e.getMessage());
            } else {
                _debug.error("HelpPage.getHelpPane(): MalformedURLException ohne detailliere Ursache.");
            }

            return null;
        }
        return HelpPage.getHelpPane(url);
    }

    /**
     * Liefert eine Map zurück, deren Schlüssel die Dateinamen und Werte die PNG-Dateien des lokalen Verzeichnisses sind.
     *
     * @return die oben beschriebene Map
     */
    public static Map<String, BufferedImage> getImages(Class<?> resourceClass) {
        HashMap<String, BufferedImage> images = new HashMap<>();
        final List<String> imageNames = getPngNames(resourceClass);
        if (imageNames == null) {
            return images;
        } else {
            for (String fileName : imageNames) {
                BufferedImage image;
                try {
                    image = ImageIO.read(resourceClass.getResourceAsStream(fileName));
                } catch (IOException e) {
                    throw new UnsupportedOperationException("IOException", e);
                }
                images.put(fileName, image);
            }

            final JFrame aFrame = new JFrame();
            MediaTracker mediaTracker = new MediaTracker(aFrame);
            int id = 0;
            for (BufferedImage image : images.values()) {
                mediaTracker.addImage(image, id);
                id++;
            }
            try {
                mediaTracker.waitForAll();
            } catch (InterruptedException ex) {
                ex.printStackTrace();
            }
        }
        return images;
    }

    @SuppressWarnings("unchecked")
    private static List<String> getPngNames(Class<?> resourceClass) {
        List<String> result = new ArrayList<>();
        URL url = resourceClass.getResource("");
        if (url == null) {
            return result;
        }
        File directory;
        try {
            directory = new File(url.toURI());
        } catch (Exception ignore) {
            directory = null;
        }
        if ((directory != null) && directory.exists()) {
            String[] files = directory.list();
            if (null != files) {
                for (final String file : files) {
                    if (file.endsWith(".png")) {
                        result.add(file);
                    }
                }
            }
        } else {
            try {
                final URLConnection connection = url.openConnection();
                if (connection == null) {
                    _debug.error("HelpPage.getPngNames(): URL.openConnection() scheiterte für " + url.getPath() + ".");
                    return result;
                }
	            if (!(connection instanceof JarURLConnection jarConnection)) {
                    _debug.error("HelpPage.getPngNames(): URL.openConnection() lieferte keine JarURLConnection.");
                    return result;
                }
                String starts = jarConnection.getEntryName();
                if (starts == null) {
                    _debug.error("HelpPage.getPngNames(): die JarURLConnection lieferte keinen Entry-Namen.");
                    return result;
                }
                JarFile jarFile = jarConnection.getJarFile();
                if (jarFile == null) {
                    _debug.error("HelpPage.getPngNames(): die JarURLConnection lieferte kein JarFile.");
                    return result;
                }
                Enumeration<JarEntry> e = jarFile.entries();
                if (e == null) {
                    _debug.error("HelpPage.getPngNames(): das JarFile lieferte kein Entries.");
                    return result;
                }
                while (e.hasMoreElements()) {
                    ZipEntry entry = e.nextElement();
                    String entryName = entry.getName();
                    if (entryName.startsWith(starts) && (entryName.lastIndexOf('/') <= starts.length()) && entryName.endsWith(".png")) {
                        if (entryName.contains("/")) {
                            entryName = entryName.substring(entryName.lastIndexOf('/') + 1);
                        }
                        result.add(entryName);
                    }
                }
            } catch (IOException ioex) {
                ioex.printStackTrace();
            }
        }
        return result;
    }

    private static JEditorPane getHelpPane(final URL url) {
        final JEditorPane helpPane = new JEditorPane();
        helpPane.putClientProperty(JEditorPane.HONOR_DISPLAY_PROPERTIES, Boolean.TRUE); // Damit die Fontgröße änderbar wird.
        helpPane.putClientProperty("charset", "UTF-8");
        // Gemäß http://stackoverflow.com/questions/8325109/java-jtextpane-html-editor-utf-8-characters-encoding kann
        // man das das charset NICHT umstellen, d.h. helpPane.setContentType("text/html;charset=UTF-8"); geht nicht
        helpPane.setEditorKit(new MyEditorKit());
        try {
            helpPane.setPage(url);
        } catch (IOException e1) {
            throw new UnsupportedOperationException("IOException", e1);
        }
        helpPane.setEditable(false);

        helpPane.addHyperlinkListener(event -> {
            if (event.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
                try {
                    helpPane.setPage(event.getURL());
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        });
        helpPane.fireHyperlinkUpdate(new HyperlinkEvent(helpPane, HyperlinkEvent.EventType.ACTIVATED, url));
        return helpPane;
    }

    @SuppressWarnings("serial")
    private static class MyEditorKit extends HTMLEditorKit {
        MyEditorKit() {
            super();
        }

        @Override
        public ViewFactory getViewFactory() {
            return new MyViewFactory();
        }
    }

    private static class MyViewFactory extends HTMLEditorKit.HTMLFactory {

        MyViewFactory() {
            super();
        }

        @Override
        public View create(Element elem) {
            View view = FACTORY.create(elem);
            if (view instanceof ImageView) {
                return new MyImageView(elem);
            }
            return view;
        }
    }

    private static class MyImageView extends ImageView {

        public MyImageView(Element elem) {
            super(elem);
            setLoadsSynchronously(true);
        }
    }
}
