/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.kappich.
 *
 * de.bsvrz.sys.funclib.kappich is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.kappich is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.kappich; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.kappich.onlinehelp;

import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.prefs.Preferences;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultHighlighter.DefaultHighlightPainter;
import javax.swing.text.Document;
import javax.swing.text.Highlighter;
import javax.swing.text.JTextComponent;

/**
 * <p>Ein {@code SearchFrame} ist ein {@link JFrame} mit einem {@link BorderLayout}. In dessen Zentrum befindet sich die dem
 * Konstruktor übergebene {@link JTextComponent}.</p>
 * <p>SearchFrame bietet die Möglichkeit mit der üblichen Tastenkombination Control-/Meta-F ein Suchpanel unten (d.h. BorderLayout.PAG_END)
 * einzublenden.</p>
 * <p>SearchFrame hat eine {@link JMenuBar Menüleiste} mit einem {@link JMenu Menü} mit einem einzigem {@link JMenu Submenü}, das
 * es erlaubt die Schriftgröße zu ändern. Wurde dem Konstruktor ein {@link Preferences}-Objekt übergaben, so wird eine vom Benutzer eingestellte
 * Schriftgröße in diesem Knoten unter dem Schlüssel {@code TextSize} gespeichert.</p>
 *
 * @author Kappich Systemberatung
 */
public class SearchFrame extends JFrame implements KeyListener, MouseListener {

    private static final String SEARCH_FRAME = "SearchFrame";
    private static final String TEXT_SIZE = "TextSize";
    private static final int TEXT_SIZE_LOWEST_VALUE = 10;
    private static final int TEXT_SIZE_HIGHEST_VALUE = 32;
    private static final int DEFAULT_TEXT_SIZE = 16;
    private final JTextComponent _textComponent;
    private final DefaultHighlightPainter _singleHighlightPainter;
    private final DefaultHighlightPainter _allHighlightPainter;
    private final String _searchPanelPosition;
    private final JPanel _outerPanel;
    private final Preferences _preferences;
    private boolean _searchPanelShown;
    private SearchPanel _searchPanel;

    /**
     * <p>Ein {@code SearchFrame} ist ein {@link JFrame} mit einem {@link BorderLayout}. In dessen Zentrum befindet sich die dem
     * Konstruktor übergebene {@link JTextComponent}.</p>
     * <p>SearchFrame bietet die Möglichkeit mit der üblichen Tastenkombination Control-/Meta-F ein Suchpanel unten (d.h. BorderLayout.PAG_END)
     * einzublenden.</p>
     * <p>SearchFrame hat eine {@link JMenuBar Menüleiste} mit einem {@link JMenu Menü} mit einem einzigem {@link JMenu Submenü}, das
     * es erlaubt die Schriftgröße zu ändern. Wurde dem Konstruktor ein {@link Preferences}-Objekt übergaben, so wird eine vom Benutzer eingestellte
     * Schriftgröße in diesem Knoten unter dem Schlüssel {@code TextSize} gespeichert.</p>
     *
     * @param textComponent die Textkomponente, in der die Suchfunktionalität durchgeführt wird
     * @param preferences   ein Preferences-Objekt zum Abspeichern der Schriftgröße
     */

    public SearchFrame(@NotNull final JTextComponent textComponent, @Nullable final Preferences preferences) {
        super();
        _textComponent = textComponent;
        _preferences = preferences;
        final Font f = _textComponent.getFont();
        _textComponent.setFont(new Font(f.getName(), f.getStyle(), DEFAULT_TEXT_SIZE));
        readPreferences();
        createMenuBar();
        // Die Verwendung der Farbe "textHighlight" hat Vor- und Nachteile: ein Vorteit ist, dass diese Farbe dem
        // Benutzer bekannt ist; ein Nachteil ist, dass sie als Highlight-Farbe benutzt wird, während (zumindest
        // unter Windows) dieselbe Farbe vom Benutzer zur Selektion benutzt wird. Besonders Ärgerlich ist letzteres,
        // wenn ein Highlight-Bereich sich mit der Selektion überschneidet.
//		UIManager uiManager = new UIManager();
//		Color singleHighlightColor = (Color) UIManager.getDefaults().get("textHighlight");
//		if(singleHighlightColor == null) {
//			singleHighlightColor = Color.YELLOW;
//		}
        // Stattdessen werden klar zu unterscheidende, freundliche Farben bevorzugt.
        Color singleHighlightColor = Color.GREEN;
        _singleHighlightPainter = new DefaultHighlightPainter(singleHighlightColor);

        Color allHighlightColor = Color.YELLOW;
        _allHighlightPainter = new DefaultHighlightPainter(allHighlightColor);

        _searchPanelShown = false;
        _searchPanelPosition = BorderLayout.PAGE_END;
        _outerPanel = new JPanel();
        _outerPanel.setLayout(new BorderLayout());
        _outerPanel.add(new JScrollPane(textComponent), BorderLayout.CENTER);
        add(_outerPanel);
        _textComponent.addKeyListener(this);
        _textComponent.addMouseListener(this);
    }

    private void createMenuBar() {
        JMenuBar menuBar = new JMenuBar();
        setJMenuBar(menuBar);

        JMenu menu = new JMenu("Schrift");
        menuBar.add(menu);

        JMenu subMenu = new JMenu("Schriftgröße");
        menu.add(subMenu);

        ButtonGroup bg = new ButtonGroup();

        int currentFontSize = _textComponent.getFont().getSize();
        for (int i = TEXT_SIZE_LOWEST_VALUE; i <= TEXT_SIZE_HIGHEST_VALUE; i += 2) {
            final int finalI = i;
            JCheckBoxMenuItem menuItem = new JCheckBoxMenuItem(Integer.toString(i) + " pt");
            bg.add(menuItem);
            menuItem.addActionListener(e -> {
                setTextSize(finalI);
                putPreferences();
            });
            if (finalI == currentFontSize) {
                menuItem.setSelected(true);
            }
            subMenu.add(menuItem);
        }
    }

    private void setTextSize(final int size) {
        Font componentsFont = _textComponent.getFont();
        Font newComponentsFont = new Font(componentsFont.getName(), componentsFont.getStyle(), size);
        _textComponent.setFont(newComponentsFont);
        if (_searchPanel != null) {
            _searchPanel.setFontSize(size);
            // geht auch ohne:
            _searchPanel.revalidate();
            _searchPanel.repaint();
        }
        // geht auch ohne:
        revalidate();
        repaint();
    }

    private void showSearchPanel() {
        boolean selectSearch = false;
        if (!_searchPanelShown) {
            if (_searchPanel == null) {
                _searchPanel = new SearchPanel(this);
                _searchPanel.setFontSize(_textComponent.getFont().getSize());
            }
            _outerPanel.add(_searchPanel, _searchPanelPosition);
            _outerPanel.revalidate();
            _searchPanelShown = true;
            selectSearch = true;
        }
        _searchPanel.suggestSearch(_textComponent.getSelectedText(), _textComponent.getCaretPosition(), selectSearch);
    }

    /**
     * Blendet das Such-Panel aus.
     */
    void hideSearchPanel() {
        if (_searchPanelShown) {
            if (_searchPanel != null) {
                _outerPanel.remove(_searchPanel);
                _outerPanel.revalidate();
                revalidate();
            }
            _searchPanelShown = false;
        }
    }

    /**
     * @return das Dokument der JTextComponent, mit der das SearchFrame-Objekt kreiert wurde; {@code null}, falls die JTextComponent selbst {@code
     *     null} ist
     */
    @Nullable
    Document getDocument() {
        if (_textComponent == null) {
            return null;
        } else {
            return _textComponent.getDocument();
        }
    }

    /**
     * Bewirkt eine Hervorhebung eines Textes durch Änderung der Hintergrundfarbe. Eine zuvor mit diese Methode erstellte Hervorhebung wird gelöscht.
     *
     * @param index  Beginn des hervorgehobenen Textes
     * @param length Länge des hervorgehobenen Textes
     */
    void highlightSingleText(int index, int length) {
        _textComponent.setCaretPosition(index + length);
        // Es ist geboten das Caret ans Ende des Wortes zu setzen.

        // Die folgenden zwei Zeilen gehören zu einer ersten Implementation. Da sie einen Trick und
        // dessen Erklärung enthalten, und eventuell nochmal gebraucht werden, wurden sie und ihre Erklärung
        // nicht gelöscht.
//		_textComponent.setCaretPosition(index);
//		_textComponent.setCaretPosition(index+length);
        // Die letzten beiden Zeilen bedürften einer Erklärung: setCaretPosition erfüllt zwei
        // Funktionen, nämlich das Setzen des Carets und das Scrollen, um das Caret bzw. den
        // umgebenden Text sichtbar zu machen. Steht das Caret (z.B. von einer alten Suche her)
        // bereits auf der Position auf die es soll, so unterbleibt das Scrollen. Hat der Benutzer
        // zwischenzeitlich selber gescrollt, so wird der umgebende Text möglicherweise nicht
        // angezeigt. Duch das doppelte setCaretPosition mit leicht unterschiedlichen Werten
        // wird dagegen das Scrollen erzwungen.

        removeAllHighlights(true);
        try {
            _textComponent.getHighlighter().addHighlight(index, index + length, _singleHighlightPainter);
        } catch (BadLocationException ignore) {
        }
    }

    /**
     * Bewirkt Hervorhebungen von Textstellen, die in der Liste übergeben werden. Löscht Hervorhebungen, die zuvor mit dieser Methode erstellt
     * wurden.
     *
     * @param pairs jede Listeneintrag beschreibt beschreibt Anfang und Länge eines hevorzuhebenden Textes
     */
    void highlightAllTexts(List<Pair<Integer>> pairs) {
        removeAllHighlights(false);
        for (Pair<Integer> p : pairs) {
            try {
                _textComponent.getHighlighter().addHighlight(p.getFirst(), p.getFirst() + p.getSecond(), _allHighlightPainter);
            } catch (BadLocationException ignore) {
            }
        }
    }

    void removeAllHighlights(boolean single) {
        Highlighter.Highlight[] hls = _textComponent.getHighlighter().getHighlights();
        if (single) {
            for (final Highlighter.Highlight hl : hls) {
                if (Objects.equals(hl.getPainter(), _singleHighlightPainter)) {
                    _textComponent.getHighlighter().removeHighlight(hl);
                    return;
                }
            }
        } else {
            for (final Highlighter.Highlight hl : hls) {
                if (Objects.equals(hl.getPainter(), _allHighlightPainter)) {
                    _textComponent.getHighlighter().removeHighlight(hl);
                }
            }
        }
    }

    /*
     * Implementation KeyListener
     */

    @Override
    public void keyTyped(final KeyEvent e) {
    }

    @Override
    public void keyPressed(final KeyEvent e) {
    }

    @Override
    public void keyReleased(final KeyEvent e) {
        if ((e.isMetaDown() || e.isControlDown()) && e.getKeyCode() == KeyEvent.VK_F) {
            showSearchPanel();
            _searchPanel.requestFocusForSearchField();
        }
    }

    /*
     * Implementation MouseListener
     */

    @Override
    public void mouseClicked(final MouseEvent e) {
        removeAllHighlights(true);
        removeAllHighlights(false);
        if (_searchPanel != null) {
            _searchPanel.setStartPosition(_textComponent.getCaretPosition());
        }
    }

    @Override
    public void mousePressed(final MouseEvent e) {
    }

    @Override
    public void mouseReleased(final MouseEvent e) {
    }

    @Override
    public void mouseEntered(final MouseEvent e) {
    }

    @Override
    public void mouseExited(final MouseEvent e) {
    }

    @Override
    public String toString() {
        return "SearchFrame{}";
    }

    /* Präferenzen */

    private void putPreferences() {
        if (_preferences != null) {
            Preferences myPrefs = _preferences.node(SEARCH_FRAME);
            myPrefs.putInt(TEXT_SIZE, _textComponent.getFont().getSize());
        }
    }

    private void readPreferences() {
        if (_preferences != null) {
            Preferences myPrefs = _preferences.node(SEARCH_FRAME);
            int textSize = myPrefs.getInt(TEXT_SIZE, 16);
            if (textSize >= TEXT_SIZE_LOWEST_VALUE && textSize <= TEXT_SIZE_HIGHEST_VALUE) {
                setTextSize(textSize);
            }
        }
    }
}

/**
 * {@code SearchPanel} ist eine Klasse, die die Funktionalität des Such-Panels im Package {@code textSearch} beinhaltet.
 */
class SearchPanel extends JPanel implements DocumentListener {

    private final SearchFrame _searchFrame;
    private final Color _noMatch;
    private JTextField _textField;
    private JButton _upButton;
    private JButton _downButton;
    private JCheckBox _highlightAllButton;
    private JCheckBox _caseSensiteveButton;
    private JButton _closeButton;
    private boolean _highlightAll;
    private boolean _caseSensitive;
    private boolean _searchDownwards;
    private int _index;     // markiert die Stelle im Text, wo das letzte Suchergebnis beginnt
    private String _oldSearch; // wonach zuletzt gesucht wurde

    /**
     * @param searchFrame das {@code SearchFrame}-Objekt, zu dem das {@code SearchPanel} gehört
     */
    public SearchPanel(SearchFrame searchFrame) {
        _searchFrame = searchFrame;
        _highlightAll = false;
        _caseSensitive = false;
        _searchDownwards = true;
        _index = 0;
        _oldSearch = "";
        initComponents();
        createLayout();
        initLogic();
        _noMatch = new Color(255, 153, 153);
        // ein moderater Rot-Ton statt der Defaultfarbe des L&F.
//		UIManager uiManager = new UIManager();
//		_noMatch = (Color) UIManager.getDefaults().get("info"); "info" ist keine gute Wahl,
//		da sie (zumindest unter Windows) nicht von der Farbe für Selektion zu unterscheiden ist.
//		Dasselbe gilt für "textInactiveText".
        _textField.getDocument().addDocumentListener(this);
        // Und nun all Komponenten auf Control-F einschwören:
        _caseSensiteveButton.addKeyListener(searchFrame);
        _upButton.addKeyListener(searchFrame);
        _downButton.addKeyListener(searchFrame);
        _caseSensiteveButton.addKeyListener(searchFrame);
        _highlightAllButton.addKeyListener(searchFrame);
        // TODO Ersetzte den KeyListener durch Key Bindings.
    }

    private static void setFontSize(Component component, final int size) {
        Font font = component.getFont();
        Font newFont = new Font(font.getName(), font.getStyle(), size);
        component.setFont(newFont);
    }

    public void requestFocusForSearchField() {
        _textField.requestFocusInWindow();
    }

    private void initComponents() {
        _textField = new JTextField(20);
        _upButton = new JButton("<");
        _downButton = new JButton(">");
        _highlightAllButton = new JCheckBox("Hervorheben", _highlightAll);
        _caseSensiteveButton = new JCheckBox("Klein-/Großschreibung", _caseSensitive);
        _closeButton = new JButton("X");
    }

    private void createLayout() {
        setLayout(new BorderLayout());
        JPanel innerWestPanel = new JPanel();
        innerWestPanel.setLayout(new BoxLayout(innerWestPanel, BoxLayout.X_AXIS));
        innerWestPanel.add(_textField);
        int interButtonWidth = 4;
        innerWestPanel.add(Box.createRigidArea(new Dimension(interButtonWidth, 0)));
        innerWestPanel.add(_upButton);
        innerWestPanel.add(Box.createRigidArea(new Dimension(interButtonWidth, 0)));
        innerWestPanel.add(_downButton);
        innerWestPanel.add(Box.createRigidArea(new Dimension(2 * interButtonWidth, 0)));
        innerWestPanel.add(_highlightAllButton);
        innerWestPanel.add(Box.createRigidArea(new Dimension(interButtonWidth, 0)));
        innerWestPanel.add(_caseSensiteveButton);
        JPanel innerEastPanel = new JPanel();
        innerEastPanel.add(_closeButton);
        innerEastPanel.setLayout(new BoxLayout(innerEastPanel, BoxLayout.X_AXIS));

        add(innerWestPanel, BorderLayout.WEST);
        add(innerEastPanel, BorderLayout.EAST);
    }

    private void initLogic() {
        initTextField();
        initUpButton();
        initDownButon();
        initHighlightAllButton();
        initCaseSensitiveButton();
        initCloseButon();
    }

    private void findNext(String search, boolean ignorePresent, boolean downwards) {
        if (search == null || search.isEmpty()) {
            _oldSearch = "";
            _searchDownwards = true;
            _searchFrame.removeAllHighlights(true);
            _searchFrame.removeAllHighlights(false);
            return;
        }
        int index = _index;
        if (ignorePresent) {
            if (downwards) {
                ++index;
            } else {
                --index;
            }
        }
        boolean found = findNext(search, index, downwards);
        if (!found) {
            if (downwards) {
                found = findNext(search, 0, true);
            } else {
                Document document = _searchFrame.getDocument();
                if (null != document) {
                    found = findNext(search, document.getLength() - search.length(), false);
                }
            }
            if (!found) {
                _textField.setBackground(_noMatch);
            }
        }
        if (found) {
            _oldSearch = search;
            _searchFrame.highlightSingleText(_index, search.length());
        } else {
            _searchFrame.removeAllHighlights(true);
        }
    }

    private boolean findNext(String search, int index, boolean downwards) {
        if (search == null || search.isEmpty()) {
            return false;
        }
        String searchStr;
        if (_caseSensitive) {
            searchStr = search;
        } else {
            searchStr = search.toLowerCase();
        }
        Document document = _searchFrame.getDocument();
        if (null != document) {
            if (downwards) {
                for (int i = index; i + searchStr.length() < document.getLength(); ++i) {
                    String match = "";
                    try {
                        match = document.getText(i, searchStr.length());
                    } catch (BadLocationException ignore) {
                    }
                    if ((_caseSensitive && match.equals(searchStr)) || (!_caseSensitive && match.toLowerCase().equals(searchStr))) {
                        _index = i;
                        _textField.setBackground(Color.WHITE);
                        return true;
                    }
                }
            } else {
                for (int i = index; i >= 0; --i) {
                    String match = "";
                    try {
                        match = document.getText(i, searchStr.length());
                    } catch (BadLocationException ignore) {
                    }
                    if ((_caseSensitive && match.equals(searchStr)) || (!_caseSensitive && match.toLowerCase().equals(searchStr))) {
                        _index = i;
                        _textField.setBackground(Color.WHITE);
                        return true;
                    }
                }
            }
        }
        return false;
    }

    private List<Pair<Integer>> findAll() {
        String searchStr = _textField.getText();
        if (searchStr == null || searchStr.isEmpty()) {
            return new ArrayList<>();
        }
        if (!_caseSensitive) {
            searchStr = searchStr.toLowerCase();
        }
        List<Pair<Integer>> retList = new ArrayList<>();
        Document document = _searchFrame.getDocument();
        if (null != document) {
            for (int i = 0; i + searchStr.length() < document.getLength(); ++i) {
                String match = "";
                try {
                    match = document.getText(i, searchStr.length());
                } catch (BadLocationException ignore) {
                }
                if ((_caseSensitive && match.equals(searchStr)) || (!_caseSensitive && match.toLowerCase().equals(searchStr))) {
                    if (i != _index) {
                        Pair<Integer> pair = new Pair<>(i, searchStr.length());
                        retList.add(pair);
                    }
                }
            }
        }
        return retList;
    }

    private void initTextField() {
        _textField.addActionListener(e -> {
            findNext(_oldSearch, true, _searchDownwards);
            if (_highlightAll) {
                _searchFrame.highlightAllTexts(findAll());
            }
        });
    }

    private void initUpButton() {
        _upButton.addActionListener(e -> {
            _searchDownwards = false;
            findNext(_oldSearch, true, false);
            if (_highlightAll) {
                _searchFrame.highlightAllTexts(findAll());
            }
        });
    }

    private void initDownButon() {
        _downButton.addActionListener(e -> {
            _searchDownwards = true;
            findNext(_oldSearch, true, true);
            if (_highlightAll) {
                _searchFrame.highlightAllTexts(findAll());
            }
        });
    }

    private void initHighlightAllButton() {
        _highlightAllButton.addActionListener(e -> {
            _highlightAll = _highlightAllButton.isSelected();
            _searchFrame.removeAllHighlights(false);
            if (_highlightAll) {
                _searchFrame.highlightAllTexts(findAll());
            }
        });

    }

    private void initCaseSensitiveButton() {
        _caseSensiteveButton.addActionListener(e -> {
            _caseSensitive = _caseSensiteveButton.isSelected();
            findNext(_textField.getText(), false, true);
            if (_highlightAll) {
                _searchFrame.highlightAllTexts(findAll());
            }
        });
    }

    private void initCloseButon() {
        _closeButton.addActionListener(e -> {
            _searchFrame.hideSearchPanel();
            _searchFrame.removeAllHighlights(true);
            _searchFrame.removeAllHighlights(false);
        });
    }

    public void suggestSearch(String search, int caretPosition, boolean selectSearch) {
        if (search != null && !search.isEmpty()) {
            _index = Math.max(caretPosition - search.length(), 0);
            // Diese Berechnung  -statt _index = caretPosition - ist notwendig, da das Caret
            // nach einer Benutzerselektion am Ende des selektierten Bereichs steht.
            _textField.setText(search);
        }
        if (selectSearch) {
            _textField.selectAll();
        }
    }

    /**
     * Setze die Startposition für eine neue Suche.
     *
     * @param startPosition die neue Startposition
     */
    public void setStartPosition(int startPosition) {
        _index = startPosition;
    }

    @Override
    public void insertUpdate(final DocumentEvent e) {
        update();
    }

    @Override
    public void removeUpdate(final DocumentEvent e) {
        update();
    }

    @Override
    public void changedUpdate(final DocumentEvent e) {
        update();
    }

    private void update() {
        String newSearch = _textField.getText();
        if (newSearch == null || newSearch.isEmpty()) {
            // leere Suche startet an alter Stelle, aber abwärts
            _oldSearch = "";
            _searchDownwards = true;
            _searchFrame.removeAllHighlights(true);
            _searchFrame.removeAllHighlights(false);
            _textField.setBackground(Color.WHITE);
            return;
        }
        findNext(newSearch, false, true);
        // wenn man hier versucht mit newSearch==_oldSearch zu optimieren, so muss
        // man die Hintergrundfarben im Textfeld und der Textkomponente selber setzen.
        if (_highlightAll) {
            _searchFrame.highlightAllTexts(findAll());
        }
    }

    public void setFontSize(final int size) {
        setFontSize(_textField, size);
        setFontSize(_upButton, size);
        setFontSize(_downButton, size);
        setFontSize(_highlightAllButton, size);
        setFontSize(_caseSensiteveButton, size);
        setFontSize(_closeButton, size);
    }

    @Override
    public String toString() {
        return "SearchPanel{}";
    }
}

class Pair<E> {

    private final E _first;
    private final E _second;

    Pair(E first, E second) {
        super();
        _first = first;
        _second = second;
    }

    public E getFirst() {
        return _first;
    }

    public E getSecond() {
        return _second;
    }

    @Override
    public String toString() {
        return "Pair{" + "_first=" + _first + ", _second=" + _second + '}';
    }
}
