/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.kappich.
 *
 * de.bsvrz.sys.funclib.kappich is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.kappich is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.kappich; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.kappich.properties;

import static java.lang.Math.abs;


import java.text.NumberFormat;
import java.util.Locale;

/**
 * @author Kappich Systemberatung
 */
public class DisplayUnit {

    private static final ThreadLocal<NumberFormat> FORMAT = ThreadLocal.withInitial(() -> NumberFormat.getNumberInstance(Locale.GERMANY));

    static {
        FORMAT.get().setGroupingUsed(false);
    }

    private final String _unit;
    private final int _power;
    private final double _factor;

    public DisplayUnit(String unit, int power) {
        this(unit, power, 1.0);
    }

    public DisplayUnit(String unit, int power, final double factor) {
        _factor = factor;
        if (power == 0) {
            throw new IllegalArgumentException();
        }
        _unit = unit;
        _power = power;
    }

    private static char[] toSuperScript(final int exponent) {
        char[] chars = String.valueOf(exponent).toCharArray();
        for (int i = 0; i < chars.length; i++) {
            switch (chars[i]) {
                case '0':
                    chars[i] = '\u2070';
                    break;
                case '1':
                    chars[i] = '\u00b9';
                    break;
                case '2':
                    chars[i] = '\u00b2';
                    break;
                case '3':
                    chars[i] = '\u00b3';
                    break;
                case '4':
                    chars[i] = '\u2074';
                    break;
                case '5':
                    chars[i] = '\u2075';
                    break;
                case '6':
                    chars[i] = '\u2076';
                    break;
                case '7':
                    chars[i] = '\u2077';
                    break;
                case '8':
                    chars[i] = '\u2078';
                    break;
                case '9':
                    chars[i] = '\u2079';
                    break;
                case '-':
                    chars[i] = '\u207b';
                    break;
            }
        }
        return chars;
    }

    @PropertyName(name = "Einheit", sortKey = 1)
    public String getUnit() {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append(_unit);
        appendPower(stringBuilder);
        return stringBuilder.toString();
    }

    public synchronized String format(double value) {
        if (Double.isInfinite(value) || Double.isNaN(value)) {
            return String.valueOf(value);
        }
        double powerFactor = Math.pow(1000, _power);
        value *= _factor;
        int powerMultiplier = 0;
        while (abs(value) > 10 * powerFactor) {
            powerMultiplier++;
            value /= powerFactor;
        }
        while (value != 0.0 && abs(value) * powerFactor < 100) {
            powerMultiplier--;
            value *= powerFactor;
        }
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append(FORMAT.get().format(value));
        stringBuilder.append('\u202f');
        appendUnit(powerMultiplier, stringBuilder);
        stringBuilder.append(_unit);
        appendPower(stringBuilder);
        return stringBuilder.toString();
    }

    @SuppressWarnings("OverlyLongMethod")
    private void appendUnit(final int powerMultiplier, final StringBuilder stringBuilder) {
        switch (powerMultiplier) {
            case 8:
                stringBuilder.append("Y");
                break;
            case 7:
                stringBuilder.append("Z");
                break;
            case 6:
                stringBuilder.append("E");
                break;
            case 5:
                stringBuilder.append("P");
                break;
            case 4:
                stringBuilder.append("T");
                break;
            case 3:
                stringBuilder.append("G");
                break;
            case 2:
                stringBuilder.append("M");
                break;
            case 1:
                stringBuilder.append("k");
                break;
            case 0:
                break;
            case -1:
                stringBuilder.append("m");
                break;
            case -2:
                stringBuilder.append("µ");
                break;
            case -3:
                stringBuilder.append("n");
                break;
            case -4:
                stringBuilder.append("p");
                break;
            case -5:
                stringBuilder.append("f");
                break;
            case -6:
                stringBuilder.append("a");
                break;
            case -7:
                stringBuilder.append("z");
                break;
            case -8:
                stringBuilder.append("y");
                break;
            default:
                stringBuilder.append("e").append(powerMultiplier * _power * 3);
        }
    }

    private void appendPower(final StringBuilder stringBuilder) {
        if (_power != 1) {
            stringBuilder.append(toSuperScript(_power));
        }
    }

    @Override
    public String toString() {
        return getUnit();
    }
}
