/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.kappich.
 *
 * de.bsvrz.sys.funclib.kappich is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.kappich is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.kappich; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.kappich.selectionlist;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.kappich.collections.SortedListModel;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.DefaultListCellRenderer;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.border.EtchedBorder;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionListener;

/**
 * Klasse, die eine Liste von Objekten dem Benutzer zur Auswahl anbietet. Enthält eine Überschrift mit einer Textbox, die zum Filtern der Elemente
 * benutzt werden kann.
 * <p>
 * Diese Klasse unterstützt generische Elemente vom Typ T
 *
 * @author Kappich Systemberatung
 * @version $Revision:$
 */
public class SelectionList<T> extends Box {

    /** Icon für die Schalter zum Deselektieren */
    private static final Icon DESELECT_ICON =
        new ImageIcon(SelectionList.class.getResource("/de/bsvrz/sys/funclib/kappich/selectionlist/active-close-button.png"));
    protected final JList<T> _jList;
    private final JTextField _filterTextField;
    private final MyListCellRenderer _cellRenderer;
    private final JLabel _numberOfSelectedObjects;
    private final JButton _deselectObjects;
    private final SelectionListMatcher<T> _selectionListMatcher;
    private final Comparator<T> _comparator;
    private Collection<? extends T> _preSelectedValues = Collections.emptyList();
    private Collection<? extends T> _objects = Collections.emptyList();
    private ObjectListRenderer<? super T> _listRenderer;
    /**
     * Erstellt eine neue SelectionList
     *
     * @param header               Überschrift im Singular ("z.B. Objekt")
     * @param headerPlural         Überschrift im Plural (z.B. "Objekte")
     * @param selectionListMatcher Funktion, die prüft, ob ein Filtertext auf ein Objekt zutrifft
     * @param filterStyle          Anzeige von Titelzeile/Überschrift/Filter
     * @param comparator           Sortierung der Objekte, null für keine Sortierung (dann ist die Suche/Filterung aber ineffizient)
     */
    public SelectionList(final String header, String headerPlural, final SelectionListMatcher<T> selectionListMatcher, FilterStyle filterStyle,
                         final Comparator<T> comparator) {
        super(BoxLayout.Y_AXIS);
        _selectionListMatcher = selectionListMatcher;
        if (comparator == null) {
            _comparator = (o1, o2) -> 0;
        } else {
            _comparator = comparator;
        }
        final Box headlineBox = Box.createHorizontalBox();
        // Label Anzahl der selektierten Objekte
        _numberOfSelectedObjects = new JLabel("0 / 0");
        _numberOfSelectedObjects.setBorder(new EtchedBorder());
        _numberOfSelectedObjects.setToolTipText("Anzahl der selektierten " + headerPlural);
        // Button deselektieren
        _deselectObjects = new JButton(DESELECT_ICON);
        _deselectObjects.setToolTipText("alle " + headerPlural + " deselektieren");
        _deselectObjects.setPreferredSize(new Dimension(20, 18));
        _deselectObjects.setEnabled(false);

        if (filterStyle == FilterStyle.HiddenFilter) {
            _filterTextField = new FilterTextField(header);
        } else {
            _filterTextField = new JTextField("");
        }
        Dimension maxSize = _filterTextField.getPreferredSize();
        maxSize.width = Integer.MAX_VALUE;
        _filterTextField.setMaximumSize(maxSize);

        _filterTextField.getDocument().addDocumentListener(new DocumentListener() {
            @Override
            public void insertUpdate(final DocumentEvent e) {
                updateFilter();
            }

            @Override
            public void removeUpdate(final DocumentEvent e) {
                updateFilter();
            }

            @Override
            public void changedUpdate(final DocumentEvent e) {
                updateFilter();
            }
        });

        if (filterStyle == FilterStyle.VisibleFilter) {
            headlineBox.add(new JLabel("Filter: "));
        }
        headlineBox.add(_filterTextField);
        headlineBox.add(Box.createRigidArea(new Dimension(5, 0)));
        headlineBox.add(Box.createHorizontalGlue());
        headlineBox.add(_numberOfSelectedObjects);
        headlineBox.add(Box.createRigidArea(new Dimension(5, 0)));
        headlineBox.add(_deselectObjects);
        // Liste der Objekte
        _jList = new JList<>(new SortedListModel<>(_comparator));

        _deselectObjects.addActionListener(e -> {
            _jList.clearSelection();
            _filterTextField.setText("");
        });

        _jList.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);

        _jList.addKeyListener(new KeyListener() {
            public void keyPressed(KeyEvent e) {
            }

            public void keyReleased(KeyEvent e) {
            }

            public void keyTyped(KeyEvent e) {
                _jList.ensureIndexIsVisible(_jList.getSelectedIndex());
            }
        });

        _jList.addListSelectionListener(e -> {
            if (!e.getValueIsAdjusting()) {
                updateHeader();
                _preSelectedValues = _jList.getSelectedValuesList();
            }
        });

        _cellRenderer = new MyListCellRenderer();
        _jList.setCellRenderer(_cellRenderer);

        _listRenderer = null;

        if (filterStyle == FilterStyle.NoHeader) {
            add(new JScrollPane(_jList));
        } else {
            add(headlineBox);
            add(Box.createRigidArea(new Dimension(0, 3)));
            add(new JScrollPane(_jList));
            setBorder(BorderFactory.createEmptyBorder(3, 3, 3, 3));
            setMinimumSize(new Dimension(0, 0));
        }

        _jList.addMouseMotionListener(new MouseMotionListener() {
            public void mouseDragged(MouseEvent e) {
            }

            public void mouseMoved(MouseEvent e) {
                int index = _jList.locationToIndex(e.getPoint());
                if (index >= 0) {
                    T object = _jList.getModel().getElementAt(index);
                    if (object != null) {
                        String tooltip = getTooltip(object);
                        _jList.setToolTipText(tooltip);
                    } else {
                        _jList.setToolTipText(null);
                    }
                } else {
                    _jList.setToolTipText(null);
                }
            }
        });
        _jList.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(final MouseEvent e) {
                if (SwingUtilities.isRightMouseButton(e)) {
                    int index = _jList.locationToIndex(e.getPoint());
                    if (!_jList.isSelectedIndex(index)) {
                        _jList.setSelectedIndex(index);
                    }
                }
            }
        });
    }

    public String getTooltip(final T object) {
        if (_listRenderer == null) {
            return null;
        }
        return _listRenderer.getTooltip(object);
    }

    public ObjectListRenderer<? super T> getListRenderer() {
        return _listRenderer;
    }

    public void setListRenderer(final ObjectListRenderer<? super T> listRenderer) {
        _listRenderer = listRenderer;
    }

    private void updateFilter() {
        setElements(_objects);
    }

    public void updateHeader() {
        int[] selectedIndices = _jList.getSelectedIndices();
        _numberOfSelectedObjects.setText(selectedIndices.length + " / " + _jList.getModel().getSize());
        _deselectObjects.setEnabled(selectedIndices.length > 0 || !_filterTextField.getText().isEmpty());
    }

    /**
     * Liefert alle sichtbaren selektierten Systemobjekte zurück
     *
     * @return
     */
    public List<T> getSelectedValues() {
        return _jList.getSelectedValuesList();
    }

    /**
     * Liefert alle selektierten Systemobjekte zurück, auch solche, die durch einen aktiven Filter aktuell nicht sichtbar sind
     *
     * @return Liste mit Systemobjekten
     */
    public Collection<? extends T> getPreSelectedValues() {
        return Collections.unmodifiableCollection(_preSelectedValues);
    }

    public void addListSelectionListener(final ListSelectionListener listSelectionListener) {
        _jList.addListSelectionListener(listSelectionListener);
    }

    public void clearSelection() {
        _jList.clearSelection();
    }

    public void selectElements(final Collection<? extends T> objects) {
        ListSelectionModel selectionModel = _jList.getSelectionModel();
        selectionModel.setValueIsAdjusting(true);
        selectionModel.clearSelection();
        SortedListModel<T> sortedListModel = (SortedListModel<T>) _jList.getModel();

        for (T object : objects) {
            int position = sortedListModel.indexOf(object);
            if (position >= 0) {
                _jList.addSelectionInterval(position, position);
            }
        }
        selectionModel.setValueIsAdjusting(false);
        _jList.ensureIndexIsVisible(_jList.getSelectedIndex());
        _preSelectedValues = objects;
    }

    /**
     * Gibt das Objekt, dessen Text die breiteste Darstellung hat, zurück (um die Länge der horizontalen Scrollbar zu berechnen)
     *
     * @param model
     *
     * @return Objekt, dessen Text die breiteste Darstellung hat,
     */
    @Nullable
    private T computeObjectWithLongestText(final SortedListModel<T> model) {
        DefaultListCellRenderer renderer = new DefaultListCellRenderer() {
            @Override
            public Component getListCellRendererComponent(final JList<?> list, final Object value, final int index, final boolean isSelected,
                                                          final boolean cellHasFocus) {
                T obj = (T) value;
                Component component = super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
                if (_listRenderer != null) {
                    setText(_listRenderer.getText(obj));
                    setIcon(_listRenderer.getIcon(obj));
                }
                return component;
            }
        };
        int maxWidth = -1;
        T maxT = null;
        for (T t : model.getElementsUnsorted()) {
            Component component = renderer.getListCellRendererComponent(_jList, t, 0, false, false);
            int width = component.getPreferredSize().width;
            if (width > maxWidth) {
                maxWidth = width;
                maxT = t;
            }
        }
        return maxT;
    }

    /**
     * Erzeugt aus einer Liste von Objekten ein DefaultListModel zum Anzeigen der Objekte in einer JList.
     *
     * @param list Liste, die in einer JList angezeigt werden sollen
     *
     * @return DefaultListModel, welches in einer JList angezeigt werden kann
     */
    private SortedListModel<T> makeListModel(Collection<? extends T> list) {
        SortedListModel<T> result = new SortedListModel<>(_comparator);
        String filter = _filterTextField.getText();
        final Pattern pattern;
        final long id;
        if (!filter.isEmpty()) {
            Pattern tmpPattern = null;
            try {
                if (filter.length() > 2 && filter.startsWith("/") && filter.endsWith("/")) {
                    tmpPattern = Pattern.compile(filter.substring(1, filter.length() - 1), Pattern.CASE_INSENSITIVE | Pattern.UNICODE_CASE);
                } else {
                    tmpPattern = Pattern.compile(filter, Pattern.CASE_INSENSITIVE | Pattern.UNICODE_CASE | Pattern.LITERAL);
                }
                _filterTextField.setToolTipText(null);
                _filterTextField.setForeground(null);
            } catch (PatternSyntaxException e) {
                _filterTextField.setToolTipText(e.getMessage());
                _filterTextField.setForeground(Color.red);
            }
            long tmpId = 0;
            try {
                tmpId = Long.parseLong(filter);
            } catch (NumberFormatException ignored) {
            }

            pattern = tmpPattern;
            id = tmpId;
        } else {
            pattern = null;
            id = 0;
        }
        _cellRenderer.setPattern(pattern);
        if (pattern != null || id != 0) {
            final List<T> builder = new ArrayList<>();
            list.parallelStream().filter(it -> _selectionListMatcher.matches(pattern, id, it)).forEachOrdered(builder::add);
            result.setElements(builder);
        } else {
            result.setElements(list);
        }
        return result;
    }

    public Collection<? extends T> getElements() {
        return Collections.unmodifiableCollection(_objects);
    }

    public void setElements(final Collection<? extends T> objects) {
        _objects = objects;
        SortedListModel<T> sortedListModel = makeListModel(_objects);
        Collection<? extends T> preSelectedValues = _preSelectedValues;
        _jList.setPrototypeCellValue(computeObjectWithLongestText(sortedListModel));
        _jList.getSelectionModel().setValueIsAdjusting(true);
        _jList.setModel(sortedListModel);
        selectElements(preSelectedValues);
        updateHeader();
    }

    public void setSelectionMode(final int selectionMode) {
        _jList.setSelectionMode(selectionMode);
    }

    @Override
    public String toString() {
        return "SelectionList{}";
    }

    public enum FilterStyle {
        /**
         * Der Filter ist versteckt und wird eingeblendet wenn auf die Überschrift geklickt wird
         */
        HiddenFilter,
        /**
         * Der Filter ist immer sichtbar
         */
        VisibleFilter,
        /**
         * Kein Header
         */
        NoHeader
    }

    private class MyListCellRenderer extends DefaultListCellRenderer {
        private Pattern _pattern;

        @Override
        public Component getListCellRendererComponent(final JList<?> list, final Object value, final int index, final boolean isSelected,
                                                      final boolean cellHasFocus) {
            T obj = (T) value;
            super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            if (_listRenderer != null) {
                setText(_listRenderer.getText(obj));
                setIcon(_listRenderer.getIcon(obj));
            }
            return this;
        }

        public Pattern getPattern() {
            return _pattern;
        }

        public void setPattern(@Nullable final Pattern pattern) {
            _pattern = pattern;
        }

        @Override
        public void setText(final String text) {
            super.setText(generateHtml(_pattern, text));
        }

        private String generateHtml(final Pattern pattern, final String text) {
            if (pattern == null || pattern.toString().isEmpty()) {
                return text;
            }
            StringBuilder builder = new StringBuilder(text.length() + 32);
            Matcher matcher = pattern.matcher(text);
            builder.append("<html>");
            builder.append(matcher.replaceFirst("<font color=\"red\">$0</font>"));
            return builder.toString();
        }
    }
}
