/*
 * Copyright 2004 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.dataIdentificationSettings.
 *
 * de.bsvrz.sys.funclib.dataIdentificationSettings is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.dataIdentificationSettings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.dataIdentificationSettings; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.dataIdentificationSettings;

import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Speichert Identifizierende Informationen wie Systemobjekt, Attributgruppe, Aspekt und Simulationsvariante zu einem Datensatz.
 *
 * @author Kappich Systemberatung
 */
public final class DataIdentification implements Comparable<DataIdentification> {
    private final SystemObject _object;
    private final DataDescription _dataDescription;

    /**
     * Erzeugt eine neue Datenidentifikation aus dem übergebenen Systemobjekt und der übergebenen Datenbeschreibung.
     *
     * @param object          Systemobjekt auf das sich die neue Datenidentifikation beziehen soll.
     * @param dataDescription Datenbeschreibung auf die sich die neue Datenidentifikation beziehen soll.
     */
    public DataIdentification(SystemObject object, DataDescription dataDescription) {
        _object = object;
        _dataDescription = dataDescription;
    }

    /**
     * Bestimmt das Systemobjekt dieser Datenidentifikation.
     *
     * @return Systemobjekt dieser Datenidentifikation
     */
    public SystemObject getObject() {
        return _object;
    }

    /**
     * Bestimmt die Datenbeschreibung dieser Datenidentifikation zurück. Die Datenbeschreibung enthält Attributgruppe, Aspekt und Simulationsvariante
     *
     * @return Datenbeschreibung dieser Datenidentifikation.
     */
    public DataDescription getDataDescription() {
        return _dataDescription;
    }

    /**
     * Bestimmt einen hashCode, der nur von den im Konstruktor übergebenen Werten abhängig ist. Verschiedene Objekte dieser Klasse, die das gleiche
     * Systemobjekt und die gleiche Datenbeschreibung enthalten werden als gleich angesehen und liefern den gleichen Hashcode.
     *
     * @return Hashcode dieses Objekts.
     *
     * @see #equals
     * @see #compareTo
     */
    public int hashCode() {
        return _object.hashCode() + _dataDescription.hashCode() * 37;
    }

    /**
     * Prüft, ob dieses Objekt dem übergebenen Objekt gleicht. Verschiedene Objekte dieser Klasse, die das gleiche Systemobjekt und die gleiche
     * Datenbeschreibung enthalten werden als gleich angesehen.
     *
     * @param object Das Objekt mit dem dieses Objekt verglichen werden soll.
     *
     * @return {@code true} falls die Objekte gleich sind; sonst {@code false}.
     *
     * @see #hashCode()
     * @see #compareTo
     */
    public boolean equals(Object object) {
	    if (!(object instanceof DataIdentification o)) {
            return false;
        }
        return _object.equals(o._object) && _dataDescription.equals(o._dataDescription);
    }

    @Override
    public int compareTo(DataIdentification other) {
        int result;
        result = _object.compareTo(other._object);
        if (result != 0) {
            return result;
        }
        result = _dataDescription.getAttributeGroup().compareTo(other._dataDescription.getAttributeGroup());
        if (result != 0) {
            return result;
        }
        result = _dataDescription.getAspect().compareTo(other._dataDescription.getAspect());
        if (result != 0) {
            return result;
        }
        result = _dataDescription.getSimulationVariant() - other._dataDescription.getSimulationVariant();
        return result;
    }

    /**
     * Liefert eine textuelle Beschreibung dieses Objekts für Debug-Zwecke.
     *
     * @return Beschreibung dieses Objekts.
     */
    public String toString() {
        return "DataIdentification{" + "_object=" + _object + ", _dataDescription=" + _dataDescription + "}";
    }
}
