/*
 * Copyright 2014-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.puk.config.
 *
 * de.bsvrz.puk.config is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.puk.config is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.puk.config.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.puk.config.configFile.datamodel;

import static de.bsvrz.dav.daf.main.impl.config.AttributeGroupUsageIdentifications.CONFIGURATION_ELEMENTS_IN_MUTABLE_SET;


import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.sys.funclib.dataSerializer.Serializer;
import de.bsvrz.sys.funclib.dataSerializer.SerializingFactory;
import de.bsvrz.sys.funclib.debug.Debug;
import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.List;

/**
 * Interface um die Speicherung einer dynamischem Menge ({@linkplain de.bsvrz.puk.config.configFile.datamodel.ConfigMutableSet}) zu realisieren. Diese
 * werden entweder als Konfigurationsdatensatz oder als eigene Datei gespeichert, je nach Implementierung
 *
 * @author Kappich Systemberatung
 */
public class MutableSetConfigDataStorage extends MutableSetStorage {

    private static final Debug _debug = Debug.getLogger();

    private final ConfigMutableSet _mutableSet;

    public MutableSetConfigDataStorage(final ConfigMutableSet mutableSet) {
        _mutableSet = mutableSet;
    }

    /**
     * Speichert die Elemente dieser Menge (auch historische) in einem konfigurierenden Datensatz ab.
     *
     * @param mutableElements Elemente dieser Menge
     *
     * @throws de.bsvrz.dav.daf.main.config.ConfigurationChangeException Falls die Elemente nicht in einem konfigurierenden Datensatz abgespeichert
     *                                                                   werden können.
     */
    @Override
    public void writeElements(final List<MutableElement> mutableElements) throws ConfigurationChangeException {
        // Liste in ein Byte-Array packen und abspeichern
        try {
            final ByteArrayOutputStream out = new ByteArrayOutputStream();
            final Serializer serializer = SerializingFactory.createSerializer(_mutableSet.getSerializerVersion(), out);
            for (MutableElement mutableElement : mutableElements) {
                if (mutableElement.getObjectId() == 0) {
                    // da hier eh immer alles neu geschrieben wird, können hier auch gleich ungültige Einträge gelöscht werden
                    continue;
                }
                serializer.writeLong(mutableElement.getObjectId());
                serializer.writeLong(mutableElement.getStartTime());
                serializer.writeLong(mutableElement.getEndTime());
                serializer.writeShort(mutableElement.getSimulationVariant());
            }
            final byte[] bytes = out.toByteArray();
            _mutableSet._systemObjectInfo.setConfigurationData(CONFIGURATION_ELEMENTS_IN_MUTABLE_SET, bytes);
            // ein Datensatz hat sich geändert -> dem Konfigurationsbereich Bescheid sagen
            _mutableSet.getConfigurationArea().setTimeOfLastChanges(ConfigConfigurationArea.KindOfLastChange.ConfigurationData);
            out.close();
        } catch (Exception ex) {
            final String errorMessage =
                "Der konfigurierende Datensatz mit den Elementen der Menge " + _mutableSet.getNameOrPidOrId() + " konnte nicht geschrieben werden";
            _debug.error(errorMessage, ex);
            throw new ConfigurationChangeException(errorMessage, ex);
        }
    }

    /**
     * Diese Methode liest den konfigurierenden Datensatz für die Elemente dieser Menge ein und gibt sie in einer Liste zurück.
     *
     * @return eine Liste von Elementen mit Zeitstempeln, die die Zugehörigkeitszeiträume repräsentieren
     */
    @Override
    public List<MutableElement> readElements() {
        // die eingelesenen Elemente werden nicht alle vorgehalten, da dies auf Dauer zu viele werden können
        try {
            byte[] bytes;
            // feste ID für die ATG-Verwendung um die Elemente einer Menge zu erhalten
            bytes = _mutableSet._systemObjectInfo.getConfigurationData(CONFIGURATION_ELEMENTS_IN_MUTABLE_SET);
            return deserializeMutableElements(_mutableSet, bytes);
        } catch (IllegalArgumentException ex) {
            final String errorMessage = "Elemente der dynamischen Menge '" + _mutableSet.getNameOrPidOrId() +
                                        "' konnten nicht ermittelt werden (evtl. wurde die Menge neu angelegt)";
            _debug.finest(errorMessage, ex.getMessage());
            return new ArrayList<>();
        } catch (Exception ex) {
            final String errorMessage = "Elemente der dynamischen Menge " + _mutableSet.getNameOrPidOrId() + " konnten nicht ermittelt werden";
            _debug.error(errorMessage, ex);
            throw new RuntimeException(errorMessage, ex);
        }
    }

}
