/*
 * Copyright 2014-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.puk.config.
 *
 * de.bsvrz.puk.config is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.puk.config is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.puk.config.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.puk.config.configFile.datamodel;

import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.util.BufferedRandomAccessFile;
import de.bsvrz.sys.funclib.debug.Debug;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * Interface um die Speicherung einer dynamischem Menge ({@linkplain ConfigMutableSet}) zu realisieren. Diese werden entweder als
 * Konfigurationsdatensatz oder als eigene Datei gespeichert, je nach Implementierung
 *
 * @author Kappich Systemberatung
 */
public class MutableSetExtFileStorage extends MutableSetStorage {

    private static final Debug _debug = Debug.getLogger();

    /**
     * Datei in der die Elementzugehörigkeit dieser Menge gespeichert werden soll, oder {@code null}, falls die Elementzugehörigkeit als Datensatz der
     * Menge gespeichert werden soll
     */
    private File _elementsFile;

    private ConfigMutableSet _mutableSet;

    private List<MutableElement> _saveElementLater;

    public MutableSetExtFileStorage(final File elementsFile, final ConfigMutableSet mutableSet) {
        _elementsFile = elementsFile;
        _mutableSet = mutableSet;
    }

    /**
     * Speichert die Elemente dieser Menge (auch historische) in einem konfigurierenden Datensatz ab.
     *
     * @param mutableElements Elemente dieser Menge
     *
     * @throws de.bsvrz.dav.daf.main.config.ConfigurationChangeException Falls die Elemente nicht in einem konfigurierenden Datensatz abgespeichert
     *                                                                   werden können.
     */
    @Override
    protected void writeElements(final List<MutableElement> mutableElements) throws ConfigurationChangeException {
        _saveElementLater = mutableElements;
        _mutableSet.getDataModel().saveSetElementsFileLater(this);
    }

    @Override
    public String toString() {
        return _elementsFile.toString();
    }

    public synchronized void saveElementsData() {
        List<MutableElement> elementsToSave = _saveElementLater;
        if (elementsToSave == null) {
            return;
        }
        if (_elementsFile.isFile()) {
            final File backupFile = new File(_elementsFile.getParentFile(), _elementsFile.getName() + ".old");
            if (backupFile.exists()) {
                if (!backupFile.delete()) {
                    _debug.warning("Backup-Datei kann nicht gelöscht werden", backupFile);
                }
            }
            if (!_elementsFile.renameTo(backupFile)) {
                _debug.warning("Backup-Datei kann nicht erstellt werden", backupFile);
            }
        }
        try {
            writeElementsToFile(elementsToSave, _elementsFile);
        } catch (IOException e) {
            _debug.error("Fehler beim Erzeugen der Datei mit der Elementzugehörigkeit einer dynamischen Menge " + _elementsFile, e);
        }
        _saveElementLater = null;
    }

    public void createBackupFile(final File targetDirectory) throws IOException {
        File targetFile = new File(targetDirectory, _elementsFile.getName());
        writeElementsToFile(readElements(), targetFile);
    }

    public void writeElementsToFile(final List<MutableElement> elementsToSave, final File file) throws IOException {
	    try (BufferedRandomAccessFile bufferedRandomAccessFile = new BufferedRandomAccessFile(file, "rw")) {
            // Version
            bufferedRandomAccessFile.writeByte(1);
            // Size
            int byteSize = MutableElement.BYTE_SIZE;
            int len = elementsToSave.size() * byteSize;
            bufferedRandomAccessFile.writeInt(len);
            bufferedRandomAccessFile.setLength(5 + len);
            // Bytes
            for (MutableElement mutableElement : elementsToSave) {
                bufferedRandomAccessFile.writeLong(mutableElement.getObjectId());
                bufferedRandomAccessFile.writeLong(mutableElement.getStartTime());
                bufferedRandomAccessFile.writeLong(mutableElement.getEndTime());
                bufferedRandomAccessFile.writeShort(mutableElement.getSimulationVariant());
            }
        }
    }

    /**
     * Diese Methode liest den konfigurierenden Datensatz für die Elemente dieser Menge ein und gibt sie in einer Liste zurück.
     *
     * @return eine Liste von Elementen mit Zeitstempeln, die die Zugehörigkeitszeiträume repräsentieren
     */
    protected List<MutableElement> readElements() {
        saveElementsData();
        // die eingelesenen Elemente werden nicht alle vorgehalten, da dies auf Dauer zu viele werden können
        try {

            byte[] bytes;
            if (_elementsFile.isFile() && _elementsFile.canRead()) {
                final FileInputStream in = new FileInputStream(_elementsFile);
	            try (in; DataInputStream din = new DataInputStream(in)) {
                    final byte version = din.readByte();
                    if (version == 1) {
                        final int size = din.readInt();
                        final byte[] readBytes = new byte[size];
                        din.readFully(readBytes);
                        bytes = readBytes;
                    } else {
                        final String errorMessage = "Elemente der dynamischen Menge " + _mutableSet.getNameOrPidOrId() +
		                        " konnten nicht ermittelt werden, unbekannte Version " + version;
                        _debug.error(errorMessage);
                        throw new RuntimeException(errorMessage);
                    }
                }
            } else {
                if (_elementsFile.exists()) {
                    _debug.warning("Datei mit der Elementzugehörigkeit einer dynamischen Menge kann nicht gelesen werden", _elementsFile.getPath());
                }
                bytes = new byte[0];
            }
            return deserializeMutableElements(_mutableSet, bytes);
        } catch (IllegalArgumentException ex) {
            final String errorMessage = "Elemente der dynamischen Menge '" + _mutableSet.getNameOrPidOrId() +
                                        "' konnten nicht ermittelt werden (evtl. wurde die Menge neu angelegt)";
            _debug.finest(errorMessage, ex.getMessage());
            return new ArrayList<>();
        } catch (Exception ex) {
            final String errorMessage = "Elemente der dynamischen Menge " + _mutableSet.getNameOrPidOrId() + " konnten nicht ermittelt werden";
            _debug.error(errorMessage, ex);
            throw new RuntimeException(errorMessage, ex);
        }
    }

}
