/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.puk.config.
 *
 * de.bsvrz.puk.config is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.puk.config is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.puk.config.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.puk.config.configFile.fileaccess;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Map von Objekt-ID zu Typ T. Diese Map entspricht grob einer {@code LinkedHashMap<Long, T>}, unterstützt aber mehrere doppelte Keys. Grund:
 * <p>
 * jemand fragt beispielsweise mit {@link de.bsvrz.dav.daf.main.config.DataModel#getObjects(long...)} folgendes Array ab:
 * <p>
 * {@code [1,3,1,1,2]}
 * <p>
 * Um für temporäre Operationen eine Map von dieser ID zu z. B. einem Systemobjekt anzulegen, darf die Anfrage nicht in {@code [1,3,2]} reduziert
 * werden, weil dann der Anfrager falsche Ergebnisse erhält.
 * <p>
 * Daher bietet diese map eine Möglichkeit, die ursprüngliche Reihenfolge und evtl. Doppelungen beizubehalten. Die Methode {@link #put(long, Object)}
 * aktualisiert immer alle Vorkommen einer ID.
 *
 * @author Kappich Systemberatung
 */
public class ObjIdMap<T> {

    /**
     * IDs in Original-Reihenfolge
     */
    private final long[] _ids;

    /**
     * Map ID → Wert
     */
    private final Map<Long, T> _values;

    /**
     * Erstellt eine neue ObjIdMap
     *
     * @param ids IDs in Originalreihenfolge
     */
    public ObjIdMap(List<Long> ids) {
        this(ids.stream().mapToLong(it -> it).toArray());
    }

    /**
     * Erstellt eine neue ObjIdMap
     *
     * @param ids IDs in Originalreihenfolge
     */
    public ObjIdMap(final long[] ids) {
        this._ids = ids;
        // Keyset erzeugen
        _values = new HashMap<>(ids.length);
        for (long id : ids) {
            _values.put(id, null);
        }
    }

    /**
     * ordnet einer ID ein Objekt vom Typ T zu.
     *
     * @param id    ID
     * @param value Wert
     */
    public void put(long id, T value) {
        assert _values.containsKey(id); // Fehlbenutzung der Klasse vermeiden
        _values.put(id, value);
    }

    /**
     * Gibt die IDs wieder zurück
     *
     * @return IDs
     */
    public long[] getIds() {
        return _ids.clone();
    }

    /**
     * Gibt die Werte zurück, die den ursprünglichen IDs zugeordnet wurden. Die Anzahl und Reihenfolge entspricht dem ursprünglichen Array, daher
     * können auch Objekte doppelt zurückgegeben werden.
     *
     * @return Werte
     */
    public List<T> values() {
        return Arrays.stream(_ids).mapToObj(_values::get).collect(Collectors.toList());
    }

}
