/*
 * Copyright 2011-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.puk.config.
 *
 * de.bsvrz.puk.config is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.puk.config is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.puk.config.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.puk.config.localCache;

import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.ObjectLookup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.HashMap;

/**
 * Klasse zur persistenten Speicherung dynamischer Objekte
 *
 * @author Kappich Systemberatung
 */
public class PersistentDynamicObjectCache implements ObjectLookup {

    private final HashMap<Long, PersistentDynamicObject> _objectsById = new HashMap<>();

    private final HashMap<String, PersistentDynamicObject> _objectsByPid = new HashMap<>();

    /**
     * Initialisiert einen neuen Cache mit einer Datei von der Festplatte
     *
     * @param dataModel Datenmodell
     * @param input     Datei
     *
     * @throws java.io.IOException Bei IO-Fehler
     */
    public PersistentDynamicObjectCache(final DataModel dataModel, final File input) throws IOException {
        if (!input.exists()) {
            return;
        }
	    try (DataInputStream dataInputStream = new DataInputStream(new BufferedInputStream(new FileInputStream(input)))) {
            final int version = dataInputStream.readInt();
            if (version != 0) {
                return;
            }
            final int size = dataInputStream.readInt();
            for (int i = 0; i < size; i++) {
                final PersistentDynamicObject persistentDynamicObject = PersistentDynamicObject.read(dataInputStream, dataModel);
                _objectsById.put(persistentDynamicObject.getId(), persistentDynamicObject);
                final String pid = persistentDynamicObject.getPid();
                if (!pid.isEmpty()) {
                    _objectsByPid.put(pid, persistentDynamicObject);
                }
            }
        }
    }

    /**
     * Initialisiert einen neuen leeren Cache
     */
    public PersistentDynamicObjectCache() {
    }

    public SystemObject getObject(final String pid) {
        if (pid.isEmpty()) {
            return null;
        }
        return _objectsByPid.get(pid);
    }

    public SystemObject getObject(final long id) {
        return _objectsById.get(id);
    }

    /**
     * Speichert ein Objekt in diesem Cache
     *
     * @param dynamicObject Dynamisches Objekt
     */
    public void storeObject(final DynamicObject dynamicObject) {
        final PersistentDynamicObject persistentDynamicObject = new PersistentDynamicObject(dynamicObject);
        _objectsById.put(persistentDynamicObject.getId(), persistentDynamicObject);
        final String pid = persistentDynamicObject.getPid();
        if (!pid.isEmpty()) {
            _objectsByPid.put(pid, persistentDynamicObject);
        }
    }

    /**
     * Schreibt den ganzen Cache auf die Festplatte
     *
     * @param output Ausgabedatei
     *
     * @throws IOException Bei IO-Fehler
     */
    public void writeToDisk(final File output) throws IOException {
	    try (DataOutputStream dataOutputStream = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(output)))) {
            // Version
            dataOutputStream.writeInt(0);
            // Anzahl Einträge
            dataOutputStream.writeInt(_objectsById.size());
            // Nacheinander alle Einträge
            for (final PersistentDynamicObject object : _objectsById.values()) {
                object.write(dataOutputStream);
            }
        }
    }

    @Override
    public String toString() {
        return "PersistentDynamicObjectCache{}";
    }
}
