/*
 * Copyright 2011-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.puk.config.
 *
 * de.bsvrz.puk.config is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.puk.config is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.puk.config.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.puk.config.main.communication.util;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.*;
import de.bsvrz.sys.funclib.debug.Debug;

import java.util.Formatter;
import java.util.List;

public class UnknownObjectHandler {

    private static final Debug _debug = Debug.getLogger();
    private final DataModel _dataModel;
    private boolean initialSuccess;

    public UnknownObjectHandler(DataModel dataModel) {
        this._dataModel = dataModel;
    }

    public SystemObject getSenderObject(Data.ReferenceValue referenceValue) {
        SystemObject sender = null;
        final long senderId = referenceValue.getId();

        short coding = (short)(senderId >> 48);
        
        StringBuilder message = null;
        
        try {
            sender = referenceValue.getSystemObject();
            if (sender == null) {
                message = new StringBuilder();
                message.append("Das SystemObjekt des Absenders einer Konfigurationsanfrage wurde nicht gefunden:\n");
            }
        } catch (RuntimeException e) {
            message = new StringBuilder();
            message.append("Das SystemObjekt des Absenders einer Konfigurationsanfrage wurde nicht gefunden: (").append(e.getMessage()).append(")\n");
        }

        if (sender != null) {
            if (!sender.isValid()) {
                message = new StringBuilder();
                message.append("Als Absender einer Konfigurationsanfrage ist ein nicht mehr gültiges Objekt angegeben\n");
            }
            if (!(sender instanceof ClientApplication) && !(sender instanceof DavApplication) && !(sender instanceof ConfigurationAuthority)) {
                if(message == null) {
                    message = new StringBuilder();
                }
                message.append(
                    "Als Absender einer Konfigurationsanfrage ist ein Objekt angegeben, das weder eine Applikation noch einen " +
                    "Konfigurationsverantwortlichen oder Datenverteiler darstellt\n");
            }
        }

        if (sender == null) {
            sender = new UnknownObject(referenceValue.getId(), _dataModel.getConfigurationAuthority().getConfigurationArea());
        }
        if (message != null) {
            message.append("  Id des Absenders: ").append(senderId).append("\n");
            message.append("  KV-Kodierung des Absenders: ").append(coding).append("\n");
            if (!(sender instanceof UnknownObject)) {
                message.append("  SystemObjekt des Absenders: ").append(sender).append("\n");
            }
            if(!initialSuccess) {
                message.append(
                    "  Eine mögliche Ursache dieses Problems könnte sein, dass beim Start des Datenverteilers die im Aufrufparameter -datenverteilerId=" +
                    " angegebene Objekt-Id nicht korrekt ist.\n  Folgende Datenverteiler sind der Konfiguration bekannt:\n");
                final SystemObjectType davType = _dataModel.getType("typ.datenverteiler");
                final List<SystemObject> davs = davType.getElements();
                Formatter formatter = new Formatter();
                formatter.format("%40s %22s %s\n", "PID", "ID", "NAME");
                for (SystemObject dav : davs) {
                    formatter.format("%40s %22d %s\n", dav.getPid(), dav.getId(), dav.getName());
                }
                message.append(formatter);
                _debug.error(message.toString());
            }
            else {
                _debug.fine(message.toString());
            }
            if(!(sender instanceof UnknownObject)) {
                // Objekt nicht Unbekannt, sondern explizit falsch
                throw new IllegalArgumentException(message.toString());
            }
        }
        else {
            initialSuccess = true;
        }
        
        return sender;
    }

}
