/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.dav.communication.accessControl;

import de.bsvrz.dav.daf.accessControl.AccessControlManager;
import de.bsvrz.dav.daf.accessControl.UserInfo;
import de.bsvrz.dav.daf.accessControl.internal.UserAction;
import de.bsvrz.dav.daf.communication.lowLevel.telegrams.BaseSubscriptionInfo;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.OneSubscriptionPerSendData;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SendSubscriptionNotConfirmed;
import de.bsvrz.dav.daf.main.SenderRole;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.dataSerializer.Deserializer;
import de.bsvrz.sys.funclib.dataSerializer.NoSuchVersionException;
import de.bsvrz.sys.funclib.dataSerializer.Serializer;
import de.bsvrz.sys.funclib.dataSerializer.SerializingFactory;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Zugriffssteuerungs-Plugin für den Datenverteiler, das Archivanfragen überwacht.
 *
 * @author Kappich Systemberatung
 * @version $Revision: 0000 $
 */
public class ArchiveAccessControlPlugin implements AccessControlPlugin {

    private static final Debug _debug = Debug.getLogger();
    private AccessControlManager _accessControlManager;
    private DataModel _dataModel;
    private ClientDavInterface _connection;
    private AttributeGroup _attributeGroup;
    private Aspect _aspect;
    private Aspect _replyAspect;

    private static int getSerVersion(byte[] data) {
        return ((data[0] & 0xFF) << 24) + ((data[1] & 0xFF) << 16) + ((data[2] & 0xFF) << 8) + (data[3] & 0xFF);
    }

    @Override
    public void initialize(final AccessControlManager accessControlManager, final ClientDavInterface clientDavInterface) {
        _accessControlManager = accessControlManager;
        _dataModel = clientDavInterface.getDataModel();
        _connection = clientDavInterface;
        _attributeGroup = _dataModel.getAttributeGroup("atg.archivAnfrageSchnittstelle");
        _aspect = _dataModel.getAspect("asp.anfrage");
        _replyAspect = _dataModel.getAspect("asp.antwort");
    }

    @Override
    public Collection<AttributeGroupUsage> getAttributeGroupUsagesToFilter() {
        if (_attributeGroup == null || _aspect == null || _replyAspect == null) {
            return List.of();
        }
        return List.of(_attributeGroup.getAttributeGroupUsage(_aspect));
    }

    @Nullable
    @Override
    public Data handleData(final long userID, final BaseSubscriptionInfo baseSubscriptionInfo, final Data data) {
        final AttributeGroupUsage usage = _dataModel.getAttributeGroupUsage(baseSubscriptionInfo.getUsageIdentification());
        SystemObject archive = _dataModel.getObject(baseSubscriptionInfo.getObjectID());

        // Stimmt die Attributgruppenverwendung des Datenpakets mit unserem Filter überein?
        // Diese Prüfung ist eigentlich unnötig, wird aber sinnvoll, wenn mehrere ATGUs gefiltert werden sollen
        if (usage.getAttributeGroup().getPid().equals("atg.archivAnfrageSchnittstelle")) {

            // Benutzerrechte ermitteln
            final UserInfo userInfo = _accessControlManager.getUserPermissions(userID);
            if (userInfo == null) {
                sendBackError(archive, data);
                return null;
            }

            // Data-Objekt analysieren
            final String request = data.getItem("nachrichtenTyp").asTextValue().getText();
            if ("Anfrage".equals(request)) {
                final byte[] dataArray = data.getUnscaledArray("daten").getByteArray();
                try {
                    // Versuchen, die Daten zu deserialisieren
                    final int serializerVersion = getSerVersion(dataArray); // Erste 4 Bytes = Serialisierer-Version

                    InputStream stream = new ByteArrayInputStream(dataArray, 4, dataArray.length - 4);
                    final Deserializer deserializer = SerializingFactory.createDeserializer(serializerVersion, stream);

                    deserializer.readInt();     // Priorität
                    deserializer.readInt();     // Größe des Empfangspuffers

                    return checkArchiveDataSpec(archive, data, userInfo, deserializer);
                } catch (NoSuchVersionException | IOException | RuntimeException e) {
                    _debug.warning("Fehler beim Analysieren der Nachricht", e);
                    // Aktion nicht erlaubt, Datenobjekt "unschädlich machen".
                    sendBackError(archive, data, e.toString());
                    return null;
                }
            }
        }
        return data;
    }

    private void sendBackError(final SystemObject archive, final Data data, final Object... errorDid) {
        SystemObject sender = data.getReferenceValue("absender").getSystemObject();
        int index = data.getUnscaledValue("anfrageIndex").intValue();
        Data returnData = _connection.createData(_attributeGroup);

        ByteArrayOutputStream bosResult = new ByteArrayOutputStream();
        try {
            final Serializer serializer = SerializingFactory.createSerializer(bosResult);
            serializer.writeByte(0);      // Fehler
            serializer.writeString("Keine Berechtigung (" + Arrays.stream(errorDid).map(String::valueOf).collect(Collectors.joining(", ")) + ")");
            serializer.writeInt(0);
        } catch (IOException e) {
            _debug.warning("Fehler beim serialisieren einer Fehlernachricht", e);
            return;
        }

        returnData.getReferenceValue("absender").setSystemObject(archive);
        returnData.getUnscaledValue("anfrageIndex").set(index);
        returnData.getTextValue("nachrichtenTyp").setText("AnfrageErgebnis");
        returnData.getUnscaledArray("daten").set(bosResult.toByteArray());

        sendData(new ResultData(sender, new DataDescription(_attributeGroup, _replyAspect), _connection.getTime(), returnData));
    }

    /**
     * Sendet einen DAF-Datensatz als Sender. Diese Methode übernimmt das An- und Abmelden des Senders.
     *
     * @param dataset datensatz
     */
    public void sendData(final ResultData dataset) {
        sendData(dataset, SenderRole.sender());
    }

    /**
     * Sendet einen DAF-Datensatz als Sender. Diese Methode übernimmt das an- und Abmelden des Senders.
     *
     * @param dataset    datensatz
     * @param senderRole Art Sender/Quelle
     */
    public void sendData(final ResultData dataset, final SenderRole senderRole) {
        try {
            _connection.subscribeSender(new DummySender(dataset), dataset.getObject(), dataset.getDataDescription(), senderRole);
        } catch (OneSubscriptionPerSendData oneSubscriptionPerSendData) {
            _debug.warning("Fehler beim senden", oneSubscriptionPerSendData);
        }
    }

    private Data checkArchiveDataSpec(final SystemObject archive, final Data data, final UserInfo userInfo, Deserializer ds) throws IOException {

        int anzQueries = ds.readInt();

        for (int i = 0; i < anzQueries; i++) {
            ds.readByte();             // Typ der Timingangabe

            ds.readByte();             // Intervallstart relativ
            ds.readLong();           // Intervallstart
            ds.readLong();             // Intervallende

            ds.readByte();             // Online aktuell
            ds.readByte();             // Online nachgeliefert
            ds.readByte();             // Nachgefordert aktuell
            ds.readByte();           // Nachgefordert nachgeliefert

            ds.readInt(); // Order
            ds.readInt(); // ArchiveRequestOption

            AttributeGroup atg = (AttributeGroup) ds.readObjectReference(_dataModel);
            Aspect asp = (Aspect) ds.readObjectReference(_dataModel);
            ds.readShort();// Simulationsvariante

            SystemObject obj = ds.readObjectReference(_dataModel);

            if (!userInfo.maySubscribeData(obj, atg, asp, UserAction.RECEIVER)) {
                sendBackError(archive, data, obj, atg, asp);
                return null;
            }
        }
        return data;
    }

    @Override
    public String toString() {
        return "ConfigAccessControlPlugin";
    }

    private class DummySender implements ClientSenderInterface {
        private final ResultData _dataset;

        public DummySender(final ResultData dataset) {
            _dataset = dataset;
        }

        @Override
        public void dataRequest(final SystemObject object, final DataDescription dataDescription, final byte state) {
            if (state == START_SENDING) {
                try {
                    _connection.sendData(_dataset);
                    _connection.unsubscribeSender(this, object, dataDescription);
                } catch (SendSubscriptionNotConfirmed sendSubscriptionNotConfirmed) {
                    sendSubscriptionNotConfirmed.printStackTrace();
                }
            }
        }

        @Override
        public boolean isRequestSupported(final SystemObject object, final DataDescription dataDescription) {
            return true;
        }
    }
}
