/*
 * Copyright 2011-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.dav.main;

import de.bsvrz.dav.daf.communication.lowLevel.AuthentificationProcess;
import de.bsvrz.dav.daf.communication.protocol.UserLogin;
import de.bsvrz.dav.daf.main.ClientDavParameters;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.impl.config.telegrams.TransmitterConnectionInfo;
import de.bsvrz.dav.dav.communication.appProtocol.T_A_HighLevelCommunication;
import de.bsvrz.dav.dav.communication.davProtocol.T_T_HighLevelCommunication;
import java.util.Collection;

/**
 * Interface für die Verwaltung der Verbindungen
 *
 * @author Kappich Systemberatung
 */
public interface LowLevelConnectionsManagerInterface {

    /**
     * Gibt die eigene Datenverteiler-id zurück
     *
     * @return die eigene Datenverteiler-id
     */
    long getTransmitterId();

    /**
     * @param error   True: ein Fehler ist aufgetreten
     * @param message genauere Beschreibung des Fehlers
     */
    void shutdown(boolean error, String message);

    /**
     * Gibt die eigene ClientDavConnection zurück
     *
     * @return die eigene ClientDavConnection
     */
    SelfClientDavConnection getSelfClientDavConnection();

    /**
     * Gibt das eigene Datenverteiler-Objekt zurück
     *
     * @return das eigene Datenverteiler-Objekt
     */
    SystemObject getMyTransmitter();

    /**
     * entfernt die angegebene Verbindung weil diese terminiert wurde
     *
     * @param connection
     */
    void removeConnection(T_A_HighLevelCommunication connection);

    /**
     * entfernt die angegebene Verbindung weil diese terminiert wurde
     *
     * @param connection
     */
    void removeConnection(T_T_HighLevelCommunication connection);

    /**
     * Gibt die Pid der lokalen Konfiguration zurück
     *
     * @return die Pid der lokalen Konfiguration
     */
    String getLocalModeConfigurationPid();

    /**
     * Gibt die Id der lokalen Konfiguration zurück
     *
     * @return die Id der lokalen Konfiguration
     */
    long getLocalModeConfigurationId();

    /**
     * Setzt die Parameter für eine lokale Konfiguration
     *
     * @param configurationPid Pid der Konfiguration
     * @param configurationId  Id der Konfiguration
     */
    void setLocalModeParameter(String configurationPid, long configurationId);

    /**
     * Wird aufgerufen, wenn die lokale Konfigurationsverbindung verfügbar ist.
     */
    void setLocalConfigurationAvailable();

    /**
     * Gibt die ServerDavParameters zurück
     *
     * @return die ServerDavParameters
     */
    ServerDavParameters getServerDavParameters();

    /**
     * Gibt die ClientDavParameters für die lokale Dav-Applikation zurück
     *
     * @return die ClientDavParameters
     */
    ClientDavParameters getClientDavParameters();

    /**
     * Gibt die angegebene Dav-Dav-Verbindung zurück
     *
     * @param transmitterId Id den verbundenen Transmitters
     *
     * @return Existierende Verbindung mit dieser Id oder null falls nicht vorhanden
     */
    T_T_HighLevelCommunication getTransmitterConnection(long transmitterId);

    /**
     * Gibt die angegebene Applikationsverbindung zurück
     *
     * @param applicationId Id der verbundenen Applikation
     *
     * @return Existierende Verbindung mit dieser Id oder null falls nicht vorhanden
     */
    T_A_HighLevelCommunication getApplicationConnection(long applicationId);

    /**
     * Diese Methode wird von der Protokollsteuerung aufgerufen, um einer Verbindung ein Gewicht zuzuweisen. Die Information wird von der
     * Wegverwaltung benutzt, wenn eine Verbindung bewertet wird.
     *
     * @param transmitterId ID des DAV
     *
     * @return Gewichtung der Verbindung
     */
    short getWeight(long transmitterId);

    /**
     * Bestimmt die Verbindungsinformationen für eine Verbindung von diesem Datenverteiler zum angegebenen Datenverteiler.
     *
     * @param connectedTransmitterId ID des DAV
     *
     * @return Verbindungsinformationen
     */
    TransmitterConnectionInfo getTransmitterConnectionInfo(long connectedTransmitterId);

    /**
     * Bestimmt die Verbindungsinformationen für eine Verbindung vom angegebenen Datenverteiler zu diesem Datenverteiler.
     *
     * @param connectedTransmitterId ID des DAV
     *
     * @return Verbindungsinformationen
     */
    TransmitterConnectionInfo getRemoteTransmitterConnectionInfo(long connectedTransmitterId);

    /**
     * Wird aufgerufen, sobald die ID einer Verbindung (sicher) bekannt ist (nach erfolgreicher Authentifizierung)
     *
     * @param communication Verbindung
     */
    void updateApplicationId(T_A_HighLevelCommunication communication);

    /**
     * Wird aufgerufen, sobald die ID einer Verbindung (sicher) bekannt ist (nach erfolgreicher Authentifizierung)
     *
     * @param communication Verbindung
     */
    void updateTransmitterId(T_T_HighLevelCommunication communication);

    /**
     * Gibt alle Appliaktionsverbindungen zurück
     *
     * @return alle Appliaktionsverbindungen
     */
    Collection<T_A_HighLevelCommunication> getApplicationConnections();

    /**
     * Gibt alle Datenverteilerverbindungen zurück
     *
     * @return alle Datenverteilerverbindungen
     */
    Collection<T_T_HighLevelCommunication> getTransmitterConnections();

    /**
     * Gibt {@code true} zurück, wenn der Datenverteiler sich gerade beendet
     *
     * @return {@code true}, wenn der Datenverteiler sich gerade beendet, sonst {@code false}
     */
    boolean isClosing();

    /**
     * Wartet darauf, dass die Parametrierung verfügbar ist. Diese Methode wird nach erfolgreichen Login aufgerufen.
     *
     * @param userId Benutzer-ID
     *
     * @return true: Es wurde erfolgreich auf die parametrierung gewartet, false: Das Warten wurde unterbrochen (z. B. der Datenverteiler wird
     *     beendet)
     */
    boolean waitForParamReady(UserLogin userId);

    /**
     * Wird aufgerufen, wenn sich der Status einer verbundenen Dav-Dav-Verbindung geändert hat um beispielsweise die
     * "atg.datenverteilerKommunikationsZustand" zu aktualisieren
     */
    void updateCommunicationState();

    /**
     * Setzt die Datenverteiler (über deren IDs), zu denen keine Verbindung aufgenommen werden sollen, weil diese temporär deaktiviert wurden (über
     * atg.deaktivierteVerbindungen)
     *
     * @param disabledConnections Deaktivierte Datenverteiler-IDs
     */
    void setDisabledTransmitterConnections(Collection<Long> disabledConnections);

    /**
     * Gibt {@code true} zurück, wenn eine Verbindung deaktiviert ist
     *
     * @return {@code true}, wenn eine Verbindung deaktiviert ist, sonst {@code false}
     */
    boolean isDisabledConnection(long remoteTransmitterId);
}
