/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.dav.dav.main;

import de.bsvrz.dav.daf.accessControl.AccessControlMode;
import de.bsvrz.dav.daf.communication.lowLevel.ParameterizedConnectionInterface;
import de.bsvrz.dav.daf.communication.srpAuthentication.SrpClientAuthentication;
import de.bsvrz.dav.daf.communication.srpAuthentication.SrpCryptoParameter;
import de.bsvrz.dav.daf.communication.srpAuthentication.SrpUtilities;
import de.bsvrz.dav.daf.main.ClientDavParameters;
import de.bsvrz.dav.daf.main.EncryptionConfiguration;
import de.bsvrz.dav.daf.main.MissingParameterException;
import de.bsvrz.dav.daf.main.authentication.AuthenticationFile;
import de.bsvrz.dav.daf.main.authentication.ClientCredentials;
import de.bsvrz.dav.daf.main.authentication.InteractiveAuthentication;
import de.bsvrz.dav.daf.main.authentication.SimpleUserProperties;
import de.bsvrz.dav.daf.main.authentication.UserProperties;
import de.bsvrz.dav.daf.main.impl.ArgumentParser;
import de.bsvrz.dav.daf.main.impl.CommunicationConstant;
import de.bsvrz.dav.daf.main.impl.InvalidArgumentException;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.nio.file.Paths;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Diese Klasse stellt die Parameter des Datenverteilers auf Server-Seite zur Verfügung. Diese Parameter werden durch den Konstruktor oder durch
 * entsprechende Setter-Methoden gesetzt und können durch entsprechende Getter-Methoden gelesen werden.
 *
 * @author Kappich Systemberatung
 */
public final class ServerDavParameters {

    /** DebugLogger für Debug-Ausgaben */
    private static final Debug DEBUG = Debug.getLogger();

    /** Parameter Schlüssel */
    private static final String LOCAL_CONFIGURATION_DATA_KEY = "-lokaleKonfiguration=";

    private static final String REMOTE_CONFIGURATION_DATA_KEY = "-remoteKonfiguration=";

    private static final String SEND_KEEP_ALIVE_TIMEOUT_KEY = "-timeoutSendeKeepAlive=";

    private static final String RECEIVE_KEEP_ALIVE_TIMEOUT_KEY = "-timeoutEmpfangeKeepAlive=";

    private static final String USER_NAME_KEY = "-benutzer=";

    private static final String AUTHENTIFICATION_FILE_KEY = "-authentifizierung=";

    private static final String AUTHENTIFICATION_PROCESS_KEY = "-authentifizierungsVerfahren=";

    private static final String TRANSMITTER_ID_KEY = "-datenverteilerId=";

    private static final String DAV_DAV_PORT_KEY = "-davDavPort=";

    private static final String DAV_DAV_PORT_OFFSET_KEY = "-davDavPortOffset=";

    private static final String DAV_APP_PORT_KEY = "-davAppPort=";

    private static final String NEIBOUR_CONNECTION_TIMEOUT_KEY = "-timeoutNachbar=";

    private static final String SYNC_RESPONCE_TIMEOUT_KEY = "-timeoutAntwort=";

    private static final String CONFIGURATION_USER_NAME_KEY = "-konfigurationsBenutzer=";

    private static final String PARAMETER_USER_NAME_KEY = "-parametrierungsBenutzer=";

    private static final String ACCESS_CONTROL_PLUGIN_KEY = "-zugriffsRechtePlugins=";

    private static final String PARAMETER_SEPARATOR = ":";

    private final int _passivePort;

    private final String _passiveCommunication;

    /** The ressource bundle of this server */
    private final ResourceBundle resourceBundle = ResourceBundle.getBundle("de.bsvrz.dav.dav.main.serverResourceBundle", Locale.getDefault());
    /** Liste mit den Plugins für die Kontrolle der Benutzerrechte über den Datenverteiler */
    private final List<String> _accessControlPlugins = new ArrayList<>();
    /** lokale Konfiguration = true */
    private boolean _localConfiguration;
    /** Die Konfigurations-Id */
    private long _configurationId;
    /** Die Konfigurations-Pid */
    private String _configurationPid;
    /** Datenverteileradresse für die Konfigurationsanbindung */
    private String _configDataTransmitterAddress;
    /** Datenverteilersubadresse für die Konfigurationsanbindung */
    private int _configDataTransmitterSubAddress;
    /**
     * Das Timeout zum Senden von KeepAlive-Telegrammen. Der Wert dient als Vorschlag für die Verhandlung mit dem Datenverteiler, der den zu
     * verwendenden Wert festlegt.
     */
    private long _receiveKeepAliveTimeout;
    /**
     * Das KeepAlive-Timeout beim Empfang von Telegrammen. Der Wert dient als Vorschlag für die Verhandlung mit dem Datenverteiler, der den zu
     * verwendenden Wert festlegt.
     */
    private long _sendKeepAliveTimeout;
    /** Der Name des Kommunikationsprotokolls (Default: TCP-IP) */
    private String _lowLevelCommunicationName;
    /** Parameter für das Kommunikationsprotokoll */
    private String _lowLevelCommunicationParameters;
    /** Der Name des Authentifikationsprozesses (Default: HMAC-MD5) */
    private String _authentificationProcessName;
    /** Der Name des Benutzers */
    private String _userName;
    /** Die lokale Datenverteiler-Id */
    private long _dataTransmitterId;
    /** Der Name des Datenverteilers (Default: Datenverteiler) */
    private String _dataTransmitterName;
    /** Die Pid des Datenverteilertyps (Default: typ.datenverteiler) */
    private String _dataTransmitterTypePid;
    /** Die Größe des Sendepuffers in Byte, der bei der Kommunikation mit dem Datenverteiler eingesetzt wird. */
    private int _davCommunicationOutputBufferSize;
    /** Die Größe des Empfangspuffers in Byte, der bei der Kommunikation mit dem Datenverteiler eingesetzt wird. */
    private int _davCommunicationInputBufferSize;
    /** Die Größe des Sendepuffers in Byte, der bei der Kommunikation mit einer Applikation eingesetzt wird. */
    private int _appCommunicationOutputBufferSize;
    /** Die Größe des Empfangspuffers in Byte, der bei der Kommunikation mit einer Applikation eingesetzt wird. */
    private int _appCommunicationInputBufferSize;
    /**
     * Die Verzögerungszeit zur Übertragung von gepufferten und zu versendenden Telegrammen. Die Übertragung der gesammelten Daten im Sendepuffer
     * findet erst statt, wenn die hier angegebene Zeit lang keine Daten mehr in der Puffer geschrieben wurden oder der Sendepuffer voll ist.
     */
    private long _communicationSendFlushDelay;
    /**
     * Die maximale Größe von Datentelegrammen. Größere Telegramme werden in mehrere Telegramme zerlegt.
     *
     * @param maxTelegramSize  Maximale Größe von versendeten Datentelegrammen.
     */
    private int _maxTelegramSize;
    /** Die Subadresse auf der der Datenverteiler auf die Datenverteilerverbindungen wartet. */
    private int _transmitterConnectionsSubAddress;
    /** Offset für die Subadresse auf der der Datenverteiler auf die Datenverteilerverbindungen wartet. */
    private int _transmitterConnectionsSubAddressOffset;
    /** Die Subadresse auf der der Datenverteiler auf die Applikationsverbindungen wartet. */
    private int _applicationConnectionsSubAddress;
    /** Der Benutzername der Konfiguration */
    private String _configurationUserName;
    /** Das Benutzerpasswort der Konfiguration */
    private ClientCredentials _configurationClientCredentials;
    /** Der Benutzername der Parametrierung */
    private String _parameterUserName;
    /** Das Benutzerpasswort der Parametrierung */
    private ClientCredentials _parameterClientCredentials;
    /** Benutzerpassworttabelle */
    private UserProperties _userProperties;
    /** Flag, das angibt, ob die Benutzerrechte durch diesen Datenverteiler geprüft werden sollen. */
    private AccessControlMode _userRightsChecking = AccessControlMode.NewDataModel;
    /** Zeit in Millisekunden, die gewartet werden soll bevor Verbindungen von anderen Datenverteilern akzeptiert werden dürfen. */
    private long _initialInterDavServerDelay;

    /** Zeit in Millisekunden, die gewartet werden soll bevor versucht wird, abgebrochene Verbindungen neu aufzubauen. */
    private long _reconnectInterDavDelay;

    /** Kennung, die (falls {@code true}) dafür sorgt, dass der Datenverteiler auf die Applikationsfertigmeldung der Parametrierung wartet. */
    private boolean _waitForParamApp;

    /**
     * Inkarnationsname der Parametrierung auf deren Applikationsfertigmeldung gewartet werden soll oder {@code null} falls der Inkarnationsname egal
     * ist.
     */
    private String _paramAppIncarnationName;

    /**
     * Pid des Konfigurationsbereichs in dem Applikationsobjekte erzeugt werden sollen oder Leerstring falls der Default-Bereich der Konfiguration
     * verwendet werden soll.
     */
    private String _configAreaPidForApplicationObjects = "";

    /**
     * Ob die alte Hmac-Authentifizierung erlaubt ist.
     */
    private boolean _allowHmacAuthentication;

    /**
     * Bevorzugte Konfiguration der Verschlüsselung
     */
    private EncryptionConfiguration _encryptionPreference;

    /**
     * Applikationsverbindungen, die aktiv aufgebaut werden
     */
    private Set<ApplicationInfo> _activeConnections;

    /**
     * Erzeugt einen neuen Parametersatz mit Defaultwerten für die einzelnen Parameter und setzt die in den übergebenen Aufrufargumenten angegebenen
     * Parameter mit den angegebenen Werten. Unbekannte Aufrufargumente werden ignoriert. Bekannte Aufrufargumente werden nach der Umsetzung auf null
     * gesetzt, um zu signalisieren, daß diese Argumente bereits interpretiert wurden.
     *
     * @param argumentList Die auszuwertenden Aufrufargumente.
     *
     * @throws MissingParameterException Falls ein Argument nicht oder nicht korrekt angegeben wurde.
     */
    public ServerDavParameters(ArgumentList argumentList) throws MissingParameterException {
        initialiseDavParameters(argumentList.getArgumentStrings());
        if (argumentList.hasArgument("-davTelegrammPuffer")) {
            try {
                final String[] davBufferValues = argumentList.fetchArgument("-davTelegrammPuffer").asNonEmptyString().split(":");
                if (davBufferValues.length < 1) {
                    throw new IllegalArgumentException("Zu wenig Argumente.");
                }
                if (davBufferValues.length > 2) {
                    throw new MissingParameterException("Zu viele Argumente.");
                }
                for (int i = 0; i < davBufferValues.length; i++) {
                    String davBufferValue = davBufferValues[i];

                    final Integer bufferSize = Integer.valueOf(davBufferValue);
                    if (bufferSize < 100000) {
                        throw new IllegalArgumentException("Puffergröße " + bufferSize + " ist nicht sinnvoll. Mindestwert: 100000.");
                    }
                    if (i == 0) {
                        _davCommunicationOutputBufferSize = bufferSize;
                    }
                    _davCommunicationInputBufferSize = bufferSize;
                }
            } catch (Exception e) {
                throw new MissingParameterException(
                    "Argument -davTelegrammPuffer sollte eine oder zwei mit Doppelpunkt getrennte Zahlen enthalten, die die Sendepuffergröße bzw. " +
                    "die Empfangspuffergröße in Bytes spezifizieren.");
            }
        }

        String waitForParamAppValue = argumentList.fetchArgument("-warteAufParametrierung=ja").asNonEmptyString();
        if (waitForParamAppValue.toLowerCase().trim().equals("ja")) {
            _waitForParamApp = true;
            _paramAppIncarnationName = null;
        } else if (waitForParamAppValue.toLowerCase().trim().equals("nein")) {
            _waitForParamApp = false;
            _paramAppIncarnationName = null;
        } else {
            _waitForParamApp = true;
            _paramAppIncarnationName = waitForParamAppValue;
        }

        _configAreaPidForApplicationObjects = argumentList.fetchArgument("-konfigurationsBereichFuerApplikationsobjekte=").asString();

        if (argumentList.hasArgument("-appTelegrammPuffer")) {
            try {
                final String[] appBufferValues = argumentList.fetchArgument("-appTelegrammPuffer").asNonEmptyString().split(":");
                if (appBufferValues.length < 1) {
                    throw new IllegalArgumentException("Zu wenig Argumente.");
                }
                if (appBufferValues.length > 2) {
                    throw new MissingParameterException("Zu viele Argumente.");
                }
                for (int i = 0; i < appBufferValues.length; i++) {
                    String appBufferValue = appBufferValues[i];

                    final Integer bufferSize = Integer.valueOf(appBufferValue);
                    if (bufferSize < 100000) {
                        throw new IllegalArgumentException("Puffergröße " + bufferSize + " ist nicht sinnvoll. Mindestwert: 100000.");
                    }
                    if (i == 0) {
                        _appCommunicationOutputBufferSize = bufferSize;
                    }
                    _appCommunicationInputBufferSize = bufferSize;
                }
            } catch (Exception e) {
                throw new MissingParameterException(
                    "Argument -appTelegrammPuffer sollte eine oder zwei mit Doppelpunkt getrennte Zahlen enthalten, die die Sendepuffergröße bzw. " +
                    "die Empfangspuffergröße in Bytes spezifizieren.");
            }
        }
        final String delayArgumentName = "-verzögerungFürAndereDatenverteiler";
        if (argumentList.hasArgument(delayArgumentName)) {
            _initialInterDavServerDelay = argumentList.fetchArgument(delayArgumentName).asRelativeTime();
        } else {
            final String alternateArgument = delayArgumentName.replaceAll("[ö]", "oe").replaceAll("[ü]", "ue");
            _initialInterDavServerDelay = argumentList.fetchArgument(alternateArgument + "=60s").asRelativeTime();
        }

        _reconnectInterDavDelay = argumentList.fetchArgument("-wiederverbindungsWartezeit=60s").asRelativeTime();
        if (_reconnectInterDavDelay < 1) {
            throw new MissingParameterException(
                "Die angegebene -wiederverbindungsWartezeit=" + _reconnectInterDavDelay + "ms ist ungültig: Muss > 0 sein.");
        }

        final String[] strings =
            argumentList.fetchArgument("-tcpKommunikationsModul=" + getLowLevelCommunicationName()).asNonEmptyString().split(":", 2);
        final String tcpCommunicationClassName = strings[0];
        final String tcpCommunicationParameters = (strings.length > 1) ? strings[1] : "";
        try {
            final Class<?> aClass = Class.forName(tcpCommunicationClassName);
            if (!tcpCommunicationParameters.isEmpty() && !(aClass.newInstance() instanceof ParameterizedConnectionInterface)) {
                throw new MissingParameterException(
                    "Das angegebene Kommunikationsverfahren " + tcpCommunicationClassName + " unterstützt keine Parameter: " +
                    tcpCommunicationParameters);
            }
        } catch (ClassNotFoundException e) {
            throw new MissingParameterException(
                "Das angegebene Kommunikationsverfahren ist nicht verfügbar, Klassenname: " + tcpCommunicationClassName, e);
        } catch (InstantiationException e) {
            throw new MissingParameterException(
                "Das angegebene Kommunikationsverfahren kann nicht instantiiert werden, Klassenname: " + tcpCommunicationClassName, e);
        } catch (IllegalAccessException e) {
            throw new MissingParameterException(
                "Auf das angegebene Kommunikationsverfahren kann nicht zugegriffen werden, Klassenname: " + tcpCommunicationClassName, e);
        }
        _lowLevelCommunicationName = tcpCommunicationClassName;
        _lowLevelCommunicationParameters = tcpCommunicationParameters;

        _allowHmacAuthentication = argumentList.fetchArgument("-erlaubeHmacAuthentifizierung=ja").booleanValue();

        _encryptionPreference = argumentList.fetchArgument("-verschluesselung=auto").asEnum(EncryptionConfiguration.class);

        if (argumentList.hasArgument("-passiv")) {
            _passivePort = argumentList.fetchArgument("-passiv").intValueBetween(0, Integer.MAX_VALUE);
            _passiveCommunication = _lowLevelCommunicationName;
        } else {
            _passivePort = 0;
            _passiveCommunication = null;
        }

        String activeConnections = argumentList.fetchArgument("-aktiv=").asString();
        _activeConnections = Arrays.stream(activeConnections.split(","))
                .map(String::trim)                // Trim each result
                .filter(s -> !s.isEmpty())        // Omit empty strings
                .map(ApplicationInfo::parse)
                .collect(Collectors.toUnmodifiableSet());
    }

    /** Gibt auf der Standardausgabe die möglichen Startargumente einer Datenverteilerapplikation aus. */
    public static void printArgumentsList() {
        System.out.println();
        System.out.println("----------Argumente des Datenverteilers----------");
        System.out.println();
        System.out.println("-rechtePrüfung=ja|nein|alt|neu");
        System.out.println("-zugriffsRechtePlugins=Plugin(Zeichenkette)[,Plugin(Zeichenkette)][,Plugin(Zeichenkette)]...");
        System.out.println("-remoteKonfiguration=Datenverteileradresse(Zeichenkette):Datenverteilersubadresse(Zahl):Konfigurationspid(Zeichenkette)");
        System.out.println("-datenverteilerId=datenverteilerId(Zeichenkette)");
        System.out.println("-benutzer=Benutzername(Zeichenkette)");
        System.out.println("-authentifizierung=Authentifizierungsdateiname(Zeichenkette)");
        System.out.println("-authentifizierungsVerfahren=Authentifizierungsverfahren(Zeichenkette)");
        System.out.println("-timeoutSendeKeepAlive=time(Zahl in Sekunden)");
        System.out.println("-timeoutEmpfangeKeepAlive=time(Zahl in Sekunden)");
        System.out.println("-timeoutNachbar=time(Zahl in Sekunden)");
        System.out.println("-timeoutAntwort=time(Zahl in Sekunden)");
        System.out.println("-davDavPort=port(Zahl)");
        System.out.println("-davAppPort=port(Zahl)");
        System.out.println("-warteAufParametrierung=wert (ja, nein oder Inkarnationsname der Parametrierung)");
        System.out.println("-konfigurationsBereichFuerApplikationsobjekte=konfigurationsbereich (Pid oder Leerstring)");
        System.out.println();
        System.out.println();
        System.out.println("Bemerkungen: ");
        System.out.println();
        System.out.println("Es gibt zwei Startmodi für den Datenverteiler:");
        System.out.println("Modus 1: wenn die Konfiguration sich lokal anmelden muss (ohne -remoteKonfiguration)");
        System.out.println("Modus 2: wenn die Konfiguration über einen anderen Datenverteiler erreichbar ist (-remoteKonfiguration=...)");
    }

    /**
     * Wertet die Default-Argumente aus und liest sie ggf. aus der {@code serverResourceBundle.properties}-Datei aus.
     *
     * @param startArguments die Default-Argumente
     *
     * @throws MissingParameterException Falls ein Argument nicht oder nicht korrekt angegeben wurde.
     */
    private void initialiseDavParameters(final String[] startArguments) throws MissingParameterException {
        try {
            String tmp, parameter;

            parameter = getParameter(startArguments, "-rechtePrüfung=");
            if (parameter == null) {
                parameter = getParameter(startArguments, "-rechtePruefung=");
            }
            if (parameter != null) {
                parameter = parameter.substring(parameter.indexOf('=') + 1).trim().toLowerCase();
                switch (parameter) {
                    case "no", "nein", "false" -> setUserRightsChecking(AccessControlMode.Disabled);
                    case "yes", "ja", "wahr", "neu" -> setUserRightsChecking(AccessControlMode.NewDataModel);
                    case "alt" -> setUserRightsChecking(AccessControlMode.OldDataModel);
                    default ->
                            throw new MissingParameterException("Aufrufparameter zur Rechteprüfung sollte den Wert 'neu', 'alt', 'ja' oder 'nein' haben");
                }
            }

            // Rechteprüfungs-Plugins
            parameter = getParameter(startArguments, ACCESS_CONTROL_PLUGIN_KEY);
            if (parameter != null) {
                try {
                    final String param = ArgumentParser.getParameter(parameter, ACCESS_CONTROL_PLUGIN_KEY);
                    if ((param == null) || (param.isEmpty())) {
                        throw new MissingParameterException(
                            "Der Parameter für die Plug-Ins zur Zugriffsrechteverwaltung muss folgende Formatierung besitzen: " +
                            ACCESS_CONTROL_PLUGIN_KEY + "=Bezeichnung[,Bezeichnung,Bezeichnung,...]");
                    }
                    final String[] plugins = param.split(",");
                    for (String plugin : plugins) {
                        try {
                            plugin = plugin.trim();
                            Class.forName(plugin);
                            _accessControlPlugins.add(plugin);
                        } catch (ClassNotFoundException ex) {
                            throw new MissingParameterException("Das angegebene Plug-In existiert nicht: " + plugin, ex);
                        }
                    }
                } catch (InvalidArgumentException ex) {
                    throw new MissingParameterException(
                        "Der Parameter für die Plug-Ins zur Zugriffsrechteverwaltung muss folgende Formatierung besitzen: " +
                        ACCESS_CONTROL_PLUGIN_KEY + "=Bezeichnung[,Bezeichnung,Bezeichnung,...]");
                }
            }

            // Lokale Konfiguration?
            parameter = getParameter(startArguments, LOCAL_CONFIGURATION_DATA_KEY);
            if (parameter == null) {
                // Remote Konfiguration?
                parameter = getParameter(startArguments, REMOTE_CONFIGURATION_DATA_KEY);
                if (parameter == null) {
                    tmp = resourceBundle.getString("LokaleKonfiguration");
                    _localConfiguration = Integer.parseInt(tmp) == 1;
                    if (_localConfiguration) {
                        _configurationPid = resourceBundle.getString("Konfiguration-PID");
                        tmp = resourceBundle.getString("Konfiguration-ID");
                        _configurationId = Long.parseLong(tmp);
                    } else {
                        _configDataTransmitterAddress = resourceBundle.getString("Konfig-Datenverteiler-Adresse");
                        tmp = resourceBundle.getString("Konfig-Datenverteiler-Subadresse");
                        _configDataTransmitterSubAddress = Integer.parseInt(tmp);
                        _configurationPid = resourceBundle.getString("Konfiguration-PID");
                    }
                } else {
                    try {
                        String[] parameters = ArgumentParser.getParameters(parameter, REMOTE_CONFIGURATION_DATA_KEY, PARAMETER_SEPARATOR);
                        if ((parameters != null) && (parameters.length == 3)) {
                            _localConfiguration = false;
                            _configDataTransmitterAddress = parameters[0];
                            _configDataTransmitterSubAddress = Integer.parseInt(parameters[1]);
                            _configurationPid = parameters[2];
                            if ("null".equals(_configurationPid)) {
                                _configurationPid = CommunicationConstant.LOCALE_CONFIGURATION_PID_ALIASE;
                            }
                        } else {
                            throw new MissingParameterException("Remote-Konfiguration-Parameter muss folgende Formatierung besitzen: " +
                                                                "-remoteKonfiguration=Zeichenkette:Zahl:Zeichenkette");
                        }
                    } catch (InvalidArgumentException | NumberFormatException ex) {
                        throw new MissingParameterException("Remote-Konfiguration-Parameter muss folgende Formatierung besitzen: " +
                                                            "-remoteKonfiguration=Zeichenkette:Zahl:Zeichenkette");
                    }
                }
            } else {
                throw new IllegalArgumentException(
                    "Aufrufargument " + LOCAL_CONFIGURATION_DATA_KEY + " wird nicht mehr unterstützt, da die Einstellung jetzt" +
                    " automatisch von der lokalen Konfiguration übernommen wird.");
            }

            //Send keep alive timeout
            parameter = getParameter(startArguments, SEND_KEEP_ALIVE_TIMEOUT_KEY);
            if (parameter == null) {
                tmp = resourceBundle.getString("Keepalive-Sendezeitgrenze");
                _sendKeepAliveTimeout = Long.parseLong(tmp);
            } else {
                try {
                    tmp = ArgumentParser.getParameter(parameter, SEND_KEEP_ALIVE_TIMEOUT_KEY);
                    if ((tmp == null) || (tmp.isEmpty())) {
                        throw new MissingParameterException(
                            "Sende-Keep-Alive-Timeout-Parameter muss folgende Formatierung besitzen: -timeoutSendeKeepAlive=Zahl");
                    }
                    _sendKeepAliveTimeout = Long.parseLong(tmp) * 1000L;
                } catch (InvalidArgumentException | NumberFormatException ex) {
                    throw new MissingParameterException(
                        "Sende-Keep-Alive-Timeout-Parameter muss folgende Formatierung besitzen: -timeoutSendeKeepAlive=Zahl");
                }
            }
            if (_sendKeepAliveTimeout < 1000) {
                throw new MissingParameterException("Timeouts müssen grösser gleich als 1 Sekunde sein");
            }

            //Receive keep alive timeout
            parameter = getParameter(startArguments, RECEIVE_KEEP_ALIVE_TIMEOUT_KEY);
            if (parameter == null) {
                tmp = resourceBundle.getString("Keepalive-Empfangszeitgrenze");
                _receiveKeepAliveTimeout = Long.parseLong(tmp);
            } else {
                try {
                    tmp = ArgumentParser.getParameter(parameter, RECEIVE_KEEP_ALIVE_TIMEOUT_KEY);
                    if ((tmp == null) || (tmp.isEmpty())) {
                        throw new MissingParameterException(
                            "Empfang-Keep-Alive-Timeout-Parameter muss folgende Formatierung besitzen: -timeoutEmpfangeKeepAlive=Zahl");
                    }
                    _receiveKeepAliveTimeout = Long.parseLong(tmp) * 1000L;
                } catch (InvalidArgumentException | NumberFormatException ex) {
                    throw new MissingParameterException(
                        "Empfang-Keep-Alive-Timeout-Parameter muss folgende Formatierung besitzen: -timeoutEmpfangeKeepAlive=Zahl");
                }
            }
            if (_receiveKeepAliveTimeout < 1000) {
                throw new MissingParameterException("Timeouts müssen grösser gleich als 1 Sekunde sein");
            }

            // User Passwort Tabelle
            parameter = getParameter(startArguments, AUTHENTIFICATION_FILE_KEY);
            if (parameter == null) {
                _userProperties = null;
            } else {
                try {
                    tmp = ArgumentParser.getParameter(parameter, AUTHENTIFICATION_FILE_KEY);
                    if ((tmp == null) || (tmp.isEmpty())) {
                        throw new MissingParameterException(
                            "Authentifizierungsdatei-Parameter muss folgende Formatierung besitzen: -authentifizierung=Zeichenkette");
                    }
                    if (tmp.equals("STDIN") || tmp.equalsIgnoreCase("interaktiv")) {
                        _userProperties = InteractiveAuthentication.getInstance();
                    } else {
                        _userProperties = new AuthenticationFile(Paths.get(tmp));
                    }
                } catch (InvalidArgumentException ex) {
                    throw new MissingParameterException(
                        "Authentifizierungsdatei-Parameter muss folgende Formatierung besitzen: -authentifizierung=Zeichenkette");
                }
            }

            //User name
            parameter = getParameter(startArguments, USER_NAME_KEY);
            String userPassword;
            if (parameter == null) {
                _userName = resourceBundle.getString("Benutzername");
                userPassword = resourceBundle.getString("Benutzerpasswort");
            } else {
                try {
                    _userName = ArgumentParser.getParameter(parameter, USER_NAME_KEY);
                    if ((_userName == null) || (_userName.isEmpty())) {
                        throw new MissingParameterException("Benutzername-Parameter muss folgende Formatierung besitzen: -benutzer=Zeichenkette");
                    }
                    userPassword = null;
                } catch (InvalidArgumentException ex) {
                    throw new MissingParameterException("Benutzername-Parameter muss folgende Formatierung besitzen: -benutzer=Zeichenkette");
                }
            }

            //Authentification process
            parameter = getParameter(startArguments, AUTHENTIFICATION_PROCESS_KEY);
            if (parameter == null) {
                _authentificationProcessName = resourceBundle.getString("AuthentificationProcessName");
            } else {
                try {
                    _authentificationProcessName = ArgumentParser.getParameter(parameter, AUTHENTIFICATION_PROCESS_KEY);
                    if ((_authentificationProcessName == null) || (_authentificationProcessName.isEmpty())) {
                        throw new MissingParameterException(
                            "Der Parameter für die Klasse des Authentifizierungsverfahren muss folgende Formatierung besitzen: " +
                            "-authentifizierungsVerfahren=Zeichenkette");
                    }
                } catch (InvalidArgumentException ex) {
                    throw new MissingParameterException(
                        "Der Parameter für die Klasse des Authentifizierungsverfahren muss folgende Formatierung besitzen: " +
                        "-authentifizierungsVerfahren=Zeichenkette");
                }
            }
            try {
                Class.forName(_authentificationProcessName);
            } catch (ClassNotFoundException ex) {
                throw new MissingParameterException(
                    "Die Implementierung des Authentifizierungsverfahrens existiert nicht: " + _authentificationProcessName);
            }

            //Konfiguration User name
            parameter = getParameter(startArguments, CONFIGURATION_USER_NAME_KEY);
            if (_localConfiguration) {
                if (parameter == null) {
                    _configurationUserName = resourceBundle.getString("configurationUserName");
                } else {
                    try {
                        _configurationUserName = ArgumentParser.getParameter(parameter, CONFIGURATION_USER_NAME_KEY);
                        if ((_configurationUserName == null) || (_configurationUserName.isEmpty())) {
                            throw new MissingParameterException(
                                "KonfigurationsbenutzerName-Parameter muss folgende Formatierung besitzen: -konfigurationsBenutzer=Zeichenkette");
                        }
                    } catch (InvalidArgumentException ex) {
                        throw new MissingParameterException(
                            "KonfigurationsbenutzerName-Parameter muss folgende Formatierung besitzen: -konfigurationsBenutzer=Zeichenkette", ex);
                    }
                }

                //Konfig user password
                if (_userProperties == null) {
                    DEBUG.warning("Keine Authentifizierungsdatei angegeben. Verwende Standardpasswort um die Konfiguration unter dem Benutzer \"" +
                                  _configurationUserName + "\" anzumelden.");
                    _configurationClientCredentials = ClientCredentials.ofString(resourceBundle.getString("configurationUserPassword"));
                } else {
                    _configurationClientCredentials = _userProperties.getClientCredentials(_configurationUserName);
                    if (_configurationClientCredentials == null) {
                        if (parameter == null) {
                            DEBUG.warning("Das Passwort für den Konfigurationsbenutzer " + _configurationUserName +
                                          " ist in der Authentifizierungsdatei nicht vorhanden.");
                            _configurationClientCredentials = ClientCredentials.ofString(resourceBundle.getString("configurationUserPassword"));
                        } else {
                            throw new MissingParameterException("Das Passwort für den Konfigurationsbenutzer " + _configurationUserName +
                                                                " ist in der Authentifizierungsdatei nicht vorhanden.");
                        }
                    }
                }
            } else if (parameter != null) {
                DEBUG.warning("Aufrufargument " + parameter + " wird ignoriert, da eine Remote-Konfiguration verwendet wird.");
            }

            //Parameter User name
            parameter = getParameter(startArguments, PARAMETER_USER_NAME_KEY);
            if (parameter == null) {
                _parameterUserName = resourceBundle.getString("parameterUserName");
            } else {
                try {
                    _parameterUserName = ArgumentParser.getParameter(parameter, PARAMETER_USER_NAME_KEY);
                    if ((_parameterUserName == null) || (_parameterUserName.isEmpty())) {
                        throw new MissingParameterException(
                            "parametrierungsBenutzer-Parameter muss folgende Formatierung besitzen: -parametrierungsBenutzer=Zeichenkette");
                    }
                } catch (InvalidArgumentException ex) {
                    throw new MissingParameterException(
                        "parametrierungsBenutzer-Parameter muss folgende Formatierung besitzen: -parametrierungsBenutzer=Zeichenkette");
                }
            }

            //Parameter password
            if (_parameterClientCredentials == null) {
                if (_userProperties == null) {
                    if (parameter == null) {
                        _parameterClientCredentials = ClientCredentials.ofString(resourceBundle.getString("parameterUserPassword"));
                        DEBUG.warning(
                            "Verwende Standard-Authentifizierungsdaten für Parametrierungsbenutzer. Bitte Benutzer für Parametrierung anlegen und " +
                            "mit Aufrufparameter " + PARAMETER_USER_NAME_KEY + " auswählen.");
                    } else {
                        throw new MissingParameterException("Keine Authentifizierungsdatei angegeben");
                    }
                } else {
                    _parameterClientCredentials = _userProperties.getClientCredentials(_parameterUserName);
                    if (parameter == null && _parameterClientCredentials == null) {
                        _parameterClientCredentials = ClientCredentials.ofString(resourceBundle.getString("parameterUserPassword"));
                        DEBUG.warning(
                            "Verwende Standard-Authentifizierungsdaten für Parametrierungsbenutzer. Bitte Benutzer für Parametrierung anlegen und " +
                            "mit Aufrufparameter " + PARAMETER_USER_NAME_KEY + " auswählen.");
                    }
                }
            }

            // Datenverteiler-Passwort
            if (_userProperties == null) {
                if (userPassword == null) {
                    throw new MissingParameterException("Keine Authentifizierungsdatei angegeben");
                } else {
                    DEBUG.warning(
                        "Es wird das Standard-Passwort für die Datenverteilerauthentifizierung verwendet, eine Authentifizierungsdatei sollte " +
                        "angegeben werden: -authentifizierung=Zeichenkette");
                    _userProperties = new SimpleUserProperties(_userName, ClientCredentials.ofString(userPassword));
                }
            }

            //Data transmitter Id
            parameter = getParameter(startArguments, TRANSMITTER_ID_KEY);
            if (parameter == null) {
                tmp = resourceBundle.getString("Datenverteiler-ID");
                _dataTransmitterId = Long.parseLong(tmp);
            } else {
                try {
                    tmp = ArgumentParser.getParameter(parameter, TRANSMITTER_ID_KEY);
                    if ((tmp == null) || (tmp.isEmpty())) {
                        throw new MissingParameterException(
                            "Datenverteiler-Id-Parameter muss folgende Formatierung besitzen: -datenverteilerId=Zeichenkette");
                    }
                    _dataTransmitterId = Long.parseLong(tmp);
                    if (_dataTransmitterId < 0) {
                        throw new MissingParameterException("Datenverteiler-Id-Parameter muss groesser 0 sein");
                    }
                } catch (InvalidArgumentException | NumberFormatException ex) {
                    throw new MissingParameterException(
                        "Datenverteiler-Id-Parameter muss folgende Formatierung besitzen: -datenverteilerId=Zeichenkette");
                }
            }

            //Dav-Dav-port
            parameter = getParameter(startArguments, DAV_DAV_PORT_KEY);
            if (parameter == null) {
                tmp = resourceBundle.getString("Dav-Dav-Subadresse");
                _transmitterConnectionsSubAddress = Integer.parseInt(tmp);
            } else {
                try {
                    tmp = ArgumentParser.getParameter(parameter, DAV_DAV_PORT_KEY);
                    if ((tmp == null) || (tmp.isEmpty())) {
                        throw new MissingParameterException(
                            "Datenverteiler-Datenverteiler-Port-Parameter muss folgende Formatierung besitzen: -davDavPort=Zahl");
                    }
                    _transmitterConnectionsSubAddress = Integer.parseInt(tmp);
                } catch (InvalidArgumentException | NumberFormatException ex) {
                    throw new MissingParameterException(
                        "Datenverteiler-Datenverteiler-Port-Parameter muss folgende Formatierung besitzen: -davDavPort=Zahl");
                }
            }
            if (_transmitterConnectionsSubAddress < 0) {
                throw new MissingParameterException("Die Subadresse muss grösser gleich 0 sein");
            }

            //Dav-Dav-Port-Offset
            parameter = getParameter(startArguments, DAV_DAV_PORT_OFFSET_KEY);
            if (parameter != null) {
                try {
                    tmp = ArgumentParser.getParameter(parameter, DAV_DAV_PORT_OFFSET_KEY);
                    if ((tmp == null) || (tmp.isEmpty())) {
                        throw new MissingParameterException(
                            "Datenverteiler-Datenverteiler-Port-Parameter muss folgende Formatierung besitzen: -davDavPortOffset=Zahl");
                    }
                    _transmitterConnectionsSubAddressOffset = Integer.parseInt(tmp);
                } catch (InvalidArgumentException | NumberFormatException ex) {
                    throw new MissingParameterException(
                        "Datenverteiler-Datenverteiler-Port-Parameter muss folgende Formatierung besitzen: -davDavPortOffset=Zahl");
                }
            }

            //Dav-Daf-port
            parameter = getParameter(startArguments, DAV_APP_PORT_KEY);
            if (parameter == null) {
                tmp = resourceBundle.getString("Dav-Daf-Subadresse");
                _applicationConnectionsSubAddress = Integer.parseInt(tmp);
            } else {
                try {
                    tmp = ArgumentParser.getParameter(parameter, DAV_APP_PORT_KEY);
                    if ((tmp == null) || (tmp.isEmpty())) {
                        throw new MissingParameterException(
                            "Datenverteiler-Applikation-Port-Parameter muss folgende Formatierung besitzen: -davAppPort=Zahl");
                    }
                    _applicationConnectionsSubAddress = Integer.parseInt(tmp);
                } catch (InvalidArgumentException | NumberFormatException ex) {
                    throw new MissingParameterException(
                        "Datenverteiler-Applikation-Port-Parameter muss folgende Formatierung besitzen: -davAppPort=Zahl");
                }
            }
            if (_applicationConnectionsSubAddress < 0) {
                throw new MissingParameterException("Die Subadresse muss grösser gleich 0 sein");
            }

            //Connection to neighbours time out
            parameter = getParameter(startArguments, NEIBOUR_CONNECTION_TIMEOUT_KEY);
            if (parameter == null) {
                tmp = resourceBundle.getString("NeighbourConnectionTimeOut");
                long connectionTime = Long.parseLong(tmp);
                if (connectionTime != 0) {
                    if (connectionTime < 1000) {
                        throw new MissingParameterException("Timeouts müssen grösser gleich als 1 Sekunde sein");
                    }
                    CommunicationConstant.MAX_WAITING_TIME_FOR_CONNECTION = connectionTime;
                } else {
                    // es wird wieder der ursprüngliche Wert (10.000.000ms) gesetzt.
                    CommunicationConstant.MAX_WAITING_TIME_FOR_CONNECTION = 10000000L;
                }
            } else {
                try {
                    tmp = ArgumentParser.getParameter(parameter, NEIBOUR_CONNECTION_TIMEOUT_KEY);
                    if ((tmp == null) || (tmp.isEmpty())) {
                        throw new MissingParameterException(
                            "Nachbardatenverteiler-Verbindung-Timeout-Parameter muss folgende Formatierung besitzen: -timeoutNachbar=Zahl");
                    }
                    long connectionTime = Long.parseLong(tmp) * 1000L;
                    if (connectionTime < 1000) {
                        throw new MissingParameterException("Timeouts müssen grösser gleich als 1 Sekunde sein");
                    }
                    CommunicationConstant.MAX_WAITING_TIME_FOR_CONNECTION = connectionTime;
                } catch (InvalidArgumentException | NumberFormatException ex) {
                    throw new MissingParameterException(
                        "Nachbardatenverteiler-Verbindung-Timeout-Parameter muss folgende Formatierung besitzen: -timeoutNachbar=Zahl");
                }
            }

            //Sync answer time out
            parameter = getParameter(startArguments, SYNC_RESPONCE_TIMEOUT_KEY);
            if (parameter == null) {
                tmp = resourceBundle.getString("SyncAnswerTimeOut");
                long responceTime = Long.parseLong(tmp);
                // Wenn der Wert aus der Resource gleich 0 ist, dann wird der Defaultwert aus CommunicationConstant benutzt.
                if (responceTime != 0) {
                    if (responceTime < 1000) {
                        throw new MissingParameterException("Timeouts müssen grösser gleich als 1 Sekunde sein");
                    }
                    CommunicationConstant.MAX_WAITING_TIME_FOR_SYNC_RESPONCE = responceTime;
                } else {
                    // es wird wieder der ursprüngliche Wert (600.000ms = 10 min) gesetzt.
                    CommunicationConstant.MAX_WAITING_TIME_FOR_SYNC_RESPONCE = 600000L;
                }
            } else {
                try {
                    tmp = ArgumentParser.getParameter(parameter, SYNC_RESPONCE_TIMEOUT_KEY);
                    if ((tmp == null) || (tmp.isEmpty())) {
                        throw new MissingParameterException(
                            "Synchrone-Anwort-Timeout-Parameter muss folgende Formatierung besitzen: -timeoutAntwort=Zahl");
                    }
                    long responseTime = Long.parseLong(tmp) * 1000L;
                    if (responseTime < 1000) {
                        throw new MissingParameterException("Timeouts müssen grösser gleich als 1 Sekunde sein");
                    }
                    CommunicationConstant.MAX_WAITING_TIME_FOR_SYNC_RESPONCE = responseTime;
                } catch (InvalidArgumentException | NumberFormatException ex) {
                    throw new MissingParameterException(
                        "Synchrone-Anwort-Timeout-Parameter muss folgende Formatierung besitzen: -timeoutAntwort=Zahl");
                }
            }

            _dataTransmitterName = resourceBundle.getString("Datenverteilersname");
            _dataTransmitterTypePid = resourceBundle.getString("Datenverteilerstyp-PID");
            _lowLevelCommunicationName = resourceBundle.getString("KommunikationProtokollName");
            try {
                Class.forName(_lowLevelCommunicationName);
            } catch (ClassNotFoundException ex) {
                throw new MissingParameterException("Die Kommunikationsverfahrensklasse existiert nicht");
            }
            tmp = resourceBundle.getString("Sendepuffergrösse");
            _appCommunicationOutputBufferSize = Integer.parseInt(tmp);
            _davCommunicationOutputBufferSize = _appCommunicationOutputBufferSize * 2;
            tmp = resourceBundle.getString("Empfangspuffergrösse");
            _appCommunicationInputBufferSize = Integer.parseInt(tmp);
            _davCommunicationInputBufferSize = _appCommunicationInputBufferSize * 2;
            tmp = resourceBundle.getString("SendeVerzögerung");
            _communicationSendFlushDelay = Long.parseLong(tmp);
            if (_communicationSendFlushDelay > 0) {
                CommunicationConstant.MAX_SEND_DELAY_TIME = _communicationSendFlushDelay;
            }
            tmp = resourceBundle.getString("MaxTelegrammGrösse");
            _maxTelegramSize = Integer.parseInt(tmp);
            if (_maxTelegramSize > 0) {
                CommunicationConstant.MAX_SPLIT_THRESHOLD = _maxTelegramSize;
            } else {
                throw new MissingParameterException("Die maximale Telegrammlänge muss grösser 0 sein");
            }
        } catch (MissingResourceException ex) {
            ex.printStackTrace();
            throw new MissingParameterException(ex.getMessage());
        }
    }

    /**
     * Bestimmt das in der Passwort-Datei gespeicherte Passwort eines bestimmten Benutzers.
     *
     * @param userName Name des Benutzers
     * @param suffix   Verbindungsziel (z.B. Datenverteiler-Pid)
     *
     * @return Passwort des Benutzers oder {@code null}, wenn kein Passwort für den Benutzer in der Passwort-Datei enthalten ist.
     */
    @Nullable
    public ClientCredentials getStoredClientCredentials(String userName, final String suffix) {
        if (_userProperties == null) {
            DEBUG.warning("Lokale Passwort-Datei nicht verfügbar. Sie sollte mit dem Aufrufargument -authentifizierung= angegeben werden.");
            return null;
        }
        return _userProperties.getClientCredentials(userName, suffix);
    }

    /**
     * Sucht in dem angegebenen Feld nach dem Parameter, der mit dem Schlüssel anfängt.
     *
     * @param arguments Feld von Startargumenten
     * @param key       der Schlüssel
     *
     * @return Der Wert zum angegebenen Schlüssel oder {@code null}, falls kein Wert hierzu existiert.
     */
    private String getParameter(String[] arguments, String key) {
        String parameter = null;
        if ((arguments == null) || (key == null)) {
            return null;
        }
        for (int i = 0; i < arguments.length; ++i) {
            String tmp = arguments[i];
            if (tmp == null) {
                continue;
            }
            if (tmp.startsWith(key)) {
                parameter = tmp;
                arguments[i] = null;
                break;
            }
        }
        return parameter;
    }

    /**
     * Bestimmt die maximale Größe von Datentelegrammen. Größere Telegramme werden in mehrere Telegramme zerlegt.
     *
     * @return maxTelegramSize  Maximale Größe von versendeten Datentelegrammen.
     */
    public int getMaxDataTelegramSize() {
        return _maxTelegramSize;
    }

    /**
     * Setzt die maximale Größe von Datentelegrammen. Größere Telegramme werden in mehrere Telegramme zerlegt.
     *
     * @param maxTelegramSize Maximale Größe von versendeten Datentelegrammen.
     */
    public void setMaxDataTelegramSize(int maxTelegramSize) {
        if (maxTelegramSize > 0) {
            _maxTelegramSize = maxTelegramSize;
        }
    }

    /**
     * Liefert die Subadresse mit der dieser Datenverteiler auf Verbindungen von anderen Datenverteilern wartet. Dies entspricht bei TCP-Verbindungen
     * der TCP-Portnummer des Server-Sockets.
     *
     * @return Subadresse mit der dieser Datenverteiler auf Verbindungen von anderen Datenverteilern wartet.
     */
    public int getTransmitterConnectionsSubAddress() {
        return _transmitterConnectionsSubAddress;
    }

    /**
     * Setzt die Subadresse mit der dieser Datenverteiler auf Verbindungen von anderen Datenverteilern wartet.
     *
     * @param port Subadresse mit der dieser Datenverteiler auf Verbindungen von anderen Datenverteilern wartet.
     */
    public void setTransmitterConnectionsSubAddress(int port) {
        _transmitterConnectionsSubAddress = port;
    }

    /**
     * Liefert die Subadresse mit der dieser Datenverteiler auf Verbindungen von anderen Datenverteilern wartet. Dies entspricht bei TCP-Verbindungen
     * der TCP-Portnummer des Server-Sockets.
     *
     * @return Subadresse mit der dieser Datenverteiler auf Verbindungen von anderen Datenverteilern wartet.
     *
     * @deprecated Statt dieser Methode sollte die Methode {@link #getTransmitterConnectionsSubAddress()} verwendet werden.
     */
    @Deprecated
    public int getTransmitterConnectionsSubAdress() {
        return getTransmitterConnectionsSubAddress();
    }

    /**
     * Setzt die Subadresse mit der dieser Datenverteiler auf Verbindungen von anderen Datenverteilern wartet.
     *
     * @param port Subadresse mit der dieser Datenverteiler auf Verbindungen von anderen Datenverteilern wartet.
     *
     * @deprecated Statt dieser Methode sollte die Methode {@link #setTransmitterConnectionsSubAddress(int)} verwendet werden.
     */
    @Deprecated
    public void setTransmitterConnectionsSubAdress(int port) {
        setTransmitterConnectionsSubAddress(port);
    }

    /**
     * Liefert einen Offset für die Subadresse mit der dieser Datenverteiler auf Verbindungen von anderen Datenverteilern wartet.
     *
     * @return Offset für die Subadresse mit der dieser Datenverteiler auf Verbindungen von anderen Datenverteilern wartet.
     */
    public int getTransmitterConnectionsSubAddressOffset() {
        return _transmitterConnectionsSubAddressOffset;
    }

    /**
     * Liefert die Subadresse mit der dieser Datenverteiler auf Verbindungen von Applikationen wartet. Dies entspricht bei TCP-Verbindungen der
     * TCP-Portnummer des Server-Sockets.
     *
     * @return Subadresse mit der dieser Datenverteiler auf Verbindungen von Applikationen wartet.
     */
    public int getApplicationConnectionsSubAddress() {
        return _applicationConnectionsSubAddress;
    }

    /**
     * Setzt die Subadresse mit der dieser Datenverteiler auf Verbindungen von Applikationen wartet.
     *
     * @param port Subadresse mit der dieser Datenverteiler auf Verbindungen von Applikationen wartet.
     */
    public void setApplicationConnectionsSubAddress(int port) {
        _applicationConnectionsSubAddress = port;
    }

    /**
     * Liefert die Subadresse mit der dieser Datenverteiler auf Verbindungen von Applikationen wartet. Dies entspricht bei TCP-Verbindungen der
     * TCP-Portnummer des Server-Sockets.
     *
     * @return Subadresse mit der dieser Datenverteiler auf Verbindungen von Applikationen wartet.
     *
     * @deprecated Statt dieser Methode sollte die Methode {@link #getApplicationConnectionsSubAddress} verwendet werden.
     */
    @Deprecated
    public int getApplicationConnectionsSubAdress() {
        return getApplicationConnectionsSubAddress();
    }

    /**
     * Setzt die Subadresse mit der dieser Datenverteiler auf Verbindungen von Applikationen wartet.
     *
     * @param port Subadresse mit der dieser Datenverteiler auf Verbindungen von Applikationen wartet.
     *
     * @deprecated Statt dieser Methode sollte die Methode {@link #setApplicationConnectionsSubAddress(int)} verwendet werden.
     */
    @Deprecated
    public void setApplicationConnectionsSubAdress(int port) {
        setApplicationConnectionsSubAddress(port);
    }

    /**
     * Gibt die Id des Datenverteilers zurück
     *
     * @return die Datenverteiler Id
     */
    public long getDataTransmitterId() {
        return _dataTransmitterId;
    }

    /**
     * Setzt die Id der Datenverteiler auf den neuen Wert
     *
     * @param dvId neue Datenverteiler Id
     */
    public void setDataTransmitterId(long dvId) {
        _dataTransmitterId = dvId;
    }

    /**
     * Bestimmt den Namen des Datenverteilers.
     *
     * @return applicationName  Name des zu erzeugenden Applikation-Objekts
     */
    public String getDataTransmitterName() {
        return _dataTransmitterName;
    }

    /**
     * Setzt den Namen des Datenverteilers.
     *
     * @param dataTransmitterName Name des Datenverteilers
     */
    public void setDataTransmitterName(String dataTransmitterName) {
        if (dataTransmitterName != null) {
            _dataTransmitterName = dataTransmitterName;
        }
    }

    /**
     * Bestimmt den Typ des Datenverteilers.
     *
     * @return dataTransmitterTypePid  PID, die den Typ des Datenverteilers.
     */
    public String getDataTransmitterTypePid() {
        return _dataTransmitterTypePid;
    }

    /**
     * Setzt den Typ des Datenverteilers.
     *
     * @param dataTransmitterTypePid PID, die den Typ des zu erzeugenden Applikations-Objekts spezifiziert.
     */
    public void setDataTransmitterTypePid(String dataTransmitterTypePid) {
        if (dataTransmitterTypePid != null) {
            _dataTransmitterTypePid = dataTransmitterTypePid;
        }
    }

    /**
     * Bestimmt das bei der Authentifizierung zu verwendende Verfahren.
     *
     * @return authentificationProcessName  Name des Verfahrens
     */
    public String getAuthentificationProcessName() {
        return _authentificationProcessName;
    }

    /**
     * Setzt das bei der Authentifizierung zu verwendende Verfahren.
     *
     * @param authentificationProcessName Name des Verfahrens
     */
    public void setAuthentificationProcessName(String authentificationProcessName) {
        if (authentificationProcessName != null) {
            _authentificationProcessName = authentificationProcessName;
        }
    }

    /**
     * Bestimmt das auf unterster Ebene einzusetzende Kommunikationsprotokoll.
     *
     * @return lowLevelCommunicationName  Name des Kommunikationsverfahrens.
     */
    public String getLowLevelCommunicationName() {
        return _lowLevelCommunicationName;
    }

    /**
     * Setzt das auf unterster Ebene einzusetzende Kommunikationsprotokoll.
     *
     * @param lowLevelCommunicationName Name des Kommunikationsverfahrens.
     */
    public void setLowLevelCommunicationName(String lowLevelCommunicationName) {
        if (lowLevelCommunicationName != null) {
            _lowLevelCommunicationName = lowLevelCommunicationName;
        }
    }

    /**
     * Bestimmt den bei der Authentifizierung zu verwendenden Benutzernamen.
     *
     * @return userName  Name des Benutzers.
     */
    public String getUserName() {
        return _userName;
    }

    /**
     * Setzt den bei der Authentifizierung zu verwendenden Benutzernamen.
     *
     * @param userName Name des Benutzers.
     */
    public void setUserName(String userName) {
        if (userName != null) {
            _userName = userName;
        }
    }

    /**
     * Bestimmt das Timeout zum Senden von KeepAlive-Telegrammen. Der Wert dient als Vorschlag für die Verhandlung mit dem Datenverteiler, der den zu
     * verwendenden Wert festlegt.
     *
     * @return timeout  Vorschlag für das Timeout zum Senden von KeepAlive-Telegrammen.
     */
    public long getSendKeepAliveTimeout() {
        return _sendKeepAliveTimeout;
    }

    /**
     * Setzt das Timeout zum Senden von KeepAlive-Telegrammen. Der Wert dient als Vorschlag für die Verhandlung mit dem Datenverteiler, der den zu
     * verwendenden Wert festlegt.
     *
     * @param timeout Vorschlag für das Timeout zum Senden von KeepAlive-Telegrammen.
     */
    public void setSendKeepAliveTimeout(long timeout) {
        if (timeout > 0) {
            _sendKeepAliveTimeout = timeout;
        }
    }

    /**
     * Bestimmt das KeepAlive-Timeout beim Empfang von Telegrammen. Der Wert dient als Vorschlag für die Verhandlung mit dem Datenverteiler, der den
     * zu verwendenden Wert festlegt.
     *
     * @return timeout  Vorschlag für das KeepAlive-Timeout beim Empfang Telegrammen.
     */
    public long getReceiveKeepAliveTimeout() {
        return _receiveKeepAliveTimeout;
    }

    /**
     * Setzt das KeepAlive-Timeout beim Empfang von Telegrammen. Der Wert dient als Vorschlag für die Verhandlung mit dem Datenverteiler, der den zu
     * verwendenden Wert festlegt.
     *
     * @param timeout Vorschlag für das KeepAlive-Timeout beim Empfang Telegrammen.
     */
    public void setReceiveKeepAliveTimeout(long timeout) {
        if (timeout > 0) {
            _receiveKeepAliveTimeout = timeout;
        }
    }

    /**
     * Bestimmt die Verzögerungszeit zur Übertragung von gepufferten und zu versendenden Telegrammen. Die Übertragung der gesammelten Daten im
     * Sendepuffer findet erst statt, wenn die hier angegebene Zeit lang keine Daten mehr in der Puffer geschrieben wurden oder der Sendepuffer voll
     * ist.
     *
     * @return Verzögerungszeit
     */
    public long getCommunicationSendFlushDelay() {
        return _communicationSendFlushDelay;
    }

    /**
     * Setzt die Verzögerungszeit zur Übertragung von gepufferten und zu versendenden Telegrammen. Die Übertragung der gesammelten Daten im
     * Sendepuffer findet erst statt, wenn die hier angegebene Zeit lang keine Daten mehr in der Puffer geschrieben wurden oder der Sendepuffer voll
     * ist.
     *
     * @param delay Verzögerungszeit
     */
    public void setCommunicationSendFlushDelay(long delay) {
        if (delay > 0) {
            _communicationSendFlushDelay = delay;
        }
    }

    /**
     * Bestimmt die Größe des Sendepuffers, der bei der Kommunikation mit dem Datenverteiler eingesetzt wird.
     *
     * @return bufferSize Größe des Sendepuffers in Byte.
     */
    public int getDavCommunicationOutputBufferSize() {
        return _davCommunicationOutputBufferSize;
    }

    /**
     * Setzt die Größe des Sendepuffers, der bei der Kommunikation mit dem Datenverteiler eingesetzt wird.
     *
     * @param bufferSize Größe des Sendepuffers in Byte.
     */
    public void setDavCommunicationOutputBufferSize(int bufferSize) {
        if (bufferSize > 0) {
            _davCommunicationOutputBufferSize = bufferSize;
        }
    }

    /**
     * Bestimmt die Größe des Empfangspuffers, der bei der Kommunikation mit dem Datenverteiler eingesetzt wird.
     *
     * @return bufferSize Größe des Empfangspuffers in Byte.
     */
    public int getDavCommunicationInputBufferSize() {
        return _davCommunicationInputBufferSize;
    }

    /**
     * Setzt die Größe des Empfangspuffers, der bei der Kommunikation mit dem Datenverteiler eingesetzt wird.
     *
     * @param bufferSize Größe des Empfangspuffers in Byte.
     */
    public void setDavCommunicationInputBufferSize(int bufferSize) {
        if (bufferSize > 0) {
            _davCommunicationInputBufferSize = bufferSize;
        }
    }

    /**
     * Bestimmt die Größe des Sendepuffers, der bei der Kommunikation mit einer Applikation eingesetzt wird.
     *
     * @return bufferSize Größe des Sendepuffers in Byte.
     */
    public int getAppCommunicationOutputBufferSize() {
        return _appCommunicationOutputBufferSize;
    }

    /**
     * Setzt die Größe des Sendepuffers, der bei der Kommunikation mit einer Applikation eingesetzt wird.
     *
     * @param bufferSize Größe des Sendepuffers in Byte.
     */
    public void setAppCommunicationOutputBufferSize(int bufferSize) {
        if (bufferSize > 0) {
            _appCommunicationOutputBufferSize = bufferSize;
        }
    }

    /**
     * Bestimmt die Größe des Empfangspuffers, der bei der Kommunikation mit einer Applikation eingesetzt wird.
     *
     * @return bufferSize Größe des Empfangspuffers in Byte.
     */
    public int getAppCommunicationInputBufferSize() {
        return _appCommunicationInputBufferSize;
    }

    /**
     * Setzt die Größe des Empfangspuffers, der bei der Kommunikation mit einer Applikation eingesetzt wird.
     *
     * @param bufferSize Größe des Empfangspuffers in Byte.
     */
    public void setAppCommunicationInputBufferSize(int bufferSize) {
        if (bufferSize > 0) {
            _appCommunicationInputBufferSize = bufferSize;
        }
    }

    /**
     * Gibt die Information zurück, ob der Datenverteiler auf eine lokale Anmeldung einer Konfigurationsapplikation warten muss.
     *
     * @return true  : im lokalen Konfigurationsbetrieb. false : implements remote Konfigurationsbetrieb.
     */
    public boolean isLocalMode() {
        return _localConfiguration;
    }

    /**
     * Gibt die Konfigurationsparameter des Lokalen Modus zurück.
     *
     * @return die Pid und die Id der Konfigurationsapplikation
     */
    public Object[] getLocalModeParameter() {
        Object[] objs = null;
        if (_localConfiguration) {
            objs = new Object[2];
            objs[0] = _configurationPid;
            objs[1] = _configurationId;
        }
        return objs;
    }

    /**
     * Setzt den Datenverteilersbetriebsmodus auf den Lokalen Modus.
     *
     * @param configPid die Pid der Konfigurationsapplikation
     * @param configId  die Id der Konfigurationsapplikation
     */
    public void setLocalModeParameter(String configPid, long configId) {
        _localConfiguration = true;
        _configurationPid = configPid;
        if ("null".equals(_configurationPid)) {
            _configurationPid = CommunicationConstant.LOCALE_CONFIGURATION_PID_ALIASE;
        }
        _configurationId = configId;
    }

    /**
     * Gibt die Konfigurationsparameter des Remote-Modus zurück.
     *
     * @return die Konfigurationsparameter des Remote-Modus
     */
    public Object[] getRemoteModeParameter() {
        Object[] objs = null;
        if (!_localConfiguration) {
            objs = new Object[3];
            objs[0] = _configDataTransmitterAddress;
            objs[1] = _configDataTransmitterSubAddress;
            objs[2] = _configurationPid;
        }
        return objs;
    }

    /**
     * Setzt den Datenverteilersbetriebsmodus auf den Remote-Modus.
     *
     * @param configDataTransmitterAddress    die Adresse des Datenverteilers wo die Konfiguration angemeldet ist.
     * @param configDataTransmitterSubAddress Datenverteilersubadresse für die Konfigurationsanbindung
     * @param configurationPid                Pid der Konfiguration
     */
    public void setRemoteModeParameter(String configDataTransmitterAddress, int configDataTransmitterSubAddress, String configurationPid) {
        _localConfiguration = false;
        _configDataTransmitterAddress = configDataTransmitterAddress;
        _configDataTransmitterSubAddress = configDataTransmitterSubAddress;
        _configurationPid = configurationPid;
        if ("null".equals(_configurationPid)) {
            _configurationPid = CommunicationConstant.LOCALE_CONFIGURATION_PID_ALIASE;
        }
    }

    /**
     * Gibt die Konfigurationsid zurück
     *
     * @return die Konfigurationsid
     */
    public long getConfigurationId() {
        return _configurationId;
    }

    /**
     * Setzt der Konfigurationsid auf den neuen Wert.
     *
     * @param configurationId Konfigurationsid
     */
    public void setConfigurationId(long configurationId) {
        _configurationId = configurationId;
    }

    /**
     * Gibt der Konfigurationsbenutzername zurück
     *
     * @return der Konfigurationsbenutzername
     */
    public String getConfigurationUserName() {
        return _configurationUserName;
    }

    /**
     * Setzt den Konfigurationsbenutzername auf den neuen Wert.
     *
     * @param configUserName Konfigurationsbenutzername
     */
    public void setConfigurationUserName(String configUserName) {
        _configurationUserName = configUserName;
    }

    /**
     * Gibt Passwort oder Login-Token für den Konfigurationsbenutzer zurück
     *
     * @return der Konfigurationsbenutzerpasswort
     */
    public ClientCredentials getConfigurationClientCredentials() {
        return _configurationClientCredentials;
    }

    /**
     * Setzt Passwort oder Login-Token für den Konfigurationsbenutzer
     *
     * @param configurationClientCredentials das Konfigurationsbenutzerpasswort
     */
    public void setConfigurationClientCredentials(ClientCredentials configurationClientCredentials) {
        _configurationClientCredentials = configurationClientCredentials;
    }

    /**
     * Gibt der Parametrierungsbenutzername zurück
     *
     * @return der Parametrierungsbenutzername
     */
    public String getParameterUserName() {
        return _parameterUserName;
    }

    /**
     * Setzt der Parametrierungsbenutzername auf den neuen Wert.
     *
     * @param paramUserName der Parametrierungsbenutzername
     */
    public void setParameterUserName(String paramUserName) {
        _parameterUserName = paramUserName;
    }

    /**
     * Gibt Passwort oder Login-Token für den Parametrierungsbenutzer zurück
     *
     * @return Passwort oder Login-Token für den Parametrierungsbenutzer
     */
    public ClientCredentials getParameterClientCredentials() {
        return _parameterClientCredentials;
    }

    /**
     * Setzt Passwort oder Login-Token für den Parametrierungsbenutzer auf den neuen Wert.
     *
     * @param paramUserPassword das Parametrierungsbenutzerpasswort
     */
    public void setParameterClientCredentials(ClientCredentials paramUserPassword) {
        _parameterClientCredentials = paramUserPassword;
    }

    public List<String> getAccessControlPlugins() {
        return Collections.unmodifiableList(_accessControlPlugins);
    }

    public boolean isUserRightsCheckingEnabled() {
        return _userRightsChecking != AccessControlMode.Disabled;
    }

    public AccessControlMode getUserRightsChecking() {
        return _userRightsChecking;
    }

    void setUserRightsChecking(final AccessControlMode userRightsChecking) {
        _userRightsChecking = userRightsChecking;
    }

    /**
     * Zeit in Millisekunden, die gewartet werden soll bevor Verbindungen von anderen Datenverteilern akzeptiert werden dürfen.
     *
     * @return Zeit in Millisekunden
     */
    public long getInitialInterDavServerDelay() {
        return _initialInterDavServerDelay;
    }

    /**
     * Zeit in Millisekunden, die gewartet werden soll bevor versucht wird, abgebrochene Verbindungen zu anderen Datenverteilern neu aufzubauen.
     *
     * @return Zeit in Millisekunden
     */
    public long getReconnectInterDavDelay() {
        return _reconnectInterDavDelay;
    }

    public void setReconnectInterDavDelay(final long reconnectInterDavDelay) {
        _reconnectInterDavDelay = reconnectInterDavDelay;
    }

    /**
     * Erzeugt einen neuen Parametersatz für eine Applikationsverbindung.
     *
     * @return Parameterobjekt zum Aufbau einer Applikationsverbindung
     *
     * @throws MissingParameterException Bei formalen Fehlern beim Lesen der Aufrufargumente oder der Defaultwerte.
     */
    public ClientDavParameters getClientDavParameters() throws MissingParameterException {
        final String configurationPid;
        final String address;
        final int subAddress;
        final String userName;
        ClientCredentials clientCredentials;
        final String applicationName;
        final String authentificationProcessName;
        final int maxTelegramSize;
        final long receiveKeepAliveTimeout;
        final long sendKeepAliveTimeout;
        final int outputBufferSize;
        final int inputBufferSize;
        final String communicationProtocolName;
        if (isLocalMode()) {
            // Lokale Konfiguration
            Object[] objects = getLocalModeParameter();
            if (objects == null) {
                throw new IllegalStateException("Inkonsistente Parameter.");
            }
            configurationPid = (String) objects[0];
            address = "127.0.0.1";  // localhost über loopback
            subAddress = getApplicationConnectionsSubAddress();
            String name = "TransmitterLocalApplication@" + System.currentTimeMillis();
            userName = name;

            clientCredentials = SrpClientAuthentication.createRandomToken(SrpCryptoParameter.getDefaultInstance());

            if (System.getProperty("srp6.disable.login") != null) {
                // Wenn SRP deaktiviert ist, muss hier ein Klartextpasswort verwendet werden
                clientCredentials = ClientCredentials.ofPassword(SrpUtilities.bytesToHex(clientCredentials.getTokenData()).toCharArray());
            }
            applicationName = name;
        } else {
            // Andere Konfiguration
            Object[] objects = getRemoteModeParameter();
            if (objects == null) {
                throw new IllegalStateException("Inkonsistente Parameter.");
            }
            address = (String) objects[0];
            subAddress = (Integer) objects[1];
            configurationPid = (String) objects[2];
            userName = getUserName();
            clientCredentials = getStoredClientCredentials(userName, configurationPid);
            applicationName = "TransmitterRemoteApplication@" + System.currentTimeMillis();
        }

        authentificationProcessName = getAuthentificationProcessName();
        maxTelegramSize = getMaxDataTelegramSize();
        receiveKeepAliveTimeout = getReceiveKeepAliveTimeout();
        sendKeepAliveTimeout = getSendKeepAliveTimeout();
        outputBufferSize = getAppCommunicationOutputBufferSize();
        inputBufferSize = getAppCommunicationInputBufferSize();
        communicationProtocolName = getLowLevelCommunicationName();
        ClientDavParameters clientDavParameters =
            new ClientDavParameters(configurationPid, address, subAddress, userName, applicationName, authentificationProcessName, maxTelegramSize,
                                    receiveKeepAliveTimeout, sendKeepAliveTimeout, outputBufferSize, inputBufferSize, communicationProtocolName,
                                    _allowHmacAuthentication, _encryptionPreference) {
                @Override
                public boolean isSelfClientDavConnection() {
                    return true;
                }
            };
        clientDavParameters.setUserProperties(new SimpleUserProperties(userName, clientCredentials));
        // Interne Datenverteilerverbindung darf keine 2. Verbindung benutzen
        clientDavParameters.setUseSecondConnection(false);

        clientDavParameters.setPassivePort(_passivePort);
        clientDavParameters.setPassiveCommunication(_passiveCommunication);

        return clientDavParameters;
    }

    public String getLowLevelCommunicationParameters() {
        return _lowLevelCommunicationParameters;
    }

    /**
     * Bestimmt, ob der Datenverteiler auf die Applikationsfertigmeldung der Parametrierung warten soll.
     *
     * @return {@code true}, falls der Datenverteiler auf die Applikationsfertigmeldung der Parametrierung warten soll
     */
    public boolean getWaitForParamApp() {
        return _waitForParamApp;
    }

    /**
     * Bestimmt den Inkarnationsnamen der Parametrierung auf deren Applikationsfertigmeldung gewartet werden soll.
     *
     * @return Inkarnationsnamen der Parametrierung auf deren Applikationsfertigmeldung gewartet werden soll oder {@code null} falls der
     *     Inkarnationsname egal ist oder nicht gewartet werden soll.
     *
     * @see #getWaitForParamApp()
     */
    public String getParamAppIncarnationName() {
        return _paramAppIncarnationName;
    }

    /**
     * Bestimmt die Pid des Konfigurationsbereichs in dem Applikationsobjekte erzeugt werden sollen.
     *
     * @return Pid des Konfigurationsbereichs in dem Applikationsobjekte erzeugt werden sollen oder Leerstring falls der Default-Bereich der
     *     Konfiguration verwendet werden soll.
     */
    public String getConfigAreaPidForApplicationObjects() {
        return _configAreaPidForApplicationObjects;
    }

    /**
     * Gibt {@code true} zurück, wenn die alte Hmac-basierte Authentifizierung erlaubt ist
     *
     * @return {@code true}, wenn die alte Hmac-basierte Authentifizierung erlaubt ist, sonst {@code false}
     */
    public boolean isHmacAuthenticationAllowed() {
        return _allowHmacAuthentication;
    }

    /**
     * Gibt die bevorzugte Verschlüsselungskonfiguration zurück.
     *
     * @return die bevorzugte Verschlüsselungskonfiguration
     */
    public EncryptionConfiguration getEncryptionPreference() {
        return _encryptionPreference;
    }

    /**
     * Setzt ob die Verbindung verschlüsselt werden soll
     *
     * @param encryptionPreference
     */
    public void setEncryptionPreference(final EncryptionConfiguration encryptionPreference) {
        _encryptionPreference = encryptionPreference;
    }

    /**
     * Setzt, ob die alte Hmac-Authentifizierung erlaubt sein soll
     *
     * @param allowHmacAuthentication
     */
    public void setAllowHmacAuthentication(final boolean allowHmacAuthentication) {
        _allowHmacAuthentication = allowHmacAuthentication;
    }

    public Set<ApplicationInfo> getActiveConnections() {
        return _activeConnections;
    }

    public void setActiveConnections(final Collection<ApplicationInfo> activeConnections) {
        _activeConnections = Set.copyOf(activeConnections);
    }

}
