/*
 * Copyright 2011-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.dav.main;

import de.bsvrz.dav.daf.communication.lowLevel.telegrams.DataTelegramInterface;
import de.bsvrz.dav.dav.subscriptions.SubscriptionInfo;
import de.bsvrz.sys.funclib.debug.Debug;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Diese Klasse dient dazu, aufgeteilte Datentelegramme wieder zusammenzusetzen. Dazu ist für jedes ankommende Telegram die aggregate()-Funktion
 * auszuführen. Sobald alle Telegramm eingetroffen sind, wird eine Liste mit den Telegrammen zurückgegeben, sonst nur eine leere Liste.
 *
 * @author Kappich Systemberatung
 */
public class TelegramAggregator<T extends DataTelegramInterface> {

    private static final Debug _debug = Debug.getLogger();

    private final Map<TelegramIdentification, List<T>> _telegrams = new HashMap<>(256);

    /**
     * Verarbeitet ein ankommendes Telegramm und gibt die Liste der Telegramme zurück, sobald alle Telegramme eingetroffen sind. Es ist daher wichtig,
     * dass alle Telegramme einer Datenidentifikation in der richtigen Reihenfolge eintreffen. Um das zu erreichen sollte pro Anmeldung nur ein
     * einzelner Thread Telegramme eintragen
     *
     * @param telegram      Telegramm
     * @param subscription  Anmeldung
     * @param applicationId ID der sendenden Applikation
     *
     * @return
     */
    public List<T> aggregate(final T telegram, final SubscriptionInfo subscription, final long applicationId) {
        final int totalTelegramsCount = telegram.getTotalTelegramsCount();
        if (totalTelegramsCount == 1) {
            // Nicht gestückeltes Telegram: Einfach zurückgeben.
            return List.of(telegram);
        }
        TelegramIdentification id = new TelegramIdentification(subscription, telegram.getDataNumber(), applicationId);
        synchronized (_telegrams) {
            final int telegramNumber = telegram.getTelegramNumber();
            if (telegramNumber == 0) {
                // Das erste Telegramm eines zerstückelten Datensatzes
                final List<T> stalledTelegramsList = new ArrayList<>(totalTelegramsCount);
                _telegrams.put(id, stalledTelegramsList);
                stalledTelegramsList.add(telegram);
            } else if (telegramNumber + 1 != totalTelegramsCount) {
                // Ein mittleres Telegramm eines zerstückelten Datensatzes
                final List<T> stalledTelegramList = _telegrams.get(id);
                if (stalledTelegramList == null) {
                    _debug.warning("Ein mittleres Telegramm ist ohne erstes Telegramm eingetroffen", telegram.getBaseSubscriptionInfo());
                } else {
                    if (telegramNumber == stalledTelegramList.size()) {
                        stalledTelegramList.add(telegram);
                    } else {
                        _debug.warning("Die Telegramme sind nicht in der richtigen Reihenfolge eingetroffen", telegram.getBaseSubscriptionInfo());
                    }
                }
            } else {
                // Das letzte Telegramm eines zerstückelten Datensatzes
                final List<T> stalledTelegramList = _telegrams.remove(id);
                if (stalledTelegramList == null) {
                    _debug.warning("Das letzte Telegramm ist ohne vorherige Telegramme eingetroffen", telegram.getBaseSubscriptionInfo());
                } else if (telegramNumber != stalledTelegramList.size()) {
                    _debug.warning("Die Telegramme sind nicht in der richtigen Reihenfolge eingetroffen", telegram.getBaseSubscriptionInfo());
                } else {
                    stalledTelegramList.add(telegram);
                    return stalledTelegramList;
                }
            }
            return List.of();
        }
    }

    private static class TelegramIdentification {
        private final SubscriptionInfo _subscriptionInfo;
        private final long _telegramNumber;
        private final long _applicationId;

        public TelegramIdentification(final SubscriptionInfo subscriptionInfo, final long telegramNumber, final long applicationId) {
            _subscriptionInfo = subscriptionInfo;
            _telegramNumber = telegramNumber;
            _applicationId = applicationId;
        }

        public SubscriptionInfo getSubscriptionInfo() {
            return _subscriptionInfo;
        }

        public long getTelegramNumber() {
            return _telegramNumber;
        }

        public long getApplicationId() {
            return _applicationId;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            final TelegramIdentification that = (TelegramIdentification) o;

            if (_telegramNumber != that._telegramNumber) {
                return false;
            }
            if (_applicationId != that._applicationId) {
                return false;
            }
            return _subscriptionInfo.equals(that._subscriptionInfo);
        }

        @Override
        public int hashCode() {
            int result = _subscriptionInfo.hashCode();
            result = 31 * result + Long.hashCode(_telegramNumber);
            result = 31 * result + Long.hashCode(_applicationId);
            return result;
        }
    }
}
