/*
 * Copyright 2011-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.dav.subscriptions;

import de.bsvrz.dav.daf.accessControl.internal.UserAction;
import de.bsvrz.dav.daf.communication.lowLevel.telegrams.ApplicationDataTelegram;
import de.bsvrz.dav.daf.communication.lowLevel.telegrams.BaseSubscriptionInfo;
import de.bsvrz.dav.daf.communication.lowLevel.telegrams.TransmitterSubscriptionType;
import de.bsvrz.dav.daf.communication.protocol.UserLogin;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.dav.main.ConnectionState;
import de.bsvrz.dav.dav.main.SubscriptionsManager;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

/**
 * Anmeldung eines entfernten Empfängers auf diesen Datenverteiler (der möglicherweise Zentraldatenverteiler ist)
 *
 * @author Kappich Systemberatung
 * @version $Revision: 11462 $
 */
public class RemoteReceiverSubscription implements RemoteReceivingSubscription {

    private final SubscriptionsManager _subscriptionsManager;

    private final TransmitterCommunicationInterface _transmitterCommunication;

    private final BaseSubscriptionInfo _baseSubscriptionInfo;

    private final Set<Long> _potentialCentralDistributors = new HashSet<>();

    private ReceiverState _receiverState;

    private ConnectionState _lastSendState;

    private DataState _lastSendDataState;

    public RemoteReceiverSubscription(final SubscriptionsManager subscriptionsManager,
                                      final TransmitterCommunicationInterface transmitterCommunication,
                                      final BaseSubscriptionInfo baseSubscriptionInfo, final Collection<Long> transmitterIds) {
        _subscriptionsManager = subscriptionsManager;
        _transmitterCommunication = transmitterCommunication;
        _baseSubscriptionInfo = baseSubscriptionInfo;
        _potentialCentralDistributors.addAll(transmitterIds);
    }

    @Override
    public boolean isDrain() {
        return false;
    }

    @Override
    public ReceiveOptions getReceiveOptions() {
        return ReceiveOptions.delayed();
    }

    @Override
    public void sendDataTelegram(final ApplicationDataTelegram applicationDataTelegram, final long applicationId) {
        _transmitterCommunication.sendData(applicationDataTelegram, applicationId, false);
        _lastSendDataState = null;
    }

    @Override
    public ReceiverState getState() {
        return _receiverState;
    }

    @Override
    public void setState(final ReceiverState receiverState, final long centralTransmitterId) {
        _receiverState = receiverState;
        ConnectionState receip;
        switch (receiverState) {
            case SENDERS_AVAILABLE:
                receip = ConnectionState.TO_REMOTE_OK;
                break;
            case MULTIPLE_REMOTE_LOCK:
                receip = ConnectionState.TO_REMOTE_MULTIPLE;
                break;
            case NOT_ALLOWED:
                receip = ConnectionState.TO_REMOTE_NOT_ALLOWED;
                break;
            case WAITING:
                return;
            default:
                receip = ConnectionState.TO_REMOTE_NOT_RESPONSIBLE;
                break;
        }
        if (!_potentialCentralDistributors.contains(centralTransmitterId)) {
            // Es ist möglicherweise ein anderer Datenverteiler zuständig als angegeben.
            // Hier ein "nicht verantwortlich" zurückmelden, damit keine "Schleifen" entstehen
            receip = ConnectionState.TO_REMOTE_NOT_RESPONSIBLE;
        }
        if (_lastSendState == receip) {
            return;
        }
        _lastSendState = receip;
        _transmitterCommunication.sendReceipt(centralTransmitterId, receip, TransmitterSubscriptionType.Receiver, this);
    }

    /**
     * Sendet an einen Empfänger falls nötig im Falle eines geänderten Anmeldestatus einen leeren Datensatz mit dem entsprechenden Inhalt.
     *
     * @param receiverState Empfängerstatus
     */
    @Override
    public void sendStateTelegram(final ReceiverState receiverState) {
	    final DataState newState = switch (receiverState) {
		    case NO_SENDERS -> DataState.NO_SOURCE;
		    case NOT_ALLOWED -> DataState.NO_RIGHTS;
		    case INVALID_SUBSCRIPTION -> DataState.INVALID_SUBSCRIPTION;
		    default -> throw new IllegalArgumentException(receiverState.toString());
	    };
        if (_lastSendDataState == newState) {
            return; // Keine mehrfachen Statusnachrichten senden
        }
        _lastSendDataState = newState;

        byte errorFlag = (byte) (newState.getCode() - 1);

        long dataIndex = _subscriptionsManager.getCurrentDataIndex(_baseSubscriptionInfo);

        _transmitterCommunication.sendData(
            new ApplicationDataTelegram(_baseSubscriptionInfo, dataIndex + 1, false, errorFlag, null, null, 1, 0, System.currentTimeMillis()),
            _subscriptionsManager.getThisTransmitterId(), false);
    }

    @Override
    public void unsubscribe() {
        setState(ReceiverState.UNKNOWN, -1);
    }

    @Override
    public BaseSubscriptionInfo getBaseSubscriptionInfo() {
        return _baseSubscriptionInfo;
    }

    @Override
    public boolean isAllowed() {
        return _subscriptionsManager.isActionAllowed(getAuthenticationState(), _baseSubscriptionInfo, UserAction.RECEIVER);
    }

    @Override
    public UserLogin getAuthenticationState() {
        return _transmitterCommunication.getUserLogin();
    }

    @Override
    public long getNodeId() {
        return _transmitterCommunication.getId();
    }

    @Override
    public ConnectionState getConnectionState() {
        return ConnectionState.FROM_REMOTE_OK;
    }

    @Override
    public long getCentralDistributorId() {
        return -1;
    }

    @Override
    public TransmitterCommunicationInterface getCommunication() {
        return _transmitterCommunication;
    }

    @Override
    public String toString() {
        return "Eingehende Anmeldung (" + _receiverState + ") als " + "Empfänger" + " auf " +
               _subscriptionsManager.subscriptionToString(_baseSubscriptionInfo) + " über " + _transmitterCommunication + " (Benutzer=" +
               _subscriptionsManager.objectToString(getAuthenticationState().toLong()) + ")";
    }

    @Override
    public Set<Long> getPotentialDistributors() {
        return Collections.unmodifiableSet(_potentialCentralDistributors);
    }

    @Override
    public void setPotentialDistributors(final Collection<Long> value) {
        _potentialCentralDistributors.clear();
        _potentialCentralDistributors.addAll(value);
    }

    @Override
    public void addPotentialDistributor(final long transmitterId) {
        _potentialCentralDistributors.add(transmitterId);
    }

    @Override
    public void removePotentialDistributor(final long transmitterId) {
        _potentialCentralDistributors.remove(transmitterId);
    }
}
