/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.accessControl.internal;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Hilfsklasse, die von einem DataLoader-Objekt die referenzierten Unterklassen ermittelt und auf eventuelle Rekursionsprobleme hinweist
 *
 * @author Kappich Systemberatung
 * @version $Revision: 0000 $
 */
class ChildrenTreeEnumerator {

    private final List<DataLoader> _alreadyVisitedList = new ArrayList<>();

    private final List<DataLoader> _alreadyFinishedList = new ArrayList<>();

    private final DafAccessControlManager _accessControlManager;

    private final DataLoader _node;

    /**
     * @param accessControlManager AccessControlManager, der über eventuelle Rekursionsprobleme informiert wird
     * @param node                 Objekt das nach Kindelementen gefragt wird
     */
    public ChildrenTreeEnumerator(final DafAccessControlManager accessControlManager, final DataLoader node) {
        _accessControlManager = accessControlManager;
        _node = node;
    }

    /**
     * Gibt alle Kindelemente eines Objekts zurück
     *
     * @return Liste mit Kindelementen
     */
    protected List<DataLoader> enumerateChildren() {
        enumerateChildrenInternal(_node);
        return Collections.unmodifiableList(_alreadyFinishedList);
    }

    /**
     * Interne rekursiv aufgerufene Hilfsfunktion zum Auflisten von Kindelementen
     *
     * @param node Objekt das nach Kindelementen gefragt wird
     */
    private void enumerateChildrenInternal(DataLoader node) {
        if (_alreadyFinishedList.contains(node)) {
            return;
        }
        if (_alreadyVisitedList.contains(node)) {
            final ArrayList<DataLoader> trace = new ArrayList<>(_alreadyVisitedList);
            trace.removeAll(_alreadyFinishedList);
            trace.add(node);
            _accessControlManager.notifyInfiniteRecursion(node, _alreadyVisitedList.get(_alreadyVisitedList.size() - 1), trace);
        } else {
            _alreadyVisitedList.add(node);
            for (DataLoader dataLoader : node.getChildObjects()) {
                enumerateChildrenInternal(dataLoader);
            }
        }
        _alreadyFinishedList.add(node);
    }

    @Override
    public String toString() {
        return "ChildrenTreeEnumerator{" + "_node=" + _node + '}';
    }
}
