/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.accessControl.internal;

import de.bsvrz.dav.daf.accessControl.AccessControlChangeListener;
import de.bsvrz.dav.daf.accessControl.AccessControlManager;
import de.bsvrz.dav.daf.accessControl.RegionManager;
import de.bsvrz.dav.daf.accessControl.UserInfo;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * AccessControlManager-Implementierung, die die Benutzerrechteverwaltung und Rechteprüfung durchführt.
 *
 * @author Kappich Systemberatung
 */
public class DafAccessControlManager implements AccessControlManager, RegionManager, Closeable {
    /** Debug */
    protected static final Debug _debug = Debug.getLogger();
    /**
     * Spezielles Long, dass das Töten des Threads bewirkt.
     */
    private static final Long POISON = 0L;

    /** Map, die BenutzerIds den Benutzerobjekten zuordnet */
    protected final HashMap<Long, UserInfoInternal> _userInfoHashMap = new HashMap<>();

    /** Map, die Berechtigungsklassen den kapselnden AccessControlUnit-Klassen zuordnet */
    protected final HashMap<SystemObject, AccessControlUnit> _authenticationClassHashMap = new HashMap<>();

    /** Map, die Rollen den kapselnden Role-Klassen zuordnet */
    protected final HashMap<SystemObject, Role> _roleHashMap = new HashMap<>();

    /** Map, die Regionen den kapselnden Region-Klassen zuordnet */
    protected final HashMap<SystemObject, Region> _regionHashMap = new HashMap<>();

    /** Datenverteilerverbindung */
    protected final ClientDavInterface _connection;
    /** Ob implizite Benutzerverwaltung durchgeführt wird, oder Benutzer mit addUser erstellt werden müssen */
    protected final boolean _useImplicitUserManagement;
    /**
     * Lock-Objekt für {@link #_userInfoHashMap}
     */
    protected final ReentrantReadWriteLock _userMapLock = new ReentrantReadWriteLock();
    /** Callback, der aufgerufen wird, wenn sich die Rechte eines Benutzers ändern */
    private final Set<AccessControlChangeListener> _changeListeners = new CopyOnWriteArraySet<>();
    /**
     * Queue für Benachrichtigungen von geänderten Benutzerparametern (damit mehrere Änderungen zusammengepackt bearbeitet werden)
     */
    private final LinkedBlockingQueue<Long> _notifyUserChangedQueue = new LinkedBlockingQueue<>();

    /**
     * Lock-Objekt, siehe {@link #getUpdateLock()}
     */
    private final Object _updateLock = new Object();

    public DafAccessControlManager(final boolean useNewDataModel, final ClientDavInterface connection, final boolean useImplicitUserManagement) {
        if (useNewDataModel && connection.getDataModel().getObject("atl.aktivitätObjekteNeu") == null) {
            throw new IllegalArgumentException(
                    "Das neue Datenmodell der Zugriffsrechte-Prüfung sollte verwendet werden, wurde aber nicht gefunden.");
        } else if (!useNewDataModel) {
            throw new IllegalArgumentException("Die Nutzung der alten Rechteprüfung ist veraltet und nicht mehr möglich.");
        }
        _connection = connection;
        _useImplicitUserManagement = useImplicitUserManagement;
        final Thread refreshThread = new Thread("Aktualisierung Benutzerrechte") {
            @Override
            public void run() {
                while (!interrupted()) {
                    try {
                        Long userId = _notifyUserChangedQueue.take();
                        if (userId.equals(POISON)) {
                            return;
                        }
                        handleUserRightsChanged(userId);
                    } catch (Exception e) {
                        _debug.error("Fehler beim Ändern von Benutzerrechten", e);
                    }
                }
            }
        };
        refreshThread.setDaemon(true);
        refreshThread.start();
    }

    @Override
    public void addChangeListener(final AccessControlChangeListener listener) {
        _changeListeners.add(Objects.requireNonNull(listener));
    }

    @Override
    public void removeChangeListener(final AccessControlChangeListener listener) {
        _changeListeners.remove(Objects.requireNonNull(listener));
    }

    private void handleUserRightsChanged(final long userId) {
        for (AccessControlChangeListener listener : _changeListeners) {
            try {
                listener.userPermissionsChanged(getUserPermissions(userId));
            } catch (Exception ex) {
                _debug.error("Fehler in Listener-Benachrichtigung bei geänderten Benutzerrechten", ex);
            }
        }
    }

    @Override
    public void close() {
        _notifyUserChangedQueue.add(POISON);
        for (Role role : _roleHashMap.values()) {
            role.stopDataListener();
        }
        for (Region region : _regionHashMap.values()) {
            region.stopDataListener();
        }
        for (AccessControlUnit accessControlUnit : _authenticationClassHashMap.values()) {
            accessControlUnit.stopDataListener();
        }
        for (UserInfoInternal userInfoInternal : _userInfoHashMap.values()) {
            userInfoInternal.stopDataListener();
        }
    }

    /**
     * Fügt eine Benutzerinformation zu der Benutzertabelle hinzu, wenn der Datenverteiler die Benutzerrechte prüfen soll. Existiert der Benutzer
     * bereits, wird lediglich die interne Referenz inkrementiert.
     *
     * @param userId BenutzerID
     */
    public final void addUser(final long userId) {
        if (_useImplicitUserManagement) {
            return;
        }
        addUserInternal(userId);
    }

    private UserInfo addUserInternal(final long userId) {
        _userMapLock.writeLock().lock();
        try {
            UserInfoInternal userInfo = _userInfoHashMap.get(userId);
            if (userInfo == null) {
                SystemObject object = _connection.getDataModel().getObject(userId);
                if (object == null || !object.isOfType("typ.benutzer")) {
                    return new DummyAccessControlManager.NoUserPermissions(userId);
                }
                userInfo = createUserInfo(object);
                _userInfoHashMap.put(userId, userInfo);
            } else {
                userInfo.incrementReference();
            }
            return userInfo;
        } finally {
            _userMapLock.writeLock().unlock();
        }
    }

    /**
     * Erstellt je nach Datenmodell-Version ein neues BenutzerInfo-Objekt das Abfragen auf die Berechtigungen eines Benutzers ermöglicht.
     *
     * @param object Benutzerobjekt
     *
     * @return Das Benutzer-Info-Objekt
     */
    private UserInfoInternal createUserInfo(final SystemObject object) {

        return new ExtendedUserInfo(_connection, this, object);
    }

    /**
     * Fragt ab, ob das neue Datenmodell benutzt wird. Das neue Datenmodell enthält eine neue Struktur der Region und Rollen-Objekten und ermöglicht
     * Beschränkungen bei der Erstellung von dynamischen Objekten.
     *
     * @return True wenn das neue Modell benutzt wird, sonst false
     */
    public boolean isUsingNewDataModel() {
        return true;
    }

    /**
     * Wird aufgerufen, wenn eine Rekursion in den Systemobjekten gefunden wurde. Dabei wird eine _Debug-Meldung ausgegeben und das Elternelement
     * angewiesen die Referenz auf das Kindobjekt zu deaktivieren.
     *
     * @param node   Der Knoten, der sich selbst referenziert
     * @param parent Der Knoten, der den problematischen Knoten referenziert
     * @param trace  Komplette Hierarchie vom Benutzer zum problematischen Objekt.
     */
    public void notifyInfiniteRecursion(final DataLoader node, final DataLoader parent, final List<DataLoader> trace) {
        String msg = "Ungültige Rekursion in den Systemobjekten. Die problematische Vererbung wird deaktiviert bis das Problem behoben wird.\n" +
                     "Objekt referenziert sich selbst: " + node + "\n" + "Vererbungskette: " + trace;
        _debug.warning(msg);
        parent.deactivateInvalidChild(node);
    }

    /**
     * Gibt die AuthenticationClass-Klasse zurück die zu dem angeforderten Systemobjekt gehört.
     *
     * @param systemObject Systemobjekt, das eine Berechtigungsklasse repräsentiert
     *
     * @return AuthenticationClass-Klasse die Abfragen auf eine Berechtigungsklasse ermöglicht
     */
    public AccessControlUnit getAuthenticationClass(final SystemObject systemObject) {
        synchronized (_authenticationClassHashMap) {
            AccessControlUnit authenticationClass = _authenticationClassHashMap.get(systemObject);
            if (null != authenticationClass) {
                return authenticationClass;
            }
            authenticationClass = new AccessControlUnit(systemObject, _connection, this);
            _authenticationClassHashMap.put(systemObject, authenticationClass);
            return authenticationClass;
        }
    }

    /**
     * Gibt die Region-Klasse zurück die zu dem angeforderten Systemobjekt gehört.
     *
     * @param systemObject Systemobjekt, das eine Region repräsentiert
     *
     * @return Region-Klasse die Abfragen auf eine Region ermöglicht
     */
    public Region getRegion(final SystemObject systemObject) {
        synchronized (_regionHashMap) {
            Region region = _regionHashMap.get(systemObject);
            if (null != region) {
                return region;
            }
            region = new Region(systemObject, _connection, this);
            _regionHashMap.put(systemObject, region);
            return region;
        }
    }

    /**
     * Gibt die Role-Klasse zurück die zu dem angeforderten Systemobjekt gehört.
     *
     * @param systemObject Systemobjekt, das eine Rolle repräsentiert
     *
     * @return Role-Klasse die Abfragen auf eine Rolle ermöglicht
     */
    public Role getRole(final SystemObject systemObject) {
        synchronized (_roleHashMap) {
            Role role = _roleHashMap.get(systemObject);
            if (null != role) {
                return role;
            }
            role = new Role(systemObject, _connection, this);
            _roleHashMap.put(systemObject, role);
            return role;
        }
    }

    /**
     * Gibt das gespeicherte BenutzerObjekt mit der angegebenen ID zurück
     *
     * @param userId Angegebene BenutzerId
     *
     * @return Das geforderte UserInfo-Objekt
     */
    @Override
    public UserInfo getUserPermissions(final long userId) {
        UserInfoInternal userInfo;
        _userMapLock.readLock().lock();
        try {
            userInfo = _userInfoHashMap.get(userId);
        } finally {
            _userMapLock.readLock().unlock();
        }
        if (_useImplicitUserManagement && userInfo == null) {
            // addUserInternal verwendet _userMapLock.writeLock(). Daher muss das readLock hier freigegeben worden sein,
            return addUserInternal(userId);
        }
        return userInfo;
    }

    @Override
    public UserInfo getUserPermissions() {
        return getUserPermissions(_connection.getLocalUser());
    }

    /**
     * Um immer einen konsistenten Zustand zu haben, darf immer nur ein DataLoader gleichzeitig pro AccessControlManager geupdatet werden. Dazu wird
     * auf dieses dummy-Objekt synchronisiert
     *
     * @return Objekt auf das Synchronisiert werden soll
     */
    @Override
    public Object getUpdateLock() {
        return _updateLock;
    }

    private void notifyUserRightsChangedAsync(final Long affectedUserId) {
        _notifyUserChangedQueue.add(affectedUserId);
    }

    /**
     * Wird aufgerufen un dem AccessControlManager zu informieren, dass ein Benutzer sich geändert hat. Der AccessControlManager wird daraufhin die
     * referenzierten Kindobjekte (Rollen, Regionen etc.) auf Rekursion überprüfen und eine Benachrichtigung senden, dass sich die Rechte des
     * Benutzers geändert haben und eventuelle vorhandene Anmeldungen entfernt werden müssen.
     *
     * @param userInfo Benutzerobjekt, das sich geändert hat
     */
    void userChanged(final UserInfoInternal userInfo) {
	    if (userInfo instanceof DataLoader userAsDataLoader) {
            enumerateChildren(userAsDataLoader); // Prüft auf Rekursion
            long userId = userInfo.getUserId();
            notifyUserRightsChangedAsync(userId);
        }
    }

    /**
     * Gibt alle Kindelemente eines Objekts zurück
     *
     * @param node Objekt das nach Kindelementen gefragt wird
     *
     * @return Liste mit Kindelementen
     */
    private List<DataLoader> enumerateChildren(final DataLoader node) {
        return new ChildrenTreeEnumerator(this, node).enumerateChildren();
    }

    /**
     * Prüft ob ein Objekt wie eine Rolle oder eine Region von einem übergeordnetem Objekt wie einem Benutzer oder einer Berechtigungsklasse
     * referenziert wird.
     *
     * @param parent        Mögliches Vaterobjekt
     * @param possibleChild Möglichen Kindobjekt
     *
     * @return True wenn das possibleChild ein Kind von parent ist.
     */
    private boolean isChildOf(final DataLoader parent, final DataLoader possibleChild) {
        final List<DataLoader> children = enumerateChildren(parent);
        return children.contains(possibleChild);
    }

    /**
     * Wird aufgerufen un dem AccessControlManager zu informieren, dass ein verwaltetes Objekt sich geändert hat. Der AccessControlManager wird
     * daraufhin nach Benutzer-Objekten suchen, die dieses Objekt verwenden und eine Benachrichtigung senden, dass sich die Rechte des Benutzers
     * geändert haben und eventuelle vorhandene Anmeldungen entfernt werden müssen.
     *
     * @param object Objekt das sich geändert hat
     */
    public void objectChanged(final DataLoader object) {
        final List<Long> affectedUserIds = new ArrayList<>();
        _userMapLock.readLock().lock();
        try {
            for (final UserInfoInternal userInfo : _userInfoHashMap.values()) {
	            if (userInfo instanceof DataLoader userAsDataLoader) {
                    if (isChildOf(userAsDataLoader, object)) {
                        affectedUserIds.add(userInfo.getUserId());
                    }
                }
            }
        } finally {
            _userMapLock.readLock().unlock();
        }

        // Im Falle das _userRightsChangeHandler der ConnectionsManager ist, synchronisiert dieser auf sich selber.
        // Daher darf der folgende Code nicht im _userMapLock stehen, sonst wäre das als verschachteltes Locking sehr
        // DeadLock-anfällig.

        // Der Fall dass zwischenzeitlich die aktuellen Benutzer geändert worden sind, ist irrelevant
        // da der Parameterdatenempfang asynchron stattfindet und daher sowieso keine festen Aussagen bzgl.
        // der Reihenfolge der kritischen Aufrufe von addUser()/getUser()/removeUser() etc. und objectChanged() gemacht werden können.
        // Benutzer, die während der Auführung dieser Zeilen angelegt werden besitzen bereits die neuen Parameterdaten
        // und sind daher unkritisch. Benutzer die währenddessen gelöscht werden sind sowieso unerheblich,
        // da diese sowieso gezwungen sind alle Anmeldungen zu entfernen und eine Aktualisierung wg. geänderter Rechte sinnlos wäre
        for (Long affectedUserId : affectedUserIds) {
            notifyUserRightsChangedAsync(affectedUserId);
        }

    }
}
