/*
 * Copyright 2004 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.communication.dataRepresentation.data.byteArray;

import de.bsvrz.dav.daf.communication.dataRepresentation.data.info.AttributeDefinitionInfo;
import de.bsvrz.dav.daf.communication.dataRepresentation.data.info.AttributeInfo;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.ObjectLookup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * Klasse, die noch zu dokumentieren ist.
 *
 * @author Kappich+Kniß Systemberatung Aachen (K2S)
 * @author Roland Schmitz (rs)
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class ByteArrayArrayData extends ByteArrayStructuredData
    implements Data.Array, Data.TextArray, Data.TimeArray, Data.ReferenceArray, Data.NumberArray {
    private final int _length;

    public ByteArrayArrayData(final byte[] bytes, final int offset, final AttributeInfo attributeInfo) {
        super(bytes, offset, attributeInfo);
        _length = attributeInfo.getElementCount(bytes, offset);
    }

    public String toParamString() {
        return super.toParamString() + ", Anzahl Elemente: " + _length;
    }

    public Data getItem(String itemName) {
        return getItem(Integer.parseInt(itemName));
    }

    @Override
    public Iterator<Data> iterator() {
        return new ArrayDataIterator();
    }

    public Data getItem(int itemIndex) {
        int itemOffset = _info.getAbsoluteElementOffset(_bytes, _offset, itemIndex);
        return ByteArrayData.create(_bytes, itemOffset, _info.getElementInfo(), itemIndex);
    }

    @Override
    public TextValue getTextValue(int itemIndex) {
        return getItem(itemIndex).asTextValue();
    }

    @Override
    public TextValue[] getTextValues() {
        TextValue[] result = new TextValue[_length];
	    Iterator<Data> iterator = iterator();
        for (int i = 0; i < _length; ++i) {
            Data data = (Data) iterator.next();
            result[i] = data.asTextValue();
        }
        return result;
    }

    @Override
    public TimeValue getTimeValue(int itemIndex) {
        return getItem(itemIndex).asTimeValue();
    }

    @Override
    public TimeValue[] getTimeValues() {
        TimeValue[] result = new TimeValue[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getTimeValue(i);
        }
        return result;
    }

    @Override
    public NumberValue getScaledValue(int itemIndex) {
        return getItem(itemIndex).asScaledValue();
    }

    @Override
    public NumberValue[] getScaledValues() {
        NumberValue[] result = new NumberValue[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getScaledValue(i);
        }
        return result;
    }

    @Override
    public NumberValue getUnscaledValue(int itemIndex) {
        return getItem(itemIndex).asUnscaledValue();
    }

    @Override
    public NumberValue[] getUnscaledValues() {
        NumberValue[] result = new NumberValue[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getUnscaledValue(i);
        }
        return result;
    }

    @Override
    public ReferenceValue getReferenceValue(int itemIndex) {
        return getItem(itemIndex).asReferenceValue();
    }

    @Override
    public ReferenceValue[] getReferenceValues() {
        ReferenceValue[] result = new ReferenceValue[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getReferenceValue(i);
        }
        return result;
    }

    public int getLength() {
        return _length;
    }

    public void setLength(int newLength) {
        throw new UnsupportedOperationException("setLength(" + newLength + "): Das Data-Objekt darf nicht verändert werden");
    }

    public void set(byte[] bytes) {
        throw new UnsupportedOperationException("set(byte[]): Das Data-Objekt darf nicht verändert werden");
    }

    public void set(short[] shorts) {
        throw new UnsupportedOperationException("set(short[]): Das Data-Objekt darf nicht verändert werden");
    }

    public void set(int[] ints) {
        throw new UnsupportedOperationException("set(int[]): Das Data-Objekt darf nicht verändert werden");
    }

    public void set(long[] longs) {
        throw new UnsupportedOperationException("set(long[]): Das Data-Objekt darf nicht verändert werden");
    }

    public void set(float[] floats) {
        throw new UnsupportedOperationException("set(float[]): Das Data-Objekt darf nicht verändert werden");
    }

    public void set(double[] doubles) {
        throw new UnsupportedOperationException("set(double[]): Das Data-Objekt darf nicht verändert werden");
    }

    public void set(final SystemObject[] systemObjects) {
        throw new UnsupportedOperationException("set(SystemObject[]): Das Data-Objekt darf nicht verändert werden");
    }

    public void set(final String[] strings) {
        throw new UnsupportedOperationException("set(String[]): Das Data-Objekt darf nicht verändert werden");
    }

    public void set(final ObjectLookup dataModel, final String... systemObjectPids) {
        throw new UnsupportedOperationException("set(ObjectLookup, String[]): Das Data-Objekt darf nicht verändert werden");
    }

    @Override
    public void setMillis(final long[] millis) {
        throw new UnsupportedOperationException("setMillis(long[]): Das Data-Objekt darf nicht verändert werden");
    }

    @Override
    public void setSeconds(final long[] seconds) {
        throw new UnsupportedOperationException("setSeconds(long[]): Das Data-Objekt darf nicht verändert werden");
    }

    @Override
    public int getMaxCount() {
        return _info.getMaxCount();
    }

    @Override
    public boolean isCountLimited() {
        return _info.isCountLimited();
    }

    @Override
    public boolean isCountVariable() {
        return _info.isCountVariable();
    }

    @Override
    public boolean isList() {
        return false;
    }

    @Override
    public boolean isArray() {
        return true;
    }

    @Override
    public boolean isPlain() {
        return false;
    }

    @Override
    public Data.Array asArray() {
        return this;
    }

    @Override
    public Data.TimeArray asTimeArray() {
        if (_info.getDefinitionInfo().isTimeAttribute()) {
            return this;
        }
        throw new UnsupportedOperationException("Das Attribut " + getName() + " enthält keine Zeitangaben");
    }

    @Override
    public long getSeconds(int itemIndex) {
        return getTimeValue(itemIndex).getSeconds();
    }

    @Override
    public long getMillis(int itemIndex) {
        return getTimeValue(itemIndex).getMillis();
    }

    @Override
    public long[] getSecondsArray() {
        long[] result = new long[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getTimeValue(i).getSeconds();
        }
        return result;
    }

    @Override
    public long[] getMillisArray() {
        long[] result = new long[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getTimeValue(i).getMillis();
        }
        return result;
    }

    @Override
    public Data.TextArray asTextArray() {
        return this;
    }

    @Override
    public String getText(int itemIndex) {
        return getTextValue(itemIndex).getText();
    }

    @Override
    public String[] getTextArray() {
        String[] result = new String[_length];
	    Iterator<Data> iterator = iterator();
        for (int i = 0; i < _length; ++i) {
            Data data = (Data) iterator.next();
            result[i] = data.asTextValue().getText();
        }
        return result;
    }

    @Override
    public Data.ReferenceArray asReferenceArray() {
        if (_info.getDefinitionInfo().isReferenceAttribute()) {
            return this;
        }
        throw new UnsupportedOperationException("Das Attribut " + getName() + " enthält keine Referenzen");
    }

    @Override
    public SystemObject getSystemObject(int itemIndex) {
        return getReferenceValue(itemIndex).getSystemObject();
    }

    @Override
    public SystemObject[] getSystemObjectArray() {
        SystemObject[] result = new SystemObject[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getReferenceValue(i).getSystemObject();
        }
        return result;
    }

    @Override
    public Data.NumberArray asScaledArray() {
        if (_info.getDefinitionInfo().isNumberAttribute()) {
            return this;
        }
        throw new UnsupportedOperationException("Das Attribut " + getName() + " enthält keine Zahlen");
    }

    public NumberValue getValue(int itemIndex) {
        return getScaledValue(itemIndex);
    }

    @Override
    public NumberValue[] getValues() {
        return getScaledValues();
    }

    @Override
    public byte byteValue(int itemIndex) {
        return getScaledValue(itemIndex).byteValue();
    }

    @Override
    public short shortValue(int itemIndex) {
        return getScaledValue(itemIndex).shortValue();
    }

    @Override
    public int intValue(int itemIndex) {
        return getScaledValue(itemIndex).intValue();
    }

    @Override
    public long longValue(int itemIndex) {
        return getScaledValue(itemIndex).longValue();
    }

    @Override
    public float floatValue(int itemIndex) {
        return getScaledValue(itemIndex).floatValue();
    }

    @Override
    public double doubleValue(int itemIndex) {
        return getScaledValue(itemIndex).doubleValue();
    }

    @Override
    public byte[] getByteArray() {
        byte[] result = new byte[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getScaledValue(i).byteValue();
        }
        return result;
    }

    @Override
    public short[] getShortArray() {
        short[] result = new short[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getScaledValue(i).shortValue();
        }
        return result;
    }

    @Override
    public int[] getIntArray() {
        int[] result = new int[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getScaledValue(i).intValue();
        }
        return result;
    }

    @Override
    public long[] getLongArray() {
        long[] result = new long[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getScaledValue(i).longValue();
        }
        return result;
    }

    @Override
    public float[] getFloatArray() {
        float[] result = new float[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getScaledValue(i).floatValue();
        }
        return result;
    }

    @Override
    public double[] getDoubleArray() {
        double[] result = new double[_length];
        for (int i = 0; i < _length; ++i) {
            result[i] = getScaledValue(i).doubleValue();
        }
        return result;
    }

    @Override
    public Data.NumberArray asUnscaledArray() {
        final AttributeDefinitionInfo definitionInfo = _info.getDefinitionInfo();
        if (definitionInfo.isNumberAttribute()) {
            if (definitionInfo.isScalableNumberAttribute()) {
                return new UnscaledNumberArrayView();
            } else {
                // Bei DoubleAttributeType und FloatAttributeType:
                return this;
            }
        }
        throw new UnsupportedOperationException("Das Attribut " + getName() + " enthält keine Zahlen");
    }

    private class ArrayDataIterator implements Iterator<Data> {
        private int _nextElementIndex;
        private int _nextElementOffset = _offset + 4;

        public boolean hasNext() {
            if (_nextElementIndex < _length) {
                return true;
            }
            return false;
        }

        public Data next() {
            if (_nextElementIndex >= _length) {
                throw new NoSuchElementException();
            }
            final AttributeInfo elementInfo = _info.getElementInfo();
            final ByteArrayData data = ByteArrayData.create(_bytes, _nextElementOffset, elementInfo, _nextElementIndex);
            _nextElementOffset += elementInfo.getSize(_bytes, _nextElementOffset);
            ++_nextElementIndex;
            return data;
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    private class UnscaledNumberArrayView implements Data.NumberArray {

        public int getLength() {
            return _length;
        }

        public void setLength(int newLength) {
            throw new UnsupportedOperationException("setLength(" + newLength + "): Das Data-Objekt darf nicht verändert werden");
        }

        public void set(byte[] bytes) {
            throw new UnsupportedOperationException("set(byte[]): Das Data-Objekt darf nicht verändert werden");
        }

        public void set(short[] shorts) {
            throw new UnsupportedOperationException("set(short[]): Das Data-Objekt darf nicht verändert werden");
        }

        public void set(int[] ints) {
            throw new UnsupportedOperationException("set(int[]): Das Data-Objekt darf nicht verändert werden");
        }

        public void set(long[] longs) {
            throw new UnsupportedOperationException("set(long[]): Das Data-Objekt darf nicht verändert werden");
        }

        public void set(float[] floats) {
            throw new UnsupportedOperationException("set(float[]): Das Data-Objekt darf nicht verändert werden");
        }

        public void set(double[] doubles) {
            throw new UnsupportedOperationException("set(double[]): Das Data-Objekt darf nicht verändert werden");
        }

        public NumberValue getValue(int itemIndex) {
            return getUnscaledValue(itemIndex);
        }

        @Override
        public NumberValue[] getValues() {
            return getUnscaledValues();
        }

        @Override
        public byte byteValue(int itemIndex) {
            return getUnscaledValue(itemIndex).byteValue();
        }

        @Override
        public short shortValue(int itemIndex) {
            return getUnscaledValue(itemIndex).shortValue();
        }

        @Override
        public int intValue(int itemIndex) {
            return getUnscaledValue(itemIndex).intValue();
        }

        @Override
        public long longValue(int itemIndex) {
            return getUnscaledValue(itemIndex).longValue();
        }

        @Override
        public float floatValue(int itemIndex) {
            return getUnscaledValue(itemIndex).floatValue();
        }

        @Override
        public double doubleValue(int itemIndex) {
            return getUnscaledValue(itemIndex).doubleValue();
        }

        @Override
        public byte[] getByteArray() {
            byte[] result = new byte[_length];
            for (int i = 0; i < _length; ++i) {
                result[i] = getUnscaledValue(i).byteValue();
            }
            return result;
        }

        @Override
        public short[] getShortArray() {
            short[] result = new short[_length];
            for (int i = 0; i < _length; ++i) {
                result[i] = getUnscaledValue(i).shortValue();
            }
            return result;
        }

        @Override
        public int[] getIntArray() {
            int[] result = new int[_length];
            for (int i = 0; i < _length; ++i) {
                result[i] = getUnscaledValue(i).intValue();
            }
            return result;
        }

        @Override
        public long[] getLongArray() {
            long[] result = new long[_length];
            for (int i = 0; i < _length; ++i) {
                result[i] = getUnscaledValue(i).longValue();
            }
            return result;
        }

        @Override
        public float[] getFloatArray() {
            float[] result = new float[_length];
            for (int i = 0; i < _length; ++i) {
                result[i] = getUnscaledValue(i).floatValue();
            }
            return result;
        }

        @Override
        public double[] getDoubleArray() {
            double[] result = new double[_length];
            for (int i = 0; i < _length; ++i) {
                result[i] = getUnscaledValue(i).doubleValue();
            }
            return result;
        }

    }
}
