/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.communication.dataRepresentation.datavalue;

import de.bsvrz.dav.daf.communication.lowLevel.telegrams.BaseSubscriptionInfo;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Diese Klasse stellt die Attribute und Funktionalitäten des Datentyps SendDataObject zur Verfügung.
 *
 * @author Kappich Systemberatung
 */
public class SendDataObject {

    /** Die Basisanmeldeinformationen */
    private BaseSubscriptionInfo _baseSubscriptionInfo;

    /** Besagt das die zu sendenden Daten nachgelieferte Daten sind. */
    private boolean _delayedDataFlag;

    /** Laufende Nummer des Datensatzes */
    private long _dataNumber;

    /** Die Zeit der Datensätze */
    private long _dataTime;

    /**
     * Fehlerkennung der Anwendungsdaten. 0: Daten vorhanden (kein fehler). 1: Quelle vorhanden aber Daten noch nicht lieferbar. 2: Quelle nicht
     * vorhanden. 3: Keine Rechte 8: Mehrere Quelle-Senke-Applikationen
     */
    private byte _errorFlag;

    /** Der Indikator zu Änderungen der einzelnen Attribute der Attributgruppe */
    private byte[] _attributesIndicator;

    /** Der zu sendende Bytestrom */
    private byte[] _data;

    /** Erzeugt ein neues Objekt ohne Parameter. Die Parameter werden zu einem Späteren Zeitpunkt über die read-Methode eingelesen. */
    public SendDataObject() {
    }

    /**
     * Erzeugt ein neues Objekt mit den gegebenen Parametern.
     *
     * @param baseSubscriptionInfo Anmeldeinformationen
     * @param delayedDataFlag      Besagt das die zu sendenden Daten nachgelieferte Daten sind.
     * @param dataNumber           Laufende Nummer des Datensatzes
     * @param time                 Die Zeit der Datensätze
     * @param errorFlag            Fehlerkennung
     * @param attributesIndicator  Indikator zu der Änderungen der einzelnen Attributen
     * @param data                 Der zu sendende Bytestrom
     */
    public SendDataObject(BaseSubscriptionInfo baseSubscriptionInfo, boolean delayedDataFlag, long dataNumber, long time, byte errorFlag,
                          byte[] attributesIndicator, byte[] data) {
        _baseSubscriptionInfo = baseSubscriptionInfo;
        _delayedDataFlag = delayedDataFlag;
        _dataNumber = dataNumber;
        _dataTime = time;
        _errorFlag = errorFlag;
        _attributesIndicator = attributesIndicator;
        _data = data;
    }

    /**
     * Gibt die Basisanmeldeinformationen zurück
     *
     * @return Basisanmeldeinformationen
     */
    public final BaseSubscriptionInfo getBaseSubscriptionInfo() {
        return _baseSubscriptionInfo;
    }

    /**
     * Gibt an, ob Interesse an nachgelieferte oder an aktuelle Datensätze besteht.
     *
     * @return true: verspätet, false: pünktlich
     */
    public final boolean getDalayedDataFlag() {
        return _delayedDataFlag;
    }

    /**
     * Gibt die Laufende Nummer des Datensatzes zurück.
     *
     * @return laufende Nummer des Datensatzes
     */
    public final long getDataNumber() {
        return _dataNumber;
    }

    /**
     * Die Zeit der Datensätze.
     *
     * @return Zeitstempel
     */
    public final long getDataTime() {
        return _dataTime;
    }

    /**
     * Gibt der Fehlercode wenn vorhanden zurück.
     *
     * @return Fehlercode
     */
    public final byte getErrorFlag() {
        return _errorFlag;
    }

    /**
     * Gibt den Indikator zu den einzelnen Attributen der Attributgruppe zurück.
     *
     * @return Indikator der Attributgruppe
     */
    public final byte[] getAttributesIndicator() {
        return _attributesIndicator;
    }

    /**
     * Gibt den Datensatz zurrück.
     *
     * @return Datensatz
     */
    public final byte[] getData() {
        return _data;
    }

    /**
     * Gibt ein String zurrück, der diesen Datensatz beschreibt.
     *
     * @return Der String, der diesen Datensatz beschreibt
     */
    public String parseToString() {
        StringBuilder str = new StringBuilder("Sende Objekt: \n");
        str.append(_baseSubscriptionInfo.toString()).append("\n");
        str.append("Nachgelieferte Daten : ").append(_delayedDataFlag).append("\n");
        str.append("Sendedaten Nummer    : ").append(_dataNumber).append("\n");
        str.append("Sendedaten Zeit      : ").append(_dataTime).append("\n");
        str.append("Fehler Information   : ").append(_errorFlag).append("\n");
        if (_attributesIndicator != null) {
            str.append("----------Attributsänderungsindikator----------\n");
            for (int i = 0; i < _attributesIndicator.length; ++i) {
                byte b = _attributesIndicator[i];
                int j = i * 8;
                if ((b & 0x01) == 0x01) {
                    str.append("Attribut an der Position ").append(j).append(" hat sich geändert.");
                }
                if ((b & 0x02) == 0x02) {
                    str.append("Attribut an der Position ").append(j + 1).append(" hat sich geändert.");
                }
                if ((b & 0x04) == 0x04) {
                    str.append("Attribut an der Position ").append(j + 2).append(" hat sich geändert.");
                }
                if ((b & 0x08) == 0x08) {
                    str.append("Attribut an der Position ").append(j + 3).append(" hat sich geändert.");
                }
                if ((b & 0x10) == 0x10) {
                    str.append("Attribut an der Position ").append(j + 4).append(" hat sich geändert.");
                }
                if ((b & 0x20) == 0x20) {
                    str.append("Attribut an der Position ").append(j + 5).append(" hat sich geändert.");
                }
                if ((b & 0x40) == 0x40) {
                    str.append("Attribut an der Position ").append(j + 6).append(" hat sich geändert.");
                }
                if ((b & 0x80) == 0x80) {
                    str.append("Attribut an der Position ").append(j + 7).append(" hat sich geändert.");
                }
            }
        }
        if (_data != null) {
            str.append(new String(_data));
        }
        return str.toString();
    }

    /**
     * Schreiben eines Datensatzes in den gegebenen DataOutputStream
     *
     * @param out Ausgabe-Stream
     *
     * @throws IOException wenn beim Schreiben vom Ausgabe-Stream Fehler aufgetreten sind.
     */
    public final void write(DataOutputStream out) throws IOException {
        _baseSubscriptionInfo.write(out);
        out.writeBoolean(_delayedDataFlag);
        out.writeLong(_dataNumber);
        out.writeLong(_dataTime);
        out.writeByte(_errorFlag);
        if (_attributesIndicator == null) {
            out.writeInt(0);
        } else {
            out.writeByte(_attributesIndicator.length);
            if (_attributesIndicator.length > 0) {
                for (final byte a_attributesIndicator : _attributesIndicator) {
                    out.writeByte(a_attributesIndicator);
                }
            }
        }
        if (_data == null) {
            out.writeInt(0);
        } else {
            out.writeInt(_data.length);
            if (_data.length > 0) {
                for (final byte a_data : _data) {
                    out.writeByte(a_data);
                }
            }
        }
    }

    /**
     * Lesen eines Datensatzes vom gegebenen DataInputStream
     *
     * @param in Eingabe-Stream
     *
     * @throws IOException wenn beim Lesen vom Eingabe-Stream Fehler aufgetreten sind.
     */
    public final void read(DataInputStream in) throws IOException {
        _baseSubscriptionInfo = new BaseSubscriptionInfo();
        _baseSubscriptionInfo.read(in);
        _delayedDataFlag = in.readBoolean();
        _dataNumber = in.readLong();
        _dataTime = in.readLong();
        _errorFlag = in.readByte();
        int size = in.readByte();
        if (size > 0) {
            _attributesIndicator = new byte[size];
            for (int i = 0; i < size; ++i) {
                _attributesIndicator[i] = in.readByte();
            }
        }
        size = in.readInt();
        if (size > 0) {
            _data = new byte[size];
            for (int i = 0; i < size; ++i) {
                _data[i] = in.readByte();
            }
        }
    }

    /**
     * Gibt die Länge dieses Telegrams zurück
     *
     * @return die Länge dieses Telegrams
     */
    public int getLength() {
        return 37 + (_attributesIndicator == null ? 0 : _attributesIndicator.length) + (_data == null ? 0 : _data.length);
    }
}
