/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.communication.lowLevel.telegrams;

import de.bsvrz.dav.daf.communication.lowLevel.QueueableTelegram;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * In dieser Klasse wird der Typ eines Telegramms definiert und Methoden zum Lesen und Schreiben deklariert. Es wird von zwei Verbindungsarten
 * ausgegangen: von DAV zu DAV und von DAF zu DAV.
 *
 * @author Kappich Systemberatung
 */

public abstract class DataTelegram implements QueueableTelegram {

    /* Die DAF-DAV Typen */

    /** Telegrammtyp: Protokollversionsverhandlung */
    public static final byte PROTOCOL_VERSION_REQUEST_TYPE = 1;

    /** Telegrammtyp: Antwort auf die Protokollversionsverhandlung */
    public static final byte PROTOCOL_VERSION_ANSWER_TYPE = 2;

    /** Telegrammtyp: Authentifizierungsanfrage */
    public static final byte AUTHENTIFICATION_TEXT_REQUEST_TYPE = 3;

    /** Telegrammtyp: Antwort auf eine Anfrage eines Authentifizierungsschluessels */
    public static final byte AUTHENTIFICATION_TEXT_ANSWER_TYPE = 4;

    /** Telegrammtyp: Authentifizierungsanfrage */
    public static final byte AUTHENTIFICATION_REQUEST_TYPE = 5;

    /** Telegrammtyp: Antwort auf Authentifizierungsanfrage */
    public static final byte AUTHENTIFICATION_ANSWER_TYPE = 6;

    /** Telegrammtyp: Kommunikationsparameterverhandlung */
    public static final byte COM_PARAMETER_REQUEST_TYPE = 7;

    /** Telegrammtyp: Antwort auf Kommunikationsparameterverhandlung */
    public static final byte COM_PARAMETER_ANSWER_TYPE = 8;

    /** Telegrammtyp: Terminierungsbefehl */
    public static final byte TERMINATE_ORDER_TYPE = 9;

    /** Telegrammtyp: Schließungsnachricht */
    public static final byte CLOSING_TYPE = 10;

    /** Telegrammtyp: Sendedatenaufforderung */
    public static final byte REQUEST_SENDER_DATA_TYPE = 11;

    /** Telegrammtyp: Telegrammlaufzeitermittlungsanfrage */
    public static final byte TELEGRAM_TIME_REQUEST_TYPE = 12;

    /** Telegrammtyp: Telegrammlaufzeitermittlungsantwort */
    public static final byte TELEGRAM_TIME_ANSWER_TYPE = 13;

    /** Telegrammtyp: KeepAlive Telegramm */
    public static final byte KEEP_ALIVE_TYPE = 14;

    /** Telegrammtyp: Sendeanmeldung */
    public static final byte SEND_SUBSCRIPTION_TYPE = 15;

    /** Telegrammtyp: ReceiveSubscriptionTelegram: Empfangsanmeldung */
    public static final byte RECEIVE_SUBSCRIPTION_TYPE = 16;

    /** Telegrammtyp: Sendeabmeldung */
    public static final byte SEND_UNSUBSCRIPTION_TYPE = 17;

    /** Telegrammtyp: Empfangsabmeldung */
    public static final byte RECEIVE_UNSUBSCRIPTION_TYPE = 18;

    /** Telegrammtyp: Austausch von Datensätzen */
    public static final byte APPLICATION_DATA_TELEGRAM_TYPE = 19;

    /* Die DAV-DAV Typen */
    /** Telegrammtyp: Verhandlung der Protokollversion (Client) */
    public static final byte TRANSMITTER_PROTOCOL_VERSION_REQUEST_TYPE = 65;

    /** Telegrammtyp: Verhandlung der Protokollversion (Server). */
    public static final byte TRANSMITTER_PROTOCOL_VERSION_ANSWER_TYPE = 66;

    /** Telegrammtyp:  Authentifizierungsbereitschaft */
    public static final byte TRANSMITTER_AUTHENTIFICATION_TEXT_REQUEST_TYPE = 67;

    /** Telegrammtyp: Aufforderung zur Authentifizierung */
    public static final byte TRANSMITTER_AUTHENTIFICATION_TEXT_ANSWER_TYPE = 68;

    /** Telegrammtyp: Übermittelung der Authentifizierungsdaten */
    public static final byte TRANSMITTER_AUTHENTIFICATION_REQUEST_TYPE = 69;

    /** Telegrammtyp: Informationen, die nach der erfolgreichen Authentifizierung übergeben werden */
    public static final byte TRANSMITTER_AUTHENTIFICATION_ANSWER_TYPE = 70;

    /** Telegrammtyp: Verhandlung der Verbindungsparameter (Client). */
    public static final byte TRANSMITTER_COM_PARAMETER_REQUEST_TYPE = 71;

    /** Telegrammtyp: Verhandlung der Verbindungsparameter (Server). */
    public static final byte TRANSMITTER_COM_PARAMETER_ANSWER_TYPE = 72;

    /** Telegrammtyp: Terminierung der Verbindung */
    public static final byte TRANSMITTER_TERMINATE_ORDER_TYPE = 73;

    /** Telegrammtyp: Abmeldung einer Datenverteiler-Datenverteiler-Verbindung */
    public static final byte TRANSMITTER_CLOSING_TYPE = 74;

    /** Telegrammtyp: Ermittlung der Telegrammlaufzeit */
    public static final byte TRANSMITTER_TELEGRAM_TIME_REQUEST_TYPE = 75;

    /** Telegrammtyp: Ermittlung der Telegrammlaufzeit */
    public static final byte TRANSMITTER_TELEGRAM_TIME_ANSWER_TYPE = 76;

    /** Telegrammtyp: Keep-Alive Telegramm */
    public static final byte TRANSMITTER_KEEP_ALIVE_TYPE = 77;

    /** Der Datenverteiler meldet Daten für Empfänger oder Sender an */
    public static final byte TRANSMITTER_DATA_SUBSCRIPTION_TYPE = 78;

    /** Der Datenverteiler quittiert eine an ihn gerichtete Datenanmeldung */
    public static final byte TRANSMITTER_DATA_SUBSCRIPTION_RECEIPT_TYPE = 79;

    /** Der Datenverteiler meldet Daten für Empfänger oder Sender bei einem anderen Datenverteiler ab */
    public static final byte TRANSMITTER_DATA_UNSUBSCRIPTION_TYPE = 80;

    /** Telegramm zur Übertragung der Anwendungsdaten */
    public static final byte TRANSMITTER_DATA_TELEGRAM_TYPE = 81;

    /** Telegramm zur Aktualisierung der Matrix der günstigsten Wege */
    public static final byte TRANSMITTER_BEST_WAY_UPDATE_TYPE = 82;

    /** Telegramm zur Anmeldung von Anmeldungslisten Abonnements */
    public static final byte TRANSMITTER_LISTS_SUBSCRIPTION_TYPE = 83;

    /** Telegrammtyp: Abmeldung von Anmeldungslisten Abonnements */
    public static final byte TRANSMITTER_LISTS_UNSUBSCRIPTION_TYPE = 84;

    /** Telegrammtyp: Kündigung von Anmeldungslisten Abonnements. */
    public static final byte TRANSMITTER_LISTS_DELIVERY_UNSUBSCRIPTION_TYPE = 85;

    /** Telegrammtyp: Änderungsmitteilung zu Anmeldungslisten. */
    public static final byte TRANSMITTER_LISTS_UPDATE_TYPE = 86;

    /** Telegrammtyp: Änderungsmitteilung zu Anmeldungslisten. */
    public static final byte TRANSMITTER_LISTS_UPDATE_2_TYPE = 87;

    /** Telegrammtyp: SRP-Telegramm 1 Client → Server */
    public static final byte SRP_REQUEST_TYPE = 88;

    /** Telegrammtyp: SRP-Telegramm 2 Server → Client */
    public static final byte SRP_ANSWER_TYPE = 89;

    /** Telegrammtyp: SRP-Telegramm 3 Client → Server */
    public static final byte SRP_VALDIATE_REQUEST_TYPE = 90;

    /** Telegrammtyp: SRP-Telegramm 4 Server → Client */
    public static final byte SRP_VALDIATE_ANSWER_TYPE = 91;

    /** Telegramm zum Anfragen einer Applikation nach der SRP-Authentifizierung */
    public static final byte APPLICATION_REQUEST_TYPE = 92;

    /** Telegramm zum Aufbauen einer Dav-Dav-Verbindung nach der SRP-Authentifizierung */
    public static final byte TRANSMITTER_REQUEST_TYPE = 93;

    /** Telegramm mit verschlüsseltem Inhalt */
    public static final byte ENCRYPTED_TYPE = 94;

    /** Anfrage Verschlüsselung abschalten */
    public static final byte DISABLE_ENCRYPTION_REQUEST_TYPE = 95;

    /** Antwort Verschlüsselung abschalten */
    public static final byte DISABLE_ENCRYPTION_ANSWER_TYPE = 96;

    /** Telegramm zur Übertragung der Anwendungsdaten Version 2 */
    public static final byte TRANSMITTER_DATA_TELEGRAM_V2_TYPE = 97;

    /** Der Telegrammtyp. */
    protected byte type;

    /** Die Telegrammpriorität. */
    protected byte priority;

    /** Die Länge des Telegrams. */
    protected int length;

    /**
     * Erstellt ein leeres Telegramm vom Typ des Übergabeparameters.
     *
     * @param _type Typ eines Telegramms.
     *
     * @return gibt eine Instanz des von {@code DataTelegram} abgeleiteten Typs zurück.
     */
    public static DataTelegram getTelegram(byte _type) {
	    return switch (_type) {
		    case PROTOCOL_VERSION_REQUEST_TYPE -> new ProtocolVersionRequest();
		    case PROTOCOL_VERSION_ANSWER_TYPE -> new ProtocolVersionAnswer();
		    case AUTHENTIFICATION_TEXT_REQUEST_TYPE -> new AuthentificationTextRequest();
		    case AUTHENTIFICATION_TEXT_ANSWER_TYPE -> new AuthentificationTextAnswer();
		    case AUTHENTIFICATION_REQUEST_TYPE -> new AuthentificationRequest();
		    case AUTHENTIFICATION_ANSWER_TYPE -> new AuthentificationAnswer();
		    case COM_PARAMETER_REQUEST_TYPE -> new ComParametersRequest();
		    case COM_PARAMETER_ANSWER_TYPE -> new ComParametersAnswer();
		    case REQUEST_SENDER_DATA_TYPE -> new RequestSenderDataTelegram();
		    case TELEGRAM_TIME_REQUEST_TYPE -> new TelegramTimeRequest();
		    case TELEGRAM_TIME_ANSWER_TYPE -> new TelegramTimeAnswer();
		    case APPLICATION_DATA_TELEGRAM_TYPE -> new ApplicationDataTelegram();
		    case RECEIVE_SUBSCRIPTION_TYPE -> new ReceiveSubscriptionTelegram();
		    case RECEIVE_UNSUBSCRIPTION_TYPE -> new ReceiveUnsubscriptionTelegram();
		    case SEND_SUBSCRIPTION_TYPE -> new SendSubscriptionTelegram();
		    case SEND_UNSUBSCRIPTION_TYPE -> new SendUnsubscriptionTelegram();
		    case TERMINATE_ORDER_TYPE -> new TerminateOrderTelegram();
		    case KEEP_ALIVE_TYPE -> new KeepAliveTelegram();
		    case CLOSING_TYPE -> new ClosingTelegram();
		    case TRANSMITTER_PROTOCOL_VERSION_REQUEST_TYPE -> new TransmitterProtocolVersionRequest();
		    case TRANSMITTER_PROTOCOL_VERSION_ANSWER_TYPE -> new TransmitterProtocolVersionAnswer();
		    case TRANSMITTER_AUTHENTIFICATION_TEXT_REQUEST_TYPE -> new TransmitterAuthentificationTextRequest();
		    case TRANSMITTER_AUTHENTIFICATION_TEXT_ANSWER_TYPE -> new TransmitterAuthentificationTextAnswer();
		    case TRANSMITTER_AUTHENTIFICATION_REQUEST_TYPE -> new TransmitterAuthentificationRequest();
		    case TRANSMITTER_AUTHENTIFICATION_ANSWER_TYPE -> new TransmitterAuthentificationAnswer();
		    case TRANSMITTER_COM_PARAMETER_REQUEST_TYPE -> new TransmitterComParametersRequest();
		    case TRANSMITTER_COM_PARAMETER_ANSWER_TYPE -> new TransmitterComParametersAnswer();
		    case TRANSMITTER_TERMINATE_ORDER_TYPE -> new TransmitterTerminateOrderTelegram();
		    case TRANSMITTER_CLOSING_TYPE -> new TransmitterClosingTelegram();
		    case TRANSMITTER_TELEGRAM_TIME_REQUEST_TYPE -> new TransmitterTelegramTimeRequest();
		    case TRANSMITTER_TELEGRAM_TIME_ANSWER_TYPE -> new TransmitterTelegramTimeAnswer();
		    case TRANSMITTER_KEEP_ALIVE_TYPE -> new TransmitterKeepAliveTelegram();
		    case TRANSMITTER_DATA_SUBSCRIPTION_TYPE -> new TransmitterDataSubscription();
		    case TRANSMITTER_DATA_SUBSCRIPTION_RECEIPT_TYPE -> new TransmitterDataSubscriptionReceipt();
		    case TRANSMITTER_DATA_UNSUBSCRIPTION_TYPE -> new TransmitterDataUnsubscription();
		    case TRANSMITTER_DATA_TELEGRAM_TYPE -> new TransmitterDataTelegram();
		    case TRANSMITTER_DATA_TELEGRAM_V2_TYPE -> new TransmitterDataTelegramV2();
		    case TRANSMITTER_BEST_WAY_UPDATE_TYPE -> new TransmitterBestWayUpdate();
		    case TRANSMITTER_LISTS_SUBSCRIPTION_TYPE -> new TransmitterListsSubscription();
		    case TRANSMITTER_LISTS_UNSUBSCRIPTION_TYPE -> new TransmitterListsUnsubscription();
		    case TRANSMITTER_LISTS_DELIVERY_UNSUBSCRIPTION_TYPE -> new TransmitterListsDeliveryUnsubscription();
		    case TRANSMITTER_LISTS_UPDATE_TYPE -> new TransmitterListsUpdate(TRANSMITTER_LISTS_UPDATE_TYPE);
		    case TRANSMITTER_LISTS_UPDATE_2_TYPE -> new TransmitterListsUpdate(TRANSMITTER_LISTS_UPDATE_2_TYPE);
		    case SRP_REQUEST_TYPE -> new SrpRequest();
		    case SRP_ANSWER_TYPE -> new SrpAnswer();
		    case SRP_VALDIATE_REQUEST_TYPE -> new SrpValidateRequest();
		    case SRP_VALDIATE_ANSWER_TYPE -> new SrpValidateAnswer();
		    case APPLICATION_REQUEST_TYPE -> new ApplicationRequest();
		    case TRANSMITTER_REQUEST_TYPE -> new TransmitterRequest();
		    case ENCRYPTED_TYPE -> new EncryptedTelegram();
		    case DISABLE_ENCRYPTION_REQUEST_TYPE -> new DisableEncryptionRequest();
		    case DISABLE_ENCRYPTION_ANSWER_TYPE -> new DisableEncryptionAnswer();
		    default -> null;
	    };
    }

    /**
     * Liest ein Telegramm vom übergegebenen DataInputStream.
     *
     * @param in Der DataInputStream.
     *
     * @throws java.io.IOException Falls der Datensatz nicht aus dem Stream gelesen werden kann.
     */
    public abstract void read(DataInputStream in) throws IOException;

    /**
     * Schreibt ein Telegramm in den übergegebenen DataOutputStream.
     *
     * @param out Der DataOutputStream.
     *
     * @throws java.io.IOException Falls der Datensatz nicht in den Stream geschrieben werden kann.
     */
    public abstract void write(DataOutputStream out) throws IOException;

    /**
     * Gibt eine String-Repräsentation dieses Datensatzes zurück.
     *
     * @return Eine String-Repräsentation dieses Datensatzes.
     */
    public abstract String parseToString();

    /**
     * Gibt die Länge des Telegramms an.
     *
     * @return Die Länge des Telegrams.
     */
    public int getLength() {
        return length;
    }

    /**
     * Liefert die Gesamtgröße des Telegramms einschließlich des führenden Typ-Bytes und der Länge des Telegramminhalts.
     *
     * @return Gesamtgröße des Telegrams in Bytes.
     */
    public int getSize() {
        return 1 + 2 + length;
    }

    /**
     * Gibt den Typ des Telegramms an.
     *
     * @return Typ des Telegramms.
     */
    public final byte getType() {
        return type;
    }

    /**
     * Gibt die Priorität des Telegramms an. Je größer der Wert, desto größer die Priorität.
     *
     * @return Priorität des Telegramms.
     */
    public final byte getPriority() {
        return priority;
    }

    /**
     * Gibt eine kurze Beschreibung des Objektes zurück.
     *
     * @return Beschreibung des Objektes
     */
    public String toString() {
        return toShortDebugString();
    }

    /**
     * Gibt eine kurze Beschreibung des Objektes zurück.
     *
     * @return Beschreibung des Objektes
     */
    public String toShortDebugString() {
        return getClass().getName() + "{" + toShortDebugParamString() + "}";
    }

    /**
     * Bestimmt eine kurze Beschreibung der Eigenschaften eines Telegramms.
     *
     * @return Beschreibung der Eigenschaften eines Telegramms
     */
    public String toShortDebugParamString() {
        return "";
    }
}
