/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.communication.lowLevel.telegrams;

import de.bsvrz.dav.daf.main.impl.CommunicationConstant;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Der Datenverteiler meldet Daten für Empfänger oder Sender bei einem anderen Datenverteiler ab.
 *
 * @author Kappich Systemberatung
 */
public class TransmitterDataUnsubscription extends DataTelegram {

    /** Die Basisinformationen eine Anmeldung */
    private BaseSubscriptionInfo baseSubscriptionInfo;

    /** Die Information ob die Anmeldung als Sender- oder Empfängeranmeldung ist 0: Senderanmeldung 1: Empfängeranmeldung */
    private byte subscriptionType;

    /** Die Liste der zu berücksichtigenden Datenverteiler */
    private long[] transmitterList;

    public TransmitterDataUnsubscription() {
        type = TRANSMITTER_DATA_UNSUBSCRIPTION_TYPE;
        priority = CommunicationConstant.SYSTEM_TELEGRAM_PRIORITY;
    }

    /**
     * @param _baseSubscriptionInfo Basisinformationen
     * @param _subscriptionState    Anmeldung als Sender oder Empfänger
     * @param _transmitterList      Liste der zu berücksichtigenden Datenverteiler
     */
    public TransmitterDataUnsubscription(BaseSubscriptionInfo _baseSubscriptionInfo, byte _subscriptionState, long[] _transmitterList) {
        type = TRANSMITTER_DATA_UNSUBSCRIPTION_TYPE;
        priority = CommunicationConstant.SYSTEM_TELEGRAM_PRIORITY;
        baseSubscriptionInfo = _baseSubscriptionInfo;
        subscriptionType = _subscriptionState;
        transmitterList = _transmitterList;
        length = 17;
        if (transmitterList != null) {
            length += transmitterList.length * 8;
        }
    }

    /**
     * Gibt die Basisanmeldeinformationen zurück.
     *
     * @return die Basisanmeldeinformationen
     */
    public final BaseSubscriptionInfo getBaseSubscriptionInfo() {
        return baseSubscriptionInfo;
    }

    /**
     * Gibt der Typ der Anmeldung zurück 0: Senderanmeldung 1: Empfängeranmeldung
     *
     * @return der Typ der Anmeldung
     */
    public final byte getSubscriptionType() {
        return subscriptionType;
    }

    /**
     * Gibt der Typ der Anmeldung zurück 0: Senderanmeldung 1: Empfängeranmeldung
     *
     * @return der Typ der Anmeldung
     */
    public final byte getSubscriptionState() {
        return subscriptionType;
    }

    /**
     * Gibt die Liste der zu berücksichtigenden Datenverteiler zurück
     *
     * @return die Liste der zu berücksichtigenden Datenverteiler
     */
    public final long[] getTransmitters() {
        return transmitterList;
    }

    public final String parseToString() {
        String str = "Datenverteiler Datenabmeldung Systemtelegramm: ";
        str += baseSubscriptionInfo.toString();
        if (subscriptionType == TransmitterSubscriptionsConstants.SENDER_SUBSCRIPTION) {
            str += ", Anmeldung als Sender";
        } else {
            str += ", Anmeldung als Empfänger";
        }
        if (transmitterList != null) {
            str += ", Potentielle Datenverteiler [ ";
	        for (long l : transmitterList) {
		        str += " " + l + " ";
            }
            str += " ]";
        }
        return str;
    }

    public final void write(DataOutputStream out) throws IOException {
        out.writeShort(length);
        baseSubscriptionInfo.write(out);
        out.writeByte(subscriptionType);
        if (transmitterList == null) {
            out.writeShort(0);
        } else {
            out.writeShort(transmitterList.length);
	        for (long l : transmitterList) {
		        out.writeLong(l);
            }
        }
    }

    public final void read(DataInputStream in) throws IOException {
        int _length = in.readShort();
        length = 17;
        if (_length < length) {
            throw new IOException("Falsche Telegrammlänge (zu kurz)");
        }
        baseSubscriptionInfo = new BaseSubscriptionInfo();
        baseSubscriptionInfo.read(in);
        subscriptionType = in.readByte();
        int size = in.readShort();
        length += size * 8;
        if (_length != length) {
            throw new IOException("Falsche Telegrammlänge (Länge " + _length + " passt nicht zu size " + size + ")");
        }
        if (size != 0) {
            transmitterList = new long[size];
            for (int i = 0; i < size; ++i) {
                transmitterList[i] = in.readLong();
            }
        }
    }
}
