/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.communication.protocol;

import de.bsvrz.dav.daf.communication.lowLevel.AuthentificationProcess;
import de.bsvrz.dav.daf.communication.lowLevel.CommunicationParameters;
import de.bsvrz.dav.daf.communication.lowLevel.ConnectionInterface;
import de.bsvrz.dav.daf.communication.lowLevel.ConnectionProperties;
import de.bsvrz.dav.daf.communication.lowLevel.LowLevelCommunication;
import de.bsvrz.dav.daf.communication.lowLevel.LowLevelCommunicationInterface;
import de.bsvrz.dav.daf.communication.lowLevel.ServerConnectionInterface;
import de.bsvrz.dav.daf.communication.lowLevel.telegrams.DataTelegram;
import de.bsvrz.dav.daf.main.ClientDavParameters;
import de.bsvrz.dav.daf.main.CommunicationError;
import de.bsvrz.dav.daf.main.ConnectionException;
import de.bsvrz.dav.daf.main.InitialisationNotCompleteException;

/**
 * Diese Klasse enthält die Eigenschaften, die benötigt werden, um eine Verbindung zum Datenverteiler aufzubauen.
 *
 * @author Kappich Systemberatung
 */
public class ClientConnectionProperties extends ConnectionProperties {

    /**
     * Enthält den via Aufrufparameter von Start/Stopp vorgegebenen Inkarnationsnamen oder {@code ""}, falls das Aufrufargument nicht angegeben
     * wurde.
     */
    private final String _incarnationName;
    /**
     * Gibt zurück, ob es sich um die eigene Verbindung des Datenverteilers handelt
     */
    private final boolean _selfClientDavConnection;
    /** Der Applikationsname */
    private String _applicationName;
    /** Die PId des Applikationstyps */
    private String _applicationTypePid;
    /** Die Pid der Konfiguration */
    private String _configurationPid;
    /** Die Kommunikationsadresse */
    private String _address;
    /** Die Kommunikationssubadresse */
    private int _subAddress;
    /** Enthält die Parameter für die Kommunikation zwischen Applikation und Datenverteiler. */
    private CommunicationParameters _communicationParameters;
    /**
     * Den Einmalpasswortindex oder -1 für kein Einmalpasswort
     */
    private int _passwordIndex;

    /** Bei passivem Verbindungsaufbau die Serververbindung */
    private ServerConnectionInterface _serverConnection;

    public ClientConnectionProperties(ClientDavParameters clientDavParameters) throws ConnectionException {
        super(null, null, clientDavParameters.getUserName(), clientDavParameters.getCommunicationParameters().getSendKeepAliveTimeout(),
              clientDavParameters.getCommunicationParameters().getReceiveKeepAliveTimeout(), clientDavParameters.getAdjustedOutputBufferSize(),
              clientDavParameters.getAdjustedInputBufferSize(), clientDavParameters.isHmacAuthenticationAllowed(),
              clientDavParameters.getEncryptionPreference());
        try {
            ConnectionInterface connection;
            if (clientDavParameters.getPassiveCommunication() != null) {
                String comProtocol = clientDavParameters.getPassiveCommunication();
                Class<?> aClass = Class.forName(comProtocol);
                if (aClass == null) {
                    throw new InitialisationNotCompleteException("Unbekannter Kommunikationsprotokollname.");
                }
                _serverConnection = (ServerConnectionInterface) aClass.newInstance();
                _serverConnection.connect(clientDavParameters.getPassivePort());
                connection = _serverConnection.accept();
            } else {
                String comProtocol = clientDavParameters.getLowLevelCommunicationName();
                if (comProtocol == null) {
                    throw new InitialisationNotCompleteException("Unbekannter Kommunikationsprotokollname.");
                }
                Class<?> aClass = Class.forName(comProtocol);
                if (aClass == null) {
                    throw new InitialisationNotCompleteException("Unbekannter Kommunikationsprotokollname.");
                }
                connection = (ConnectionInterface) aClass.newInstance();
            }
            setLowLevelCommunication(new LowLevelCommunication(connection, clientDavParameters.getAdjustedOutputBufferSize(),
                                                               clientDavParameters.getAdjustedInputBufferSize(),
                                                               clientDavParameters.getCommunicationParameters().getSendKeepAliveTimeout(),
                                                               clientDavParameters.getCommunicationParameters().getReceiveKeepAliveTimeout(),
                                                               LowLevelCommunication.HANDLE_CONFIG_RESPONCES_MODE,
                                                               clientDavParameters.getPassiveCommunication() != null));

            String authentificationName = clientDavParameters.getAuthentificationProcessName();
            if (authentificationName == null) {
                throw new InitialisationNotCompleteException("Unbekanntes Authentifikationsverfahren.");
            }
            final Class<?> aClass = Class.forName(authentificationName);
            if (aClass == null) {
                throw new InitialisationNotCompleteException("Unbekanntes Authentifikationsverfahren.");
            }
            setAuthentificationProcess((AuthentificationProcess) aClass.newInstance());

            _applicationName = clientDavParameters.getApplicationName();
            _incarnationName = clientDavParameters.getIncarnationName();
            _applicationTypePid = clientDavParameters.getApplicationTypePid();
            _configurationPid = clientDavParameters.getConfigurationPid();
            _address = clientDavParameters.getDavCommunicationAddress();
            _subAddress = clientDavParameters.getDavCommunicationSubAddress();
            _communicationParameters = clientDavParameters.getCommunicationParameters();
            _selfClientDavConnection = clientDavParameters.isSelfClientDavConnection();
            _passwordIndex = clientDavParameters.getPasswordIndex();
        } catch (ClassNotFoundException | IllegalAccessException | InstantiationException ex) {
            throw new InitialisationNotCompleteException("Fehler beim Erzeugen der logischen Verbindung zum Datenverteiler.", ex);
        } catch (CommunicationError communicationError) {
            throw new InitialisationNotCompleteException("Fehler beim Erzeugen der logischen Server-Verbindung.", communicationError);
        }
    }

    /**
     * Gibt die Parameter für die Kommunikation zwischen Applikation und Datenverteiler zurück.
     *
     * @return die Parameter für die Kommunikation zwischen Applikation und Datenverteiler
     */
    public CommunicationParameters getCommunicationParameters() {
        return _communicationParameters;
    }

    /**
     * Gibt den Namen der Applikation zurück.
     *
     * @return der Name der Applikation
     */
    public final String getApplicationName() {
        return _applicationName;
    }

    /**
     * Setzt den Namen der Applikation.
     *
     * @param applicationName Name der Applikation
     */
    public final void setApplicationName(String applicationName) {
        _applicationName = applicationName;
    }

    /**
     * Liefert den via Aufrufparameter von Start/Stopp vorgegebenen Inkarnationsnamen.
     *
     * @return Inkarnationsname oder {@code ""}, falls das entsprechende Aufrufargument nicht angegeben wurde.
     */
    public String getIncarnationName() {
        return _incarnationName;
    }

    /**
     * Gibt den Typ der Applikation zurück.
     *
     * @return Typ der Applikation
     */
    public final String getApplicationTypePid() {
        return _applicationTypePid;
    }

    /**
     * Setzt den Typ der Applikation.
     *
     * @param applicationTypePid Typ der Applikation
     */
    public final void setApplicationTypePid(String applicationTypePid) {
        _applicationTypePid = applicationTypePid;
    }

    /**
     * Gibt die Pid der Konfiguration zurück.
     *
     * @return die Pid der Konfiguration
     */
    public final String getConfigurationPid() {
        return _configurationPid;
    }

    /**
     * Setzt die Pid der Konfiguration.
     *
     * @param configurationPid Pid der Konfiguration
     */
    public final void setConfigurationPid(String configurationPid) {
        _configurationPid = configurationPid;
    }

    /**
     * Gibt die Kommunikationsadresse des Datenverteilers zurück.
     *
     * @return die Kommunikationsadresse
     */
    public final String getCommunicationAddress() {
        return _address;
    }

    /**
     * Setzt die Kommunikationsadresse des Datenverteilers.
     *
     * @param address die Kommunikationsadresse des Datenverteilers
     */
    public final void setCommunicationAddress(String address) {
        _address = address;
    }

    /**
     * Gibt die Kommunikationssubadresse des Datenverteilers zurück.
     *
     * @return die Kommunikationssubadresse
     */
    public final int getCommunicationSubAddress() {
        return _subAddress;
    }

    /**
     * Setzt die Kommunikationssubadresse des Datenverteilers.
     *
     * @param subAddress die Kommunikationssubadresse
     */
    public final void setCommunicationSubAddress(int subAddress) {
        _subAddress = subAddress;
    }

    /**
     * Gibt {@code true} zurück, wenn es sich um die lokale Verbindung des Datenverteilers handelt
     *
     * @return {@code true}, wenn es sich um die lokale Verbindung des Datenverteilers handelt, sonst {@code false}
     */
    public boolean isSelfClientDavConnection() {
        return _selfClientDavConnection;
    }

    /**
     * Gibt den Einmalpasswortindex oder -1 für kein Einmalpasswort zurück
     *
     * @return den Einmalpasswortindex oder -1 für kein Einmalpasswort
     */
    public int getPasswordIndex() {
        return _passwordIndex;
    }

    /**
     * Setzt den passwortindex
     *
     * @param passwordIndex Einmalpasswortindex oder -1 für kein Einmalpasswort
     */
    public void setPasswordIndex(final int passwordIndex) {
        _passwordIndex = passwordIndex;
    }

    /**
     * Diese Methode wird von der Protokollschicht DaV-DAF aufgerufen, wenn die Kommunikationskanäle geschlossen werden sollen.
     *
     * @param error               Besagt, ob es sich um eine Terminierung mit Fehler handelt.
     * @param message             der Fehlertext
     * @param terminationTelegram Terminierungstelegramm
     *
     * @since 3.13
     */
    public void disconnect(final boolean error, final String message, final DataTelegram terminationTelegram) {
        LowLevelCommunicationInterface lowLevelCommunication = getLowLevelCommunication();
        if (lowLevelCommunication != null) {
            lowLevelCommunication.disconnect(error, message, terminationTelegram);
        }
        ServerConnectionInterface serverConnection = _serverConnection;
        if (serverConnection != null) {
            serverConnection.disconnect();
        }
    }
}
