/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.main;

import de.bsvrz.dav.daf.accessControl.AccessControlManager;
import de.bsvrz.dav.daf.accessControl.AccessControlMode;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestManager;
import de.bsvrz.dav.daf.main.authentication.ClientCredentials;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.ClientApplication;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.ConfigurationAuthority;
import de.bsvrz.dav.daf.main.config.ConfigurationTaskException;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.DavApplication;
import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.DynamicObjectType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import java.io.IOException;
import java.util.Collection;

/**
 * Schnittstellenklasse, die die logische Verbindung zum Datenverteiler repräsentiert. Über ein Objekt dieser Klasse kann die Verbindung zum
 * Datenverteiler aufgenommen und wieder terminiert, sowie Daten an- und abgemeldet und gelesen und geschrieben werden. Außerdem wird über die
 * logische Verbindung der Zugriff auf spezielle System-Objekte ermöglicht.
 *
 * @author Kappich Systemberatung
 */
public interface ClientDavInterface {

    /**
     * Initialisiert einen Kommunikationskanal zum Datenverteiler mit den in den {@link ClientDavParameters Verbindungsparametern} angegebenen
     * Adressinformationen. Nach dem Aufbau der physischen Verbindung wird die in der Kommunikation mit dem Datenverteiler zu verwendende
     * Protokollversion verhandelt und die logische Verbindung in den Zustand {@code Initialisiert} überführt.
     *
     * @throws ConnectionException Wenn die physische Verbindung zum Datenverteiler nicht hergestellt werden konnte.
     * @throws CommunicationError  Wenn bei der initialen Kommunikation mit dem Datenverteiler Fehler aufgetreten sind.
     * @see ClientDavParameters
     */
    void connect() throws CommunicationError, ConnectionException;

    /**
     * Terminiert die Verbindung mit dem Datenverteiler. Der physische Kommunikationskanal zum Datenverteiler wird geschlossen und die logische
     * Verbindung in den Zustand {@code AußerBetrieb} überführt. Bei einer normalen Terminierung wird sichergestellt, daß alle im Sendepuffer
     * enthaltenen Telegramme vor dem Schließen des Kommunikationskanals gesendet werden. Bei einer abnormalen Terminierung wird der
     * Kommunikationskanal sofort geschlossen.
     *
     * @param error   Information, ob es sich um eine abnormale Terminierung der Verbindung handelt.
     * @param message Fehlertext, der im Falle einer abnormalen Terminierung die Ursache des Terminierung beschreibt, oder {@code null} wenn die
     *                Ursache nicht bekannt ist.
     */
    void disconnect(boolean error, String message);

    ///////////////////////////////////////////////////////////////////////
    ////////////////////Authentifizierung Schnittstelle////////////////////
    ///////////////////////////////////////////////////////////////////////

    /**
     * Start der logischen Verbindung mit dem Datenverteiler. Dabei wird eine Authentifizierung des Benutzers beim Datenverteiler durchgeführt und die
     * logische Verbindung in den Zustand {@code InBetrieb} überführt. Als Benutzername und Passwort werden die entsprechenden Werte aus den {@link
     * ClientDavParameters Verbindungsparametern} benutzt.
     *
     * @throws InconsistentLoginException Wenn Benutzername oder Passwort nicht korrekt sind.
     * @throws CommunicationError         Wenn bei der Kommunikation mit dem Datenverteiler Fehler aufgetreten sind.
     */
    void login() throws InconsistentLoginException, CommunicationError;

    /**
     * Start der logischen Verbindung mit dem Datenverteiler. Dabei wird eine Authentifizierung des Benutzers beim Datenverteiler durchgeführt und die
     * logische Verbindung in den Zustand {@code InBetrieb} überführt.
     *
     * @param userName Name des Benutzers für die Authentifizierung.
     * @param password Passwort des Benutzers für die Authentifizierung.
     *
     * @throws InconsistentLoginException Wenn Benutzername oder Passwort nicht korrekt sind.
     * @throws CommunicationError         Wenn bei der Kommunikation mit dem Datenverteiler Fehler aufgetreten sind.
     */
    void login(String userName, String password) throws InconsistentLoginException, CommunicationError;

    /**
     * Start der logischen Verbindung mit dem Datenverteiler. Dabei wird eine Authentifizierung des Benutzers beim Datenverteiler durchgeführt und die
     * logische Verbindung in den Zustand {@code InBetrieb} überführt.
     *
     * @param userName Name des Benutzers für die Authentifizierung.
     * @param password Passwort des Benutzers für die Authentifizierung.
     *
     * @throws InconsistentLoginException Wenn Benutzername oder Passwort nicht korrekt sind.
     * @throws CommunicationError         Wenn bei der Kommunikation mit dem Datenverteiler Fehler aufgetreten sind.
     */
    default void login(String userName, final char[] password) throws InconsistentLoginException, CommunicationError {
        login(userName, new String(password));
    }

    /**
     * Start der logischen Verbindung mit dem Datenverteiler. Dabei wird eine Authentifizierung des Benutzers beim Datenverteiler durchgeführt und die
     * logische Verbindung in den Zustand {@code InBetrieb} überführt.
     *
     * @param userName          Name des Benutzers für die Authentifizierung.
     * @param clientCredentials Passwort oder Login-Token
     *
     * @throws InconsistentLoginException Wenn Benutzername oder Passwort nicht korrekt sind.
     * @throws CommunicationError         Wenn bei der Kommunikation mit dem Datenverteiler Fehler aufgetreten sind.
     */
    default void login(String userName, final ClientCredentials clientCredentials) throws InconsistentLoginException, CommunicationError {
        login(userName, new String(clientCredentials.getPassword()));
    }

    /**
     * Anmeldung zum Empfangen von Daten. Mit der Anmeldung wird von der Applikation ein Objekt bereitgestellt, daß bei nachfolgenden Aktualisierungen
     * der Daten entsprechend benachrichtigt wird.
     *
     * @param receiver        Ein von der Applikation bereitzustellendes Objekt, das bei Aktualisierungen entsprechende Methodenaufrufe erhält.
     * @param objects         Feld mit System-Objekten für die die spezifizierten Daten anzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den anzumeldenden Daten.
     * @param options         Für die Anmeldung zu verwendende Optionen.
     * @param role            Für die Anmeldung zu verwendende Rolle (Empfänger oder Senke).
     * @param cacheTime       Vorhaltezeitraum in Millisekunden. Der Vorhaltezeitraum spezifiziert, wie lange empfangene Daten zwischengespeichert
     *                        werden sollen. Der Wert {@code -1} setzt den Vorhaltezeitraum auf den Default-Wert.
     */
    void subscribeReceiver(ClientReceiverInterface receiver, SystemObject[] objects, DataDescription dataDescription, ReceiveOptions options,
                           ReceiverRole role, long cacheTime);

    /**
     * Anmeldung zum Empfangen von Daten. Mit der Anmeldung wird von der Applikation ein Objekt bereitgestellt, daß bei nachfolgenden Aktualisierungen
     * der Daten entsprechend benachrichtigt wird.
     *
     * @param receiver        Ein von der Applikation bereitzustellendes Objekt, das bei Aktualisierungen entsprechende Methodenaufrufe erhält.
     * @param objects         Feld mit System-Objekten für die die spezifizierten Daten anzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den anzumeldenden Daten.
     * @param options         Für die Anmeldung zu verwendende Optionen.
     * @param role            Für die Anmeldung zu verwendende Rolle (Empfänger oder Senke).
     */
    void subscribeReceiver(ClientReceiverInterface receiver, SystemObject[] objects, DataDescription dataDescription, ReceiveOptions options,
                           ReceiverRole role);

    /**
     * Anmeldung zum Empfangen von Daten. Mit der Anmeldung wird von der Applikation ein Objekt bereitgestellt, daß bei nachfolgenden Aktualisierungen
     * der Daten entsprechend benachrichtigt wird.
     *
     * @param receiver        Ein von der Applikation bereitzustellendes Objekt, das bei Aktualisierungen entsprechende Methodenaufrufe erhält.
     * @param object          System-Objekt für das die spezifizierten Daten anzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den anzumeldenden Daten.
     * @param options         Für die Anmeldung zu verwendende Optionen.
     * @param role            Für die Anmeldung zu verwendende Rolle (Empfänger oder Senke).
     * @param cacheTime       Vorhaltezeitraum in Millisekunden. Der Vorhaltezeitraum spezifiziert, wie lange empfangene Daten zwischengespeichert
     *                        werden sollen. Der Wert {@code -1} setzt den Vorhaltezeitraum auf den Default-Wert.
     */
    void subscribeReceiver(ClientReceiverInterface receiver, SystemObject object, DataDescription dataDescription, ReceiveOptions options,
                           ReceiverRole role, long cacheTime);

    /**
     * Anmeldung zum Empfangen von Daten. Mit der Anmeldung wird von der Applikation ein Objekt bereitgestellt, daß bei nachfolgenden Aktualisierungen
     * der Daten entsprechend benachrichtigt wird.
     *
     * @param receiver        Ein von der Applikation bereitzustellendes Objekt, das bei Aktualisierungen entsprechende Methodenaufrufe erhält.
     * @param object          System-Objekt für das die spezifizierten Daten anzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den anzumeldenden Daten.
     * @param options         Für die Anmeldung zu verwendende Optionen.
     * @param role            Für die Anmeldung zu verwendende Rolle (Empfänger oder Senke).
     */
    void subscribeReceiver(ClientReceiverInterface receiver, SystemObject object, DataDescription dataDescription, ReceiveOptions options,
                           ReceiverRole role);

    /**
     * Anmeldung zum Empfangen von Daten. Mit der Anmeldung wird von der Applikation ein Objekt bereitgestellt, daß bei nachfolgenden Aktualisierungen
     * der Daten entsprechend benachrichtigt wird.
     *
     * @param receiver        Ein von der Applikation bereitzustellendes Objekt, das bei Aktualisierungen entsprechende Methodenaufrufe erhält.
     * @param objects         Liste mit System-Objekten für die die spezifizierten Daten anzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den anzumeldenden Daten.
     * @param options         Für die Anmeldung zu verwendende Optionen.
     * @param role            Für die Anmeldung zu verwendende Rolle (Empfänger oder Senke).
     * @param cacheTime       Vorhaltezeitraum in Millisekunden. Der Vorhaltezeitraum spezifiziert, wie lange empfangene Daten zwischengespeichert
     *                        werden sollen. Der Wert {@code -1} setzt den Vorhaltezeitraum auf den Default-Wert.
     *
     * @since 3.13 Seit 3.13 akzeptiert diese Methode Collections mit Subtypen von SystemObject
     */
    void subscribeReceiver(ClientReceiverInterface receiver, Collection<? extends SystemObject> objects, DataDescription dataDescription,
                           ReceiveOptions options, ReceiverRole role, long cacheTime);

    /**
     * Anmeldung zum Empfangen von Daten. Mit der Anmeldung wird von der Applikation ein Objekt bereitgestellt, daß bei nachfolgenden Aktualisierungen
     * der Daten entsprechend benachrichtigt wird.
     *
     * @param receiver        Ein von der Applikation bereitzustellendes Objekt, das bei Aktualisierungen entsprechende Methodenaufrufe erhält.
     * @param objects         Liste mit System-Objekten für die die spezifizierten Daten anzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den anzumeldenden Daten.
     * @param options         Für die Anmeldung zu verwendende Optionen.
     * @param role            Für die Anmeldung zu verwendende Rolle (Empfänger oder Senke).
     *
     * @since 3.13 Seit 3.13 akzeptiert diese Methode Collections mit Subtypen von SystemObject
     */
    default void subscribeReceiver(ClientReceiverInterface receiver, Collection<? extends SystemObject> objects, DataDescription dataDescription,
                                   ReceiveOptions options, ReceiverRole role) {
        for (SystemObject object : objects) {
            subscribeReceiver(receiver, object, dataDescription, options, role);
        }
    }

    /**
     * Anmeldung zum Senden von Daten. Für jedes mit {@code objects} angegebene Objekt wird ein Sendeanmeldung für die mit {@code dataDescription}
     * spezifizierten Daten beim Datenverteiler durchgeführt. Bei Anmeldungen als Quelle (siehe Parameter {@code role}) wird nach der Anmeldung
     * automatisch ein leerer Datensatz versendet.
     *
     * @param sender          Anwendungs-Objekt, an das Sendesteuerungen gesendet werden.
     * @param objects         Liste mit System-Objekten für die die spezifizierten Daten anzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den anzumeldenden Daten.
     * @param role            Für die Anmeldung zu verwendende Rolle (Quelle oder Sender).
     *
     * @throws OneSubscriptionPerSendData Wenn bereits eine lokale Sendeanmeldung für die gleichen Daten von einem anderen Anwendungsobjekt vorliegt.
     * @since 3.13 Seit 3.13 akzeptiert diese Methode Collections mit Subtypen von SystemObject
     */
    default void subscribeSender(ClientSenderInterface sender, Collection<? extends SystemObject> objects, DataDescription dataDescription,
                                 SenderRole role) throws OneSubscriptionPerSendData {
        for (SystemObject object : objects) {
            subscribeSender(sender, object, dataDescription, role);
        }
    }

    /**
     * Anmeldung zum Senden von Daten. Für jedes mit {@code objects} angegebene Objekt wird ein Sendeanmeldung für die mit {@code dataDescription}
     * spezifizierten Daten beim Datenverteiler durchgeführt. Bei Anmeldungen als Quelle (siehe Parameter {@code role}) wird nach der Anmeldung
     * automatisch ein leerer Datensatz versendet.
     *
     * @param sender          Anwendungs-Objekt, an das Sendesteuerungen gesendet werden.
     * @param objects         Feld mit System-Objekten, für die die spezifizierten Daten anzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den anzumeldenden Daten.
     * @param role            Für die Anmeldung zu verwendende Rolle (Quelle oder Sender).
     *
     * @throws OneSubscriptionPerSendData Wenn bereits eine lokale Sendeanmeldung für die gleichen Daten von einem anderen Anwendungsobjekt vorliegt.
     */
    void subscribeSender(ClientSenderInterface sender, SystemObject[] objects, DataDescription dataDescription, SenderRole role)
        throws OneSubscriptionPerSendData;

    /**
     * Anmeldung zum Senden von Daten. Für das angegebene Objekt wird ein Sendeanmeldung für die mit {@code dataDescription} spezifizierten Daten beim
     * Datenverteiler durchgeführt. Bei Anmeldungen als Quelle (siehe Parameter {@code role}) wird nach der Anmeldung automatisch ein leerer Datensatz
     * versendet.
     *
     * @param sender          Anwendungs-Objekt, an das Sendesteuerungen gesendet werden.
     * @param object          System-Objekt, für das die spezifizierten Daten anzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den anzumeldenden Daten.
     * @param role            Für die Anmeldung zu verwendende Rolle (Quelle oder Sender).
     *
     * @throws OneSubscriptionPerSendData Wenn bereits eine lokale Sendeanmeldung für die gleichen Daten von einem anderen Anwendungsobjekt vorliegt.
     */
    void subscribeSender(ClientSenderInterface sender, SystemObject object, DataDescription dataDescription, SenderRole role)
        throws OneSubscriptionPerSendData;

    /**
     * Anmeldung als Quelle und versenden von initialen Daten. Für die in {@code initialData} enthaltenen Daten wird ein entsprechende Sendeanmeldung
     * beim Datenverteiler durchgeführt und anschließend wird der übergebene Datensatz als initialer Datensatz versendet.
     *
     * @param sender      Anwendungs-Objekt, an das Sendesteuerungen gesendet werden.
     * @param initialData Initialer Datensatz, der nach der entsprechenden Anmeldung zu versenden ist.
     *
     * @throws OneSubscriptionPerSendData Wenn bereits eine lokale Sendeanmeldung für die gleichen Daten von einem anderen Anwendungsobjekt vorliegt.
     */
    void subscribeSource(ClientSenderInterface sender, ResultData initialData) throws OneSubscriptionPerSendData;

    /**
     * Abmeldung von angemeldeten Daten. Die Methode macht eine mit der Methode {@link #subscribeReceiver} durchgeführte Empfangsanmeldung wieder
     * rückgängig.
     *
     * @param receiver        Das Anwendungsobjekt, das bei der Anmeldung benutzt wurde.
     * @param objects         Feld mit System-Objekten für die die spezifizierten Daten abzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den abzumeldenden Daten.
     */
    void unsubscribeReceiver(ClientReceiverInterface receiver, SystemObject[] objects, DataDescription dataDescription);

    /**
     * Abmeldung von angemeldeten Daten. Die Methode macht eine mit der Methode {@link #subscribeReceiver} durchgeführte Empfangsanmeldung wieder
     * rückgängig.
     *
     * @param receiver        Das Anwendungsobjekt, das bei der Anmeldung benutzt wurde.
     * @param objects         Liste mit System-Objekten für die die spezifizierten Daten abzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den abzumeldenden Daten.
     *
     * @since 3.13 Seit 3.13 akzeptiert diese Methode Collections mit Subtypen von SystemObject
     */
    default void unsubscribeReceiver(ClientReceiverInterface receiver, Collection<? extends SystemObject> objects, DataDescription dataDescription) {
        for (SystemObject object : objects) {
            unsubscribeReceiver(receiver, object, dataDescription);
        }
    }

    /**
     * Abmeldung von angemeldeten Daten. Die Methode macht eine mit der Methode {@link #subscribeReceiver} durchgeführte Empfangsanmeldung wieder
     * rückgängig.
     *
     * @param receiver        Das Anwendungsobjekt, das bei der Anmeldung benutzt wurde.
     * @param object          System-Objekt für das die spezifizierten Daten abzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den abzumeldenden Daten.
     */
    void unsubscribeReceiver(ClientReceiverInterface receiver, SystemObject object, DataDescription dataDescription);

    /**
     * Abmeldung von angemeldeten Daten. Die Methode macht eine mit der Methode {@link #subscribeSender} durchgeführte Sendeanmeldung wieder
     * rückgängig.
     *
     * @param sender          Das Anwendungsobjekt, das bei der Anmeldung benutzt wurde.
     * @param objects         Liste mit System-Objekten für die die spezifizierten Daten abzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den abzumeldenden Daten.
     *
     * @since 3.13 Seit 3.13 akzeptiert diese Methode Collections mit Subtypen von SystemObject
     */
    default void unsubscribeSender(ClientSenderInterface sender, Collection<? extends SystemObject> objects, DataDescription dataDescription) {
        for (SystemObject object : objects) {
            unsubscribeSender(sender, object, dataDescription);
        }
    }

    /**
     * Abmeldung von angemeldeten Daten. Die Methode macht eine mit der Methode {@link #subscribeSender} durchgeführte Sendeanmeldung wieder
     * rückgängig.
     *
     * @param sender          Das Anwendungsobjekt, das bei der Anmeldung benutzt wurde.
     * @param objects         Feld mit System-Objekten für die die spezifizierten Daten abzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den abzumeldenden Daten.
     */
    void unsubscribeSender(ClientSenderInterface sender, SystemObject[] objects, DataDescription dataDescription);

    /**
     * Abmeldung von angemeldeten Daten. Die Methode macht eine mit der Methode {@link #subscribeSender} durchgeführte Sendeanmeldung wieder
     * rückgängig.
     *
     * @param sender          Das Anwendungsobjekt, das bei der Anmeldung benutzt wurde.
     * @param object          System-Objekt für das die spezifizierten Daten abzumelden sind.
     * @param dataDescription Beschreibende Informationen zu den abzumeldenden Daten.
     */
    void unsubscribeSender(ClientSenderInterface sender, SystemObject object, DataDescription dataDescription);

    //////////////////////////////////////////////////////////////////
    ////////////////////Konfiguration Schnittstelle////////////////////
    //////////////////////////////////////////////////////////////////

    /**
     * Gibt das Datenmodell zurück, über das auf die lokale Konfiguration zugegriffen werden kann.
     *
     * @return Datenmodell zum Zugriff auf die Konfiguration
     */
    DataModel getDataModel();

    /**
     * Gibt ein Datenmodell zurück, über das auf eine beliebige Konfiguration zugegriffen werden kann.
     *
     * @param configAuthority Systemobjekt zum Konfigurationsverantwortlichen der Konfiguration
     *
     * @return Datenmodell zum Zugriff auf die Konfiguration.
     *
     * @throws de.bsvrz.dav.daf.main.config.ConfigurationTaskException Falls die angegebene Konfiguration innerhalb von 10 Minuten nicht geantwortet
     *                                                                 hat oder ein Fehler in der Kommunikation auftrat
     */
    DataModel getDataModel(SystemObject configAuthority) throws ConfigurationTaskException;

    /**
     * Gibt ein Datenmodell zurück, über das auf eine beliebige Konfiguration zugegriffen werden kann.
     *
     * @param configAuthority Pid des Konfigurationsverantwortlichen der Konfiguration
     *
     * @return Datenmodell zum Zugriff auf die Konfiguration.
     *
     * @throws de.bsvrz.dav.daf.main.config.ConfigurationTaskException Falls die angegebene Konfiguration innerhalb von 10 Minuten nicht geantwortet
     *                                                                 hat oder ein Fehler in der Kommunikation auftrat
     */
    DataModel getDataModel(String configAuthority) throws ConfigurationTaskException;

    /**
     * Gibt ein Datenmodell zurück, über das auf eine beliebige Konfiguration zugegriffen werden kann. Diese Funktion kann verwendet werden, wenn das
     * Systemobjekt des Konfigurationsverantwortlichen der lokalen Konfiguration nicht bekannt ist.
     *
     * @param configAuthority Id des Konfigurationsverantwortlichen der Konfiguration
     *
     * @return Datenmodell zum Zugriff auf die Konfiguration.
     *
     * @throws de.bsvrz.dav.daf.main.config.ConfigurationTaskException Falls die angegebene Konfiguration innerhalb von 10 Minuten nicht geantwortet
     *                                                                 hat oder ein Fehler in der Kommunikation auftrat
     */
    DataModel getDataModel(long configAuthority) throws ConfigurationTaskException;

    //////////////////////////////////////////////////////////////////
    ////////////////////Angemeldete Applikationen Schnittstelle////////////////////
    //////////////////////////////////////////////////////////////////

    /**
     * Gibt das Stellvertreterobjekt für den Datenverteiler, mit dem die Applikation verbunden ist, zurück.
     *
     * @return Stellvertreterobjekt für den verbundenen Datenverteiler.
     */
    DavApplication getLocalDav();

    /**
     * Gibt das Stellvertreterobjekt für diese Applikation zurück. Die Erzeugung des entsprechenden Objekts in der Konfiguration wird vom
     * Datenverteiler nach dem Verbindungsaufbau und der erfolgreichen Authentifizierung veranlasst.
     *
     * @return Stellvertreterobjekt für die lokale Applikation.
     */
    ClientApplication getLocalApplicationObject();

    /**
     * Bestimmt das Systemobjekt für den angemeldeten Benutzer.
     *
     * @return Stellvertreterobjekt für den angemeldeten Benutzer.
     */
    DynamicObject getLocalUser();

    /**
     * Gibt das Stellvertreterobjekt des zugeordneten Konfigurationsverantwortlichen zurück.
     *
     * @return Konfigurationsverantwortlicher.
     */
    ConfigurationAuthority getLocalConfigurationAuthority();

    ///////////////////////////////////////////////////////////////////////
    ////////////////////Telegrammlaufzeit Schnittstelle////////////////////
    ///////////////////////////////////////////////////////////////////////

    /**
     * Bestimmt die Telegrammlaufzeit von dieser Applikation zum lokalen Datenverteiler und zurück.
     *
     * @return Telegrammlaufzeit in Millisekunden oder {@code -1}, wenn innerhalb 60 Sekunden keine Antwort empfangen wurde.
     */
    long getDavRoundTripTime();

    /**
     * Zugriff auf zwischengespeicherte Datensätze. Über die optionale Zeitangabe kann auch auf historische Datensätze im Zwischenspeicher zugegriffen
     * werden.
     *
     * @param objects         Feld mit System-Objekten für die die spezifizierten Daten zu lesen sind.
     * @param dataDescription Beschreibende Informationen zu den gewünschten Daten.
     * @param options         Für den Zugriff zu verwendenden Optionen
     * @param history         Spezifikation der gewünschten Historie oder {@code null} wenn nur die aktuellen Daten gewünscht sind.
     *
     * @return Zeitlich sortiertes Feld mit den gewünschten Daten.
     */
    ResultData[] getCachedData(SystemObject[] objects, DataDescription dataDescription, ReceiveOptions options, HistorySpecification history);

    /**
     * Zugriff auf den aktuellen Datensatz eines System-Objekts. Wenn der spezifizierte Datensatz zwischengespeichert ist, dann wird er sofort zur
     * Verfügung gestellt. Wenn der Datensatz nicht angemeldet ist, dann wird er implizit angemeldet. Falls noch kein Ergebnis vorliegt, wartet die
     * Methode darauf. Eine implizite Anmeldung wird wieder rückgängig gemacht, wenn über eine angegebene Zeitspanne kein Zugriff mehr auf das Datum
     * erfolgte.
     *
     * @param object             System-Objekt für das die spezifizierten Daten zu lesen sind.
     * @param dataDescription    Beschreibende Informationen zu dem zu lesenden Datensatz.
     * @param unsubscriptionTime Relative Zeitangabe in Millisekunden nach der eine implizite Anmeldung wieder abgemeldet werden kann.
     *
     * @return Der aktuelle Datensatz.
     */
    ResultData getData(SystemObject object, DataDescription dataDescription, long unsubscriptionTime);

    /**
     * Zugriff auf die aktuellen Datensätze von verschiedenen System-Objekten. Wenn einzelne der spezifizierten Datensätze nicht angemeldet sind, dann
     * werden sie implizit angemeldet. Falls zu einzelnen spezifizierten Datensätzen noch kein Ergebnis vorliegt, wird darauf gewartet. Eine implizite
     * Anmeldung wird wieder rückgängig gemacht, wenn über eine angegebene Zeitspanne kein Zugriff mehr auf das Datum erfolgte.
     *
     * @param objects            Feld mit System-Objekten für die die spezifizierten Daten zu lesen sind.
     * @param dataDescription    Beschreibende Informationen der zu lesenden Daten.
     * @param unsubscriptionTime Relative Zeitangabe in Millisekunden nach der eine implizite Anmeldung wieder abgemeldet werden kann.
     *
     * @return Feld mit den aktuellen Datensätzen.
     */
    ResultData[] getData(SystemObject[] objects, DataDescription dataDescription, long unsubscriptionTime);

    /**
     * Erzeugt ein neues {@link Data}-Objekt zu der angegebenen Attributgruppe. Mit diesem Data-Objekt können Daten zu der Attributgruppe hinterlegt
     * werden. Das erzeugte Data-Objekt kann dann mit Werten befüllt werden (z. B. {@code data.getTextValue("Attribut").setText("Test");}) und dann
     * später entweder {@linkplain SystemObject#setConfigurationData(AttributeGroup, Data) konfigurierend gespeichert werden} oder {@linkplain
     * #sendData(ResultData) als Online-Datensatz versendet werden}. Für letzteres muss das Data-Objekt noch in ein {@link ResultData} verpackt
     * werden, um Meta-Informationen wie den Datenzeitstempel zu ergänzen.
     * <p>
     * Diese Methode kann durch {@link de.bsvrz.dav.daf.main.config.AttributeGroup#createData} ersetzt werden, falls keine Kompatibilität mit alten
     * Kernsoftware-Versionen erforderlich ist.
     *
     * @param attributeGroup Attributgruppe des neuen Datensatzes.
     *
     * @return Neuen initialisierten Datensatz.
     */
    default Data createData(AttributeGroup attributeGroup) {
        return attributeGroup.createData();
    }

    /**
     * Sendet einen Ergebnisdatensatz zum Datenverteiler. Die Daten müssen vorher zum Senden angemeldet worden sein. Falls beim ersten Sendeversuch
     * keine positive/negative Sendesteuerung vorhanden ist, wird ein fest vorgegebener Zeitraum abgewartet in dem eine positive/negative
     * Sendesteuerung vorliegen muss. Wenn nach Ablauf des Zeitraums keine positive/negative Sendesteuerung vorliegt, wird eine {@code
     * SendSubscriptionNotConfirmed} Exception geworfen, falls die positive Sendesteuerung vor dem Ablauf des Zeitraums vorliegt, werden die Daten
     * verschickt, liegt eine negative Sendesteuerung vor, wird eine {@code SendSubscriptionNotConfirmed} Exception geworfen. <br> Falls die
     * Sendesteuerung nach dem ersten erfolgreichen Sendeversuch wieder negativ wird, und es sollen erneut Daten verschickt werden, wird eine {@code
     * SendSubscriptionNotConfirmed} Exception geworfen ohne die fest vorgegebene Zeitspanne abzuwarten.
     *
     * @param result Ergebnis mit dem zu sendenden Datensatz.
     *
     * @throws DataNotSubscribedException   Wenn die Daten nicht zum Senden angemeldet waren.
     * @throws SendSubscriptionNotConfirmed Wenn beim Senden als einfacher Sender gesendet wird, ohne die Sendesteuerung abzuwarten.
     */
    void sendData(ResultData result) throws DataNotSubscribedException, SendSubscriptionNotConfirmed;

    /**
     * Sendet mehrere Ergebnisdatensätze zum Datenverteiler. Die Daten müssen vorher zum Senden angemeldet worden sein. Falls beim ersten Sendeversuch
     * keine positive/negative Sendesteuerung vorhanden ist, wird ein fest vorgegebener Zeitraum abgewartet in dem eine positive/negative
     * Sendesteuerung vorliegen muss. Wenn nach Ablauf des Zeitraums keine positive/negative Sendesteuerung vorliegt, wird eine {@code
     * SendSubscriptionNotConfirmed} Exception geworfen, falls die positive Sendesteuerung vor dem Ablauf des Zeitraums vorliegt, werden die Daten
     * verschickt, liegt eine negative Sendesteuerung vor, wird eine {@code SendSubscriptionNotConfirmed} Exception geworfen. <br> Falls die
     * Sendesteuerung nach dem ersten erfolgreichen Sendeversuch wieder negativ wird, und es sollen erneut Daten verschickt werden, wird eine {@code
     * SendSubscriptionNotConfirmed} Exception geworfen ohne die fest vorgegebene Zeitspanne abzuwarten.
     *
     * @param results Die zu sendenden Ergebnisdatensätze.
     *
     * @throws DataNotSubscribedException   Wenn nicht alle Datensätze zum Senden angemeldet waren.
     * @throws SendSubscriptionNotConfirmed Wenn beim Senden als einfacher Sender gesendet wird, ohne die Sendesteuerung abzuwarten.
     */
    void sendData(ResultData[] results) throws DataNotSubscribedException, SendSubscriptionNotConfirmed;

    /**
     * Setzt das Objekt, das für die Behandlung von Fehlern der Kommunikationsverbindung zuständig ist. Die Applikation kann mit dem übergebenen
     * Objekt selbst steuern, wie Verbindungsfehler dargestellt wird, und ob sich die Applikation beenden oder eine neue Verbindung aufbauen soll.
     * Wenn diese Methode nicht aufgerufen wird, wird als Default ein Objekt der Klasse {@link SystemTerminator} benutzt, das bei
     * Kommunikationsfehlern eine Fehlermeldung ausgibt und die Applikation terminiert.
     *
     * @param closer Objekt für die Behandlung von Kommunikationsfehlern oder null, falls bei einem Verbindungsfehler nichts passieren soll.
     */
    void setCloseHandler(ApplicationCloseActionHandler closer);

    /**
     * Bestimmt die aktuelle Zeit oder, wenn die Applikation mit der Simulationsvariante einer Offline-Simulation gestartet wurde, die simulierte
     * Zeit.
     *
     * @return Zeitpunkt in Millisekunden seit 1970.
     *
     * @throws IllegalStateException Wenn die simulierte Zeit im Fall einer Simulation nicht bestimmt werden kann.
     */
    long getTime();

    /**
     * Blockiert den aufrufenden Thread für die spezifizierte Zeit. Die angegebene Dauer der Pause wird in Realzeit oder, wenn die Applikation mit der
     * Simulationsvariante einer Offline-Simulation gestartet wurde, im Zeitfluss der Simulation berücksichtigt.
     *
     * @param timeToSleep Wartezeit in Millisekunden seit 1970.
     *
     * @throws IllegalStateException Wenn der simulierte Zeitfluss im Fall einer Simulation nicht bestimmt werden kann.
     */
    void sleep(long timeToSleep);

    /**
     * Blockiert den aufrufenden Thread bis die spezifizierte Zeit erreicht ist. Der angegebene Zeitpunkt wird in Realzeit oder, wenn die Applikation
     * mit der Simulationsvariante einer Offline-Simulation gestartet wurde, im Zeitfluss der Simulation berücksichtigt.
     *
     * @param absoluteTime Abzuwartender Zeitpunkt in Millisekunden seit 1970.
     *
     * @throws IllegalStateException Wenn der simulierte Zeitfluss im Fall einer Simulation nicht bestimmt werden kann.
     */
    void sleepUntil(long absoluteTime);

    /**
     * Liefert ein Stellvertreterobjekt zurück, mit dem streambasierte Archivanfragen über das lokale Archivsystem abgewickelt werden können. Dieser
     * Aufruf ist äquivalent zu dem Aufruf {@code getArchive(getLocalConfigurationAuthority)}.
     *
     * @return Stellvertreterobjekt zur Abwicklung von Archivanfragen über das lokale Archivsystem.
     *
     * @see #getLocalConfigurationAuthority()
     * @see #getArchive(SystemObject)
     */
    ArchiveRequestManager getArchive();

    /**
     * Liefert ein Stellvertreterobjekt zurück, mit dem streambasierte Archivanfragen über ein bestimmtes Archivsystem abgewickelt werden können. Es
     * wird eine Verbindung zu demjenigen Archivsystem hergestellt, dass über das angegebene System-Objekt identifiziert wird.
     *
     * @param archiveSystem Archivsystem über das Archivanfragen abgewickelt werden sollen.
     *
     * @return Stellvertreterobjekt zur Abwicklung von Archivanfragen über das spezifizierte Archivsystem.
     */
    ArchiveRequestManager getArchive(SystemObject archiveSystem);

    /**
     * Liefert den aktuellen Zustand der Verschlüsselung zurück.
     *
     * @return Zustand der Verschlüsselung
     */
    default EncryptionStatus getEncryptionStatus() {
        return EncryptionStatus.notEncrypted();
    }

    /**
     * Liefert den aktuellen Zustand der Authentifizierung zurück.
     *
     * @return Zustand der Authentifizierung
     */
    default AuthenticationStatus getAuthenticationStatus() {
        return AuthenticationStatus.notAuthenticated();
    }

    /**
     * Gibt den Standard-Konfigurationsbereich für den angegebenen dynamischen Typen zurück. Die Methode greift auf den Parameterdatensatz
     * `atg.verwaltungDynamischerObjekte` zu. Sollte dieser nicht vorhanden sein oder für den angegebenen Typen keine Zuordnung definiert sein, wird
     * der Standardbereich des Konfigurationsverantwortlichen verwendet.
     * <p>
     * Diese Methode bietet die gleiche Kern-Funktionalität wie die de.bsvrz.sys.funclib.dynobj, aber keine erweiterten Möglichkeiten zum
     * Anlegen/Löschen von Objekten.
     *
     * @param dynamicObjectType Typ
     *
     * @return Default-Bereich in dem Objekte dieses Typs angelegt werden (kann in Ausnahmefällen null sein, wenn kein Parameter definiert wurde und
     *     kein Defaultbereich konfiguriert wurde)
     *
     * @see ConfigurationArea#createDynamicObject(DynamicObjectType, String, String, Collection)
     * @since 3.9.2
     */
    ConfigurationArea getDefaultConfigurationArea(DynamicObjectType dynamicObjectType);

    /**
     * Gibt detaillierte Informationen über die bestehenden Datenanmeldungen an einer Datenidentifikation zurück.
     * <p>
     * Diese Abfrage greift auf interne Datenstrukturen im Datenverteiler zurück und ist ausschließlich für die Problemdiagnose und Überwachung
     * gedacht. Sie sollte nicht in der normalen Programmlogik eingesetzt werden.
     * <p>
     * Insbesondere darf diese Abfrage nicht dazu verwendet werden, festzustellen, ob schon ein Programm Daten zu einer Attributgruppenverwendung
     * liefert. Hier wäre es weiterhin besser einen eigenen Sender oder Empfänger anzumelden und auf die Sendesteuerung bzw. empfangenen Datensätze zu
     * reagieren.
     *
     * @param davApplication    Datenverteiler, der gefragt werden soll
     * @param object            Systemobjekt, das die Anmeldung betrifft
     * @param usage             Attributgruppenverwendung, für das die Anmeldungen am angegebenen Objekt ermittelt werden sollen
     * @param simulationVariant Simulationsvariante der zu ermittelnden Anmeldungen
     *
     * @return Klasse mit Informationen über die angemeldeten Applikationen auf dieses Datum
     *
     * @throws IOException Falls der Datenverteiler die Anfrage nicht erlaubt oder nicht unterstützt
     * @since 3.9.2
     */
    ClientSubscriptionInfo getSubscriptionInfo(DavApplication davApplication, SystemObject object, AttributeGroupUsage usage, short simulationVariant)
        throws IOException;

    /**
     * Gibt Informationen über alle Datenanmeldungen einer Applikation zurück.
     * <p>
     * Diese Abfrage greift auf interne Datenstrukturen im Datenverteiler zurück und ist ausschließlich für die Problemdiagnose und Überwachung
     * gedacht. Sie sollte nicht in der normalen Programmlogik eingesetzt werden.
     * <p>
     * Insbesondere darf diese Abfrage nicht dazu verwendet werden, festzustellen, ob schon ein Programm Daten zu einer Attributgruppenverwendung
     * liefert. Hier wäre es weiterhin besser einen eigenen Sender oder Empfänger anzumelden und auf die Sendesteuerung bzw. empfangenen Datensätze zu
     * reagieren.
     * <p>
     * Einige Applikationen melden sehr viele Daten an. Dadurch kann diese Anfrage entsprechend lange dauern.
     *
     * @param davApplication Datenverteiler, der gefragt werden soll
     * @param application    Applikation von der vorhandene Anmeldungen abgefragt werden sollen
     *
     * @return Klasse mit Informationen über die angemeldeten Daten der Applikation
     *
     * @throws IOException Falls der Datenverteiler die Anfrage nicht erlaubt oder nicht unterstützt
     * @since 3.9.2
     */
    ApplicationSubscriptionInfo getSubscriptionInfo(DavApplication davApplication, ClientApplication application) throws IOException;

    /**
     * Bestimmt die Verbindungsparameter der Datenverteiler-Applikationsfunktionen. Es wird eine schreibgeschützte Kopie zurückgegeben.
     *
     * @return Verbindungsparameter der Datenverteiler-Applikationsfunktionen
     *
     * @see de.bsvrz.dav.daf.main.ClientDavParameters#clone(boolean)
     */
    ClientDavParameters getClientDavParameters();

    /**
     * Ergänzt einen neuen Beobachter für bestimmte Zustandsänderungen bzgl. der Verbindung zum Datenverteiler.
     *
     * @param davConnectionListener Beobachterobjekt, das hinzugefügt werden soll.
     */
    void addConnectionListener(DavConnectionListener davConnectionListener);

    /**
     * Entfernt einen Beobachter für bestimmte Zustandsänderungen bzgl. der Verbindung zum Datenverteiler.
     *
     * @param davConnectionListener Beobachterobjekt, das entfernt werden soll.
     */
    void removeConnectionListener(DavConnectionListener davConnectionListener);

    /**
     * Methode, die von der Applikation aufgerufen werden kann, um explizit eine Applikations-Fertigmeldung zu versenden. Die SW-Einheit Start/Stopp
     * verzögert den Start von Applikationen, die von einer anderen Applikation abhängig sind, bis diese Applikation die Fertigmeldung versendet. Die
     * Fertigmeldung wird automatisch nach dem Verbindungsaufbau versendet, wenn die Applikation nicht vor der Authentifizierung der Verbindung die
     * Methode {@link #enableExplicitApplicationReadyMessage()} aufgerufen hat.
     */
    void sendApplicationReadyMessage();

    /**
     * Diese Methode sollte von Applikationen, die den Zeitpunkt zum Versand der Applikations-Fertigmeldung selbst vorgeben möchten, vor der
     * Authentifizierung einer Datenverteilerverbindung aufrufen. Der Aufruf dieser Methode bewirkt, dass nach dem Verbindungsaufbau automatisch eine
     * Noch-Nicht-Fertigmeldung am Applikationsobjekt versendet wird. Die Applikation sollte dann nach erfolgreicher Initialisierung den Versand der
     * Fertig-Meldung mit der Methode {@link #sendApplicationReadyMessage()} veranlassen.
     */
    void enableExplicitApplicationReadyMessage();

    /**
     * Vergleicht den angegebenen Benutzernamen und das angegebene Benutzerpasswort mit den entsprechenden Werten, die beim Verbindungsaufbau zum
     * Datenverteiler benutzt wurden.
     *
     * @param userName Zu prüfender Benutzername
     * @param password Zu prüfendes Passwort
     *
     * @return {@code true}, wenn eine Verbindung zum Datenverteiler besteht und mit dem angegebenen Benutzernamen und Passwort aufgebaut wurde, sonst
     *     {@code false}
     */
    boolean checkLoggedUserNameAndPassword(String userName, String password);

    /**
     * Gibt eine Klasse zurück, die für diese Verbindung Transaktions-Anmeldungen durchführt.
     *
     * @return Klasse die Funktionen zu Transaktionen bietet
     */
    Transactions getTransactions();

    /**
     * Gibt {@code true} zurück, wenn die Verbindung aufgebaut wurde.
     *
     * @return {@code true}, wenn die Verbindung aufgebaut wurde, sonst {@code false}
     *
     * @since 3.9.2
     */
    boolean isConnected();

    /**
     * Gibt {@code true} zurück, wenn der Benutzer erfolgreich eingeloggt ist.
     *
     * @return {@code true}, wenn der Benutzer erfolgreich eingeloggt ist, sonst {@code false}
     *
     * @since 3.9.2
     */
    boolean isLoggedIn();

    /**
     * Gibt ein Objekt zurück, mit dem die Rechte des eigenen oder eines anderen Benutzers abgefragt werden können.
     * <p>Diese Rechte werden clientseitig anhand der vorhandenen Daten ermittelt und können ggf. vom Laufzeitverhalten abweichen, insbesondere wenn
     * die Rechte von mehreren Datenverteilern mit unterschiedlicher Parametrierung geprüft werden oder ein Datensatz unter Verwendung von mehreren
     * Benutzern übertragen wird.</p>
     * <p>Eine Rückgabe von {@code true} von
     * {@link de.bsvrz.dav.daf.accessControl.UserInfo#maySubscribeData(de.bsvrz.dav.daf.communication.lowLevel.telegrams.BaseSubscriptionInfo,
     * de.bsvrz.dav.daf.accessControl.internal.UserAction) UserInfo.maySubscribeData} heißt also nicht, dass der Datensatz in jedem Fall ankommt,
     * sondern nur, dass der lokale Datenverteiler diesen anhand der aktuellen Parametrierung (voraussichtlich) nicht filtern wird.</p>
     * <p>Es werden nur die Rechte beim lokalen Datenverteiler geprüft!</p>
     * <p>Das zurückgegebene Objekt kann verwendet werden, um beispielsweise Buttons oder Menüs bei fehlenden Rechten zu deaktivieren oder
     * auszublenden.</p>
     * <p>Außerdem kann das Objekt dazu verwendet werden, statt dem Datenverteiler innerhalb einer Applikation beliebige Rechte lokal zu prüfen. So
     * kann bspw. die PuA-Applikation, die Aufträge eines Benutzers annimmt und dann in dessen Namen Daten abfragt, prüfen, ob der Benutzer berechtigt
     * ist, die Daten überhaupt abzufragen. Der Datenverteiler kann diese Prüfung nicht vornehmen, da aus seiner Sicht alle Daten von der
     * PuA-Applikation (und dem Benutzer unter dem diese gestartet wurde) selbst abgefragt werden.</p>
     * <p><b>Hinweis:</b> Ist die verwendete Rechteprüfung des Datenverteilers {@linkplain #getAccessControlMode() nicht abfragbar}, weil dieser eine
     * zu alte Version verwendet oder das Datenmodell veraltet ist, wird angenommen, dass die {@linkplain AccessControlMode#OldDataModel alte
     * Rechteprüfung} verwendet wird. Im Zweifelsfall werden dadurch Operationen als verboten aufgeführt, die eigentlich erlaubt sind. Das Problem
     * kann durch ein Update des Datenmodells und der Kernsoftware am Datenverteiler gelöst werden.</p>
     *
     * @return AccessControlManager
     *
     * @since 3.13
     */
    AccessControlManager getAccessControlManager();

    /**
     * Gibt die vom lokalen Datenverteiler verwendete Art der Rechteprüfung zurück.
     *
     * @return Art der Rechteprüfung oder {@code null}, falls nicht ermittelbar
     *
     * @since 3.13
     */
    AccessControlMode getAccessControlMode();

    /**
     * Gibt die vom angegebenen Datenverteiler verwendete Art der Rechteprüfung zurück.
     *
     * @param davApplication Datenverteiler-Objekt
     *
     * @return Art der Rechteprüfung oder {@code null}, falls nicht ermittelbar
     *
     * @since 3.13
     */
    AccessControlMode getAccessControlMode(final DavApplication davApplication);
}
