/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.main;

/**
 * Hilfsmethoden zu Data-Objekten
 *
 * @author Kappich Systemberatung
 * @since 3.12
 */
public final class DataUtil {

    /**
     * Kopiert die Inhalte eines Data-Objekts in ein anderes Data-Objekt (sofern möglich).
     * <p>
     * Die Struktur der Data-Objekte sollte möglichst kompatibel sein. Ist das nicht der Fall gelten folgende Regeln:
     * </p>
     * <ul>
     * <li>
     * Attribute, die in beiden Objekten gleich sind (oder kompatibel sind), werden beibehalten.
     * </li>
     * <li>
     * Attribute, die in der Zielversion weggefallen sind, werden gelöscht.
     * </li>
     * <li>
     * Attribute, die in der Zielversion hinzugefügt wurden, werden auf den Default-Wert gesetzt (falls vorhanden).
     * </li>
     * <li>
     * Attribute, die inkompatibel geändert wurden, werden auf undefiniert gelassen.
     * </li>
     * </ul>
     *
     * @param from Quell-Datum
     * @param to   Ziel-Datum
     */
    public static void copy(final Data from, final Data to) {
        // Diese Methode ruft sich selbst rekursiv auf.
        try {
            if (from.isList() && to.isList()) {
                copyList(from, to);
            } else if (from.isArray() && to.isArray()) {
                copyArray(from, to);
            } else if (from.isPlain() && to.isPlain()) {
                copyPlain(from, to);
            }
        } catch (Exception ignored) {
            // Irgendwas war nicht kompatibel, z. B. Arraylänge. Daher dieses Datum nicht kopieren.
        }
        // sonst auf undefiniert lassen, da inkompatibel
    }

    private static void copyPlain(final Data from, final Data to) {
        to.asTextValue().setText(from.asTextValue().getText());
    }

    private static void copyArray(final Data from, final Data to) {
        Data.Array toArray = to.asArray();
        Data.Array fromArray = from.asArray();
        toArray.setLength(fromArray.getLength());
        for (int i = 0; i < toArray.getLength(); i++) {
            copy(fromArray.getItem(i), toArray.getItem(i));
        }
    }

    private static void copyList(final Data from, final Data to) {
        for (Data toItem : to) {
            String name = toItem.getName();
            if (from.containsItem(name)) {
                copy(from.getItem(name), toItem);
            } else {
                // Neue Attribute auf Default setzen.
                toItem.setToDefault();
            }
        }
    }

    /**
     * TextArray-Ansicht auf ein beliebiges Data.Array (z. B. von unskalierten Werten)
     *
     * @author Kappich Systemberatung
     */
    public static class StandardTextArrayAdapter implements Data.TextArray {
        private final Data.Array _array;

        /**
         * Erstellt einen neuen StandardTextArrayAdapter
         *
         * @param data Adaptiertes Array
         */
        public StandardTextArrayAdapter(final Data.Array data) {
            _array = data;
        }

        @Override
        public int getLength() {
            return _array.getLength();
        }

        @Override
        public void setLength(final int newLength) {
            _array.setLength(newLength);
        }

        @Override
        public Data.TextValue getTextValue(final int itemIndex) {
            return _array.getItem(itemIndex).asTextValue();
        }

        @Override
        public Data.TextValue[] getTextValues() {
            Data.TextValue[] result = new Data.TextValue[getLength()];
            for (int i = 0; i < result.length; i++) {
                result[i] = getTextValue(i);
            }
            return result;
        }

        @Override
        public void set(final String... strings) {
            setLength(strings.length);
            for (int i = 0; i < strings.length; i++) {
                getTextValue(i).setText(strings[i]);
            }
        }

        @Override
        public String getText(final int itemIndex) {
            return getTextValue(itemIndex).getText();
        }

        @Override
        public String[] getTextArray() {
            String[] result = new String[getLength()];
            for (int i = 0; i < result.length; i++) {
                result[i] = getText(i);
            }
            return result;
        }

        @Override
        public String toString() {
            return _array.toString();
        }
    }
}
