/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.main.archive;

/**
 * Klasse die die vier verschiedenen Datensatzarten des Archivsystem repräsentieren kann (siehe Technische Anforderungen Archivsystem).
 *
 * @author Kappich Systemberatung
 */
public final class ArchiveDataKind {

    /**
     * Datensatzart für online erhaltene aktuelle Datensätze (siehe Technische Anforderungen Archivsystem).
     */
    public static final ArchiveDataKind ONLINE = new ArchiveDataKind("aktuelle Daten", 1, false, false);

    /**
     * Datensatzart für online erhaltene als nachgelieferte markierte Datensätze (siehe Technische Anforderungen Archivsystem).
     */
    public static final ArchiveDataKind ONLINE_DELAYED = new ArchiveDataKind("nachgelieferte Daten", 2, false, true);

    /**
     * Datensatzart für nachgeforderte aktuelle Datensätze (siehe Technische Anforderungen Archivsystem).
     */
    public static final ArchiveDataKind REQUESTED = new ArchiveDataKind("nachgefordert-aktuelle Daten", 3, true, false);

    /**
     * Datensatzart für nachgeforderte als nachgeliefert markierte Datensätze (siehe Technische Anforderungen Archivsystem).
     */
    public static final ArchiveDataKind REQUESTED_DELAYED = new ArchiveDataKind("nachgefordert-nachgelieferte Daten", 4, true, true);

    private final String _name;
    private final int _code;
    private final boolean _requested;
    private final boolean _delayed;

    private ArchiveDataKind(String name, int code, boolean requested, boolean delayed) {
        _name = name;
        _code = code;
        _requested = requested;
        _delayed = delayed;
    }

    /**
     * Liefert eine zur angegebenen Kodierung korrespondierende Datensatzart zurück. Die Kodierung einer Datensatzart kann mit der Methode {@link
     * #getCode()} bestimmt werden.
     *
     * @param code Kodierung der gewünschten Datensatzart.
     *
     * @return Zur angegebenen Kodierung korrespondierende Datensatzart.
     *
     * @throws IllegalArgumentException Wenn eine ungültige Kodierung übergeben wurde.
     * @see #getCode
     */
    public static ArchiveDataKind getInstance(int code) {
	    return switch (code) {
		    case 1 -> ONLINE;
		    case 2 -> ONLINE_DELAYED;
		    case 3 -> REQUESTED;
		    case 4 -> REQUESTED_DELAYED;
		    default -> throw new IllegalArgumentException("Undefinierte Objektkodierung");
	    };
    }

    /**
     * Bestimmt die Kodierung dieser Datensatzart. Die Kodierung ist innerhalb der Klasse eindeutig. Das entsprechende Objekt kann aus der Kodierung
     * mit Hilfe der Methode {@link #getInstance(int)} wiederhergestellt werden.
     *
     * @return Eindeutige Kodierung dieser Datensatzart.
     *
     * @see #getInstance(int)
     */
    public int getCode() {
        return _code;
    }

    public int hashCode() {
        return _code;
    }

    @Override
    public boolean equals(final Object obj) {
        return this == obj;
    }

    /**
     * Bestimmt, ob diese Datensatzart nachgeforderte Datensätze enthalten kann.
     *
     * @return {@code true} für die Datensatzarten {@link #REQUESTED} und {@link #REQUESTED_DELAYED}, sonst {@code false}.
     */
    public boolean isRequested() {
        return _requested;
    }

    /**
     * Bestimmt, ob diese Datensatzart als nachgeliefert gekennzeichnete Datensätze enthalten kann.
     *
     * @return {@code true} für die Datensatzarten {@link #ONLINE_DELAYED} und {@link #REQUESTED_DELAYED}, sonst {@code false}.
     */
    public boolean isDelayed() {
        return _delayed;
    }

    /**
     * Liefert eine textuelle Beschreibung dieser Datensatzart zurück. Das genaue Format ist nicht festgelegt und kann sich ändern.
     *
     * @return Beschreibung dieses Objekts.
     */
    public String toString() {
        return _name;
    }

}
