/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.main.config;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

/**
 * Schnittstellenklasse zum Zugriff auf die Eigenschaften eines Applikations-Objekts, das nach erfolgreicher Authentifizierung einer Applikation vom
 * Datenverteiler erzeugt wird. Der Name des Objekts und der konkrete Typ wird von der Applikation beim Aufbau der logischen Verbindung zum
 * Datenverteiler festgelegt.
 *
 * @author Kappich+Kniß Systemberatung Aachen (K2S)
 * @author Roland Schmitz (rs)
 * @author Stephan Homeyer (sth)
 * @version $Revision$ / $Date$ / ($Author$)
 * @see de.bsvrz.dav.daf.main.ClientDavInterface#login
 */
public interface ClientApplication extends DynamicObject {

    /**
     * Ermittelt, welcher Datenverteiler das Applikationsobjekt angelegt hat. Diese Methode benötigt kb.systemModellGlobal in Mindestversion 40.
     *
     * @return Datenverteiler-Objekt oder null falls nicht ermittelbar
     *
     * @since 3.13
     */
    @Nullable
    default DavApplication getDavApplication() {
        AttributeGroup attributeGroup = getDataModel().getAttributeGroup("atg.applikationsEigenschaften");
        if (attributeGroup == null) {
            return null;
        }
        Data data = getConfigurationData(attributeGroup);
        if (data == null) {
            return null;
        }

        // Nicht getSystemObject() um Exception zu vermeiden falls nicht auflösbar
        return getDataModel().getObject(data.getReferenceValue("Datenverteiler").getId(), DavApplication.class);
    }

    /**
     * Ermittelt, unter welchem Benutzer die Applikation eingeloggt ist. Diese Methode benötigt kb.systemModellGlobal in Mindestversion 40. Ist der
     * Benutzer nicht dem lokalen Konfigurationsverantwortlichen zugeordnet, liefert die Methode {@code null}, außer {@code allowForeign} (optionaler
     * Parameter) ist {@code true}.
     *
     * @return Benutzer-Objekt oder null falls nicht ermittelbar
     *
     * @since 3.13
     */
    @Nullable
    default DynamicObject getUser() {
        return getUser(false);
    }

    /**
     * Ermittelt, unter welchem Benutzer die Applikation eingeloggt ist. Diese Methode benötigt kb.systemModellGlobal in Mindestversion 40. Ist der
     * Benutzer nicht dem lokalen Konfigurationsverantwortlichen zugeordnet, liefert die Methode {@code null}, außer {@code allowForeign} (optionaler
     * Parameter) ist {@code true}.
     *
     * @param allowForeign Benutzer von fremden Konfigurationsverantwortlichen zurückgeben? Dies funktioniert nicht immer zuverlässig, kann
     *                     performancemäßig länger dauern und kann auch aus Sicherheitsgründen (z. B. falls der Benutzer für eine Rechteprüfung
     *                     genutzt werden soll) problematisch sein. Aus diesem Grund ist das Verhalten standardmäßig deaktiviert.
     *
     * @return Benutzer-Objekt oder null falls nicht ermittelbar
     *
     * @since 3.13
     */
    @Nullable
    default DynamicObject getUser(boolean allowForeign) {

        if (!allowForeign && !getConfigurationArea().getConfigurationAuthority().equals(getDataModel().getConfigurationAuthority())) {
            // Falscher KV
            return null;
        }

        AttributeGroup attributeGroup = getDataModel().getAttributeGroup("atg.applikationsEigenschaften");
        if (attributeGroup == null) {
            return null;
        }
        Data data = getConfigurationData(attributeGroup);
        if (data == null) {
            return null;
        }

        // Nicht getSystemObject() um Exception zu vermeiden falls nicht auflösbar
        return getDataModel().getObject(data.getReferenceValue("Benutzer").getId(), DynamicObject.class);
    }
}
