/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2008-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.main.config;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

/**
 * Schnittstelle zum Zugriff auf die Eigenschaften von online änderbaren Mengen. Eine Applikation kann sich auf Änderungen einer dynamischen Menge
 * {@link #addChangeListener(MutableSetChangeListener) anmelden} und auch wieder {@link #removeChangeListener(MutableSetChangeListener) abmelden}.
 *
 * @author Kappich Systemberatung
 */
public interface MutableSet extends ObjectSet, ConfigurationCommunicationInterface, MutableCollection {

    /**
     * Meldet einen Beobachter für die Änderungen dieser Menge an. Bei Änderungen der Menge wird die Methode {@link MutableSetChangeListener#update}
     * des angegebenen Beobachters aufgerufen.
     *
     * @param listener Beobachter für Änderungen der Menge.
     */
    public void addChangeListener(MutableSetChangeListener listener);

    /**
     * Meldet einen Beobachter für die Änderungen dieser Menge wieder ab.
     *
     * @param listener Ein bisher für Änderungen der Menge angemeldeter Beobachter.
     */
    public void removeChangeListener(MutableSetChangeListener listener);

    /**
     * Gibt den Konfigurationsverantwortlichen zurück, der die Menge verwaltet. Dieser kann sich bei dynamischen Mengen vom KV unterscheiden, dem der
     * Konfigurationsbereich gehört.
     *
     * @return Konfigurationsverantwortlicher, oder null falls unbekannt
     *
     * @since 3.13
     */
    @Nullable
    default ConfigurationAuthority getConfigurationAuthority() {
        ConfigurationAuthority result = getConfigurationArea().getConfigurationAuthority();

        final DataModel configuration = getDataModel();
        final AttributeGroup atg = configuration.getAttributeGroup("atg.dynamischeMenge");
        if (atg != null) {
            final Data data = this.getConfigurationData(atg);
            if (data != null) {
                final String elementsManagementPid = data.getTextValue("verwaltung").getValueText();
                if (!elementsManagementPid.isEmpty()) {
                    SystemObject obj = configuration.getObject(elementsManagementPid);
                    if (obj instanceof ConfigurationAuthority) {
                        result = (ConfigurationAuthority) obj;
                    } else {
                        result = null;
                    }
                }
            }
        }

        return result;
    }
}
