/*
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.main.config;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

/**
 * Schnittstelle, zum Aufsuchen von Objekten.
 *
 * @author Roland Schmitz (rs), Kappich Systemberatung
 * @version $Revision: none $, $Date: 04.12.2006 $, $Author: rs $
 */
public interface ObjectLookup {
    /**
     * Liefert das System-Objekt mit der angegebenen PID zurück.
     *
     * @param pid Die permanente ID des System-Objekts
     *
     * @return Das gewünschte System-Objekt oder {@code null}, wenn es kein Objekt mit der angegebenen PID gibt.
     *
     * @see DataModel
     */
    @Nullable
    public SystemObject getObject(String pid);

    /**
     * Liefert das System-Objekt mit der angegebenen Objekt-ID zurück.
     *
     * @param id Die Objekt-ID des System-Objekts
     *
     * @return Das gewünschte System-Objekt oder {@code null}, wenn es kein Objekt mit der angegebenen ID gibt.
     *
     * @see DataModel
     */
    @Nullable
    public SystemObject getObject(long id);

    /**
     * Liefert das System-Objekt mit der angegebenen PID zurück, falls es vom angegebenen Typ ist.
     * <p>
     * <b>Diese Methode ist neu in Kernsoftware 3.14 und sollte nur benutzt werden,
     * wenn Kompatibilität mit einer alten Kernsoftware nicht erforderlich ist.</b>
     *
     * @param pid  Die permanente ID des System-Objekts
     * @param type Klasse des gewünschten Rückgabetyps
     * @param <T>  Rückgabe-Typ
     *
     * @return Das gewünschte System-Objekt oder {@code null}, wenn es kein Objekt mit der angegebenen PID gibt.
     *
     * @throws IllegalArgumentException Wenn der Typ des erhaltenen Objektes nicht mit den geforderten Typ {@code T} übereinstimmt.
     * @see DataModel
     * @since 3.14
     */
    @Nullable
    default <T extends SystemObject> T getObject(String pid, Class<T> type) {
        SystemObject object = getObject(pid);
        // Die JVM optimiert diesen Aufruf, das ist nur unwesentlich langsamer als ein `instanceof` mit der konkreten Klasse.
        if (object == null || type.isInstance(object)) {
            return (T) object;
        }
        throw new IllegalArgumentException(
            "Das Objekt mit der PID \"" + pid + "\" ist nicht vom geforderten Typ \"" + type.getSimpleName() + "\". Erhaltener Typ: \"" +
            object.getClass().getSimpleName() + "\" (" + object.getType().getPid() + ")");
    }

    /**
     * Liefert das System-Objekt mit der angegebenen Objekt-ID zurück, falls es vom angegebenen Typ ist.
     * <p>
     * <b>Diese Methode ist neu in Kernsoftware 3.14 und sollte nur benutzt werden,
     * wenn Kompatibilität mit einer alten Kernsoftware nicht erforderlich ist.</b>
     *
     * @param id   Die Objekt-ID des System-Objekts
     * @param type Klasse des gewünschten Rückgabetyps
     * @param <T>  Rückgabe-Typ
     *
     * @return Das gewünschte System-Objekt oder {@code null}, wenn es kein Objekt mit der angegebenen ID gibt.
     *
     * @throws IllegalArgumentException Wenn der Typ des erhaltenen Objektes nicht mit den geforderten Typ {@code T} übereinstimmt.
     * @see DataModel
     * @since 3.14
     */
    @Nullable
    default <T extends SystemObject> T getObject(long id, Class<T> type) {
        SystemObject object = getObject(id);
        // Die JVM optimiert diesen Aufruf, das ist nur unwesentlich langsamer als ein `instanceof` mit der konkreten Klasse.
        if (object == null || type.isInstance(object)) {
            return (T) object;
        }
        throw new IllegalArgumentException(
            "Das Objekt mit der ID \"" + id + "\" ist nicht vom geforderten Typ \"" + type.getSimpleName() + "\". Erhaltener Typ: \"" +
            object.getClass().getSimpleName() + "\" (" + object.getType().getPid() + ")");
    }
}
